import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useState } from 'react';
import Swal from 'sweetalert2';

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Label } from '@/components/ui/label';

import { ArrowLeft, Save, Trash2 } from 'lucide-react';

type Illness = {
    id: number;
    illness_name: string;
};

type PageProps = {
    illness: Illness;
};

export default function MedicalsIllnessesEdit() {
    const { illness } = usePage<PageProps>().props;
    const [illnessName, setIllnessName] = useState(illness.illness_name);
    const [errors, setErrors] = useState<{ illness_name?: string }>({});

    const onSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        setErrors({});

        router.patch(
            route('medicals.illnesses.update', illness.id),
            { illness_name: illnessName },
            {
                preserveScroll: true,
                onSuccess: () => {
                    Swal.fire({ icon: 'success', title: 'Illness updated', timer: 1400, showConfirmButton: false });
                    router.visit(route('medicals.illnesses.index'));
                },
                onError: (err: any) => setErrors(err || {}),
            },
        );
    };

    const onDelete = async () => {
        const { isConfirmed } = await Swal.fire({
            title: `Delete "${illness.illness_name}"?`,
            text: 'This will permanently remove the illness.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, delete',
            cancelButtonText: 'Cancel',
            confirmButtonColor: '#dc2626',
            cancelButtonColor: '#6b7280',
            reverseButtons: true,
        });
        if (!isConfirmed) return;

        router.delete(route('medicals.illnesses.destroy', illness.id), {
            onSuccess: () => {
                Swal.fire({ icon: 'success', title: 'Illness deleted', timer: 1400, showConfirmButton: false });
                router.visit(route('medicals.illnesses.index'));
            },
        });
    };

    return (
        <AppLayout
            breadcrumbs={[
                { title: 'Medicals', href: route('medicals.patients.index') },
                { title: 'Illnesses', href: route('medicals.illnesses.index') },
                { title: 'Edit', href: route('medicals.illnesses.edit', illness.id) },
            ]}
        >
            <Head title={`Edit Illness • ${illness.illness_name}`} />

              <div className="mx-4 my-6 md:mx-8">
                <Card className="">
                    <CardHeader>
                        <CardTitle>Edit Illness</CardTitle>
                        <CardDescription>Update the illness name.</CardDescription>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={onSubmit} className="space-y-6">
                            <div className="grid gap-2">
                                <Label htmlFor="illness_name">Illness Name</Label>
                                <Input
                                    id="illness_name"
                                    value={illnessName}
                                    onChange={(e) => setIllnessName(e.target.value)}
                                    placeholder="e.g., Hypertension"
                                />
                                {errors.illness_name && (
                                    <div className="text-sm text-red-600">{errors.illness_name}</div>
                                )}
                            </div>

                            <div className="flex items-center justify-between">
                                <Button asChild variant="outline">
                                    <Link href={route('medicals.illnesses.index')}>
                                        <ArrowLeft className="mr-2 h-4 w-4" />
                                        Back
                                    </Link>
                                </Button>

                                <div className="flex items-center gap-2">
                                    <Button type="button" variant="destructive" onClick={onDelete}>
                                        <Trash2 className="mr-2 h-4 w-4" />
                                        Delete
                                    </Button>
                                    <Button type="submit">
                                        <Save className="mr-2 h-4 w-4" />
                                        Save
                                    </Button>
                                </div>
                            </div>
                        </form>
                    </CardContent>
                </Card>
            </div>
        </AppLayout>
    );
}
