import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useMemo, useRef, useState } from 'react';
import Swal from 'sweetalert2';

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';

import { Plus, Search as SearchIcon, Pencil, Trash2 } from 'lucide-react';

type Illness = {
  id: number;
  illness_name: string;
  created_at?: string | null;
  updated_at?: string | null;
};

type Pagination<T> = {
  data: T[];
  current_page: number;
  last_page: number;
  per_page: number;
  total: number;
};

type PageProps = {
  illnesses: Pagination<Illness>;
  filters: { search?: string; per_page?: number };
};

function useDebounced<T>(value: T, delay = 450) {
  const [v, setV] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setV(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return v;
}

export default function MedicalsIllnessesIndex() {
  const { illnesses, filters } = usePage<PageProps>().props;

  const [search, setSearch] = useState(filters.search ?? '');
  const [perPage, setPerPage] = useState<number>(filters.per_page ?? 20);

  const debouncedSearch = useDebounced(search, 450);
  const firstRun = useRef(true);

  const applyQuery = (page = 1, overrides?: Partial<Record<string, string | number>>) => {
    router.get(
      route('medicals.illnesses.index'),
      {
        page,
        search: overrides?.search ?? search,
        per_page: overrides?.per_page ?? perPage,
      },
      { preserveState: true, replace: true, preserveScroll: true },
    );
  };

  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    applyQuery(1, { search: debouncedSearch });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  const confirmDelete = async (illness: Illness) => {
    const { isConfirmed } = await Swal.fire({
      title: `Delete "${illness.illness_name}"?`,
      text: 'This will permanently remove the illness. (Patient links will be detached in your backend as needed.)',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
    });
    if (!isConfirmed) return;

    router.delete(route('medicals.illnesses.destroy', illness.id), {
      preserveScroll: true,
      onSuccess: () =>
        Swal.fire({ icon: 'success', title: 'Illness deleted', timer: 1400, showConfirmButton: false }),
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.patients.index') },
        { title: 'Illnesses', href: route('medicals.illnesses.index') },
      ]}
    >
      <Head title="Medicals • Illnesses" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        {/* Header + actions */}
        <div className="flex flex-col items-start justify-between gap-3 sm:flex-row sm:items-center">
          <h1 className="text-2xl font-bold">Illnesses</h1>
          <div className="flex gap-2">
            <Button asChild>
              <Link href={route('medicals.illnesses.create')}>
                <Plus className="mr-2 h-4 w-4" />
                Add Illness
              </Link>
            </Button>
          </div>
        </div>

        {/* Big search bar */}
        <Card>
          <CardContent className="pt-6">
            <div className="relative">
              <SearchIcon className="text-muted-foreground absolute left-4 top-1/2 h-5 w-5 -translate-y-1/2" />
              <Input
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && applyQuery(1)}
                placeholder="Search illnesses…"
                className="h-12 w-full rounded-xl pl-12 pr-[8.5rem] text-base"
              />
              <Button
                type="button"
                className="absolute right-2 top-1/2 -translate-y-1/2 h-10 rounded-lg px-4"
                onClick={() => applyQuery(1)}
                title="Search"
              >
                <SearchIcon className="mr-2 h-4 w-4" />
                Search
              </Button>
            </div>

            <div className="mt-3 flex items-center justify-between">
              <div className="text-xs text-muted-foreground">
                Showing {illnesses.data.length} of {illnesses.total}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">Per page</span>
                <select
                  value={perPage}
                  onChange={(e) => {
                    const n = parseInt(e.target.value, 10);
                    setPerPage(n);
                    applyQuery(1, { per_page: n });
                  }}
                  className="h-9 rounded-md border px-2 text-sm"
                >
                  <option value={10}>10</option>
                  <option value={20}>20</option>
                  <option value={50}>50</option>
                  <option value={100}>100</option>
                </select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Table */}
        <Card>
          <CardHeader>
            <CardTitle>List</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto rounded-xl border">
              <Table className="min-w-[720px]">
                <TableHeader>
                  <TableRow>
                    <TableHead className="uppercase">#</TableHead>
                    <TableHead className="uppercase">Illness Name</TableHead>
                    <TableHead className="uppercase text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {illnesses.data.map((ill, idx) => (
                    <TableRow key={ill.id}>
                      <TableCell>{(illnesses.current_page - 1) * illnesses.per_page + idx + 1}</TableCell>
                      <TableCell className="font-medium">{ill.illness_name}</TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-2">
                          <Button asChild size="icon" variant="secondary" title="Edit">
                            <Link href={route('medicals.illnesses.edit', ill.id)}>
                              <Pencil className="h-4 w-4" />
                            </Link>
                          </Button>
                          <Button size="icon" variant="destructive" onClick={() => confirmDelete(ill)} title="Delete">
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                  {illnesses.data.length === 0 && (
                    <TableRow>
                      <TableCell colSpan={3} className="py-8 text-center text-sm text-muted-foreground">
                        No illnesses found.
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>

            {/* Pagination (simple) */}
            <div className="mt-4 flex items-center justify-center gap-2">
              <Button
                variant="outline"
                disabled={illnesses.current_page <= 1}
                onClick={() => applyQuery(illnesses.current_page - 1)}
              >
                ← Prev
              </Button>
              <div className="text-sm">
                Page {illnesses.current_page} of {illnesses.last_page}
              </div>
              <Button
                variant="outline"
                disabled={illnesses.current_page >= illnesses.last_page}
                onClick={() => applyQuery(illnesses.current_page + 1)}
              >
                Next →
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
