import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useMemo, useRef, useState } from 'react';
import ReactPaginate from 'react-paginate';
import * as XLSX from 'xlsx';
import Swal from 'sweetalert2';
import 'sweetalert2/dist/sweetalert2.min.css';
import moment from 'moment';
import {
  Eye, Pencil, Trash2, FileSpreadsheet, Search as SearchIcon,
  Building2, Stethoscope, SlidersHorizontal, X,
} from 'lucide-react';

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Card, CardContent } from '@/components/ui/card';
import { DropdownMenu, DropdownMenuCheckboxItem, DropdownMenuContent, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { Separator } from '@/components/ui/separator';

// Types
type CompanyOpt = { id: number | string; company_name: string };

type Certificate = {
  id: number; status: string;
  validity?: number | null; released_at?: string | null;
  certificate_number?: string | null; update_reason?: string | null;
  certificate_location?: string | null; created_at?: string | null; updated_at?: string | null;
};

type PhysicalExam = {
  id: number; bmi: string; bmi_status?: string | null;
  bp_dia?: string | null; bp_repeat_dia?: string | null; bp_repeat_sys?: string | null;
  bp_status?: string | null; bp_sys?: string | null; created_at?: string | null;
  first_bp_time?: string | null; height?: string | null; referral_comment?: string | null;
  temp?: string | null; pulse?: string | null; last_bp_time?: string | null;
  left_vision?: string | null; patient_id: number; right_vision?: string | null;
  updated_at?: string | null; weight?: string | null;
};

type PatientRow = {
  id: number; first_name?: string | null; last_name?: string | null;
  company?: string | null; company_email?: string | null; company_phone?: string | null; company_phone2?: string | null;
  age?: number | null; national_id?: string | null; date_of_birth?: string | null; phone_number?: string | null;
  updated_at?: string | null; created_at?: string | null; employee_number?: string | null; last_x_ray?: string | null;
  category?: string | null; certificate_status?: string | null; certificates?: Certificate[]; physical_records?: PhysicalExam[];
  swabs?: { id: number; patient_id: number; status: string; file?: string | null; comment?: string | null; created_at?: string | null; updated_at?: string | null; }[];
};

type PageProps = {
  patients: {
    data: PatientRow[]; current_page: number; last_page: number; per_page: number; total: number;
  };
  filters: {
    search?: string; location?: string; company?: number | string | null; swab_status?: string;
    certificate_status?: string; per_page?: number; sort?: 'cert_created_at'|'created_at'|'last_name'; direction?: 'asc'|'desc';
  };
  companies?: CompanyOpt[];
};

// Helpers
function useDebounced<T>(value: T, delay = 450) {
  const [v, setV] = useState(value);
  useEffect(() => { const id = setTimeout(() => setV(value), delay); return () => clearTimeout(id); }, [value, delay]);
  return v;
}
function locationBadgeVariant(loc?: string | null): 'default' | 'secondary' | 'destructive' | 'outline' {
  const L = (loc || '').toUpperCase();
  if (L === 'HARARE') return 'default';
  if (L === 'MUTARE' || L === 'BULAWAYO') return 'secondary';
  if (L === 'FAILED') return 'destructive';
  return 'outline';
}
function statusBadgeVariant(st?: string | null): 'default' | 'secondary' | 'destructive' | 'outline' {
  const s = (st || '').toUpperCase();
  if (s === 'RELEASED' || s === 'READY' || s === 'VALID') return 'default';
  if (s === 'PENDING' || s === 'MONITORING') return 'secondary';
  if (s === 'FAILED') return 'destructive';
  return 'outline';
}
function latestCertificate(p: PatientRow): Certificate | null {
  if (!p?.certificates?.length) return null;
  const c = [...p.certificates].sort((a, b) => (new Date(b.created_at || '').getTime() || 0) - (new Date(a.created_at || '').getTime() || 0));
  return c[0] ?? null;
}
function flattenForExport(rows: PatientRow[], visibleKeys: string[]) {
  return rows.map((item) => {
    const cert = latestCertificate(item);
    const base: Record<string, any> = {
      ID: item.id,
      'LAST MEDICAL DATE': cert?.created_at ? moment(cert.created_at).format('YYYY-MM-DD HH:mm') : '',
      'FIRST NAME': item.first_name ?? '',
      'LAST NAME': item.last_name ?? '',
      COMPANY: item.company ?? '',
      'NATIONAL ID': item.national_id ?? '',
      LOCATION: cert?.certificate_location ?? '',
      'PHONE NUMBER': item.phone_number ?? '',
      'EMPLOYEE NUMBER': item.employee_number ?? '',
      'TYPE OF MEDICAL': item.category ?? '',
      'CERTIFICATE STATUS': item.certificate_status ?? cert?.status ?? '',
    };
    const filtered: Record<string, any> = {};
    visibleKeys.forEach((key) => { if (key in base) filtered[key] = base[key]; });
    return filtered;
  });
}
function exportToExcel(data: any[], filename: string) {
  if (!data.length) return;
  const worksheet = XLSX.utils.json_to_sheet(data);
  const cols: { wch: number }[] = [];
  Object.keys(data[0]).forEach(() => cols.push({ wch: 20 }));
  (worksheet as any)['!cols'] = cols;
  const workbook = XLSX.utils.book_new();
  XLSX.utils.book_append_sheet(workbook, worksheet, 'Patients');
  XLSX.writeFile(workbook, filename);
}

type ColumnDef = {
  key: string; label: string;
  render: (p: PatientRow, rowIndex: number, pageOffset: number) => React.ReactNode;
  always?: boolean;
};

const COLUMN_DEFS: ColumnDef[] = [
  { key: 'ID', label: 'ID', render: (_p, idx, offset) => offset + idx + 1 },
  {
    key: 'LAST MEDICAL DATE',
    label: 'Last Medical Date',
    render: (p) => {
      const cert = latestCertificate(p);
      return cert?.created_at ? moment(cert.created_at).format('dddd D MMMM YYYY [at] h:mm A') : '—';
    },
  },
  {
    key: 'FIRST NAME',
    label: 'First Name',
    render: (p) => <Link className="hover:underline" href={route('medicals.patients.show', p.id)}>{p.first_name ?? '—'}</Link>,
  },
  {
    key: 'LAST NAME',
    label: 'Last Name',
    render: (p) => <Link className="hover:underline" href={route('medicals.patients.show', p.id)}>{p.last_name ?? '—'}</Link>,
  },
  { key: 'COMPANY', label: 'Company', render: (p) => p.company ?? '—' },
  { key: 'NATIONAL ID', label: 'National ID', render: (p) => p.national_id ?? '—' },
  {
    key: 'LOCATION', label: 'Location',
    render: (p) => {
      const loc = latestCertificate(p)?.certificate_location ?? '—';
      return loc === '—' ? '—' : <Badge variant={locationBadgeVariant(loc)}>{loc}</Badge>;
    },
  },
  { key: 'PHONE NUMBER', label: 'Phone Number', render: (p) => p.phone_number ?? '—' },
  { key: 'EMPLOYEE NUMBER', label: 'Employee No.', render: (p) => p.employee_number ?? '—' },
  { key: 'TYPE OF MEDICAL', label: 'Type Of Medical', render: (p) => p.category ?? '—' },
  {
    key: 'CERTIFICATE STATUS', label: 'Certificate Status',
    render: (p) => {
      const status = p.certificate_status ?? latestCertificate(p)?.status ?? '';
      return <Badge variant={statusBadgeVariant(status)}>{status || '—'}</Badge>;
    },
  },
  {
    key: 'RENEW',
    label: 'New Consultation?',
    render: (p) => (
      <Button onClick={() => startConsultationGlobal(p.id)} className="rounded-full" size="sm" title="Start new consultation">
        <Stethoscope className="mr-2 h-4 w-4" /> Renew Medical
      </Button>
    ),
  },
  {
    key: 'ACTIONS', label: 'Actions', always: true,
    render: (p) => (
      <div className="flex justify-end gap-2">
        <Button asChild size="icon" variant="outline" title="View">
          <Link href={route('medicals.patients.show', p.id)}><Eye className="h-4 w-4" /></Link>
        </Button>
        <Button asChild size="icon" variant="secondary" title="Edit">
          <Link href={route('medicals.attendees.edit', p.id)}><Pencil className="h-4 w-4" /></Link>
        </Button>
        <Button size="icon" variant="destructive" onClick={() => confirmDeleteGlobal(p.id)} title="Delete">
          <Trash2 className="h-4 w-4" />
        </Button>
      </div>
    ),
  },
];

// allow renderers to call handlers
let confirmDeleteGlobal: (id: number) => void = () => {};
let startConsultationGlobal: (id: number) => void = () => {};

export default function PneumoIndex() {
  const { patients, filters, companies = [] } = usePage<PageProps>().props;
  const baseRoute = route('medicals.pneumo'); // <— THIS PAGE’S ROUTE
  const PAGE_TITLE = 'Pneumoconiosis Patients';

  // filters
  const [search, setSearch] = useState(filters.search ?? '');
  const [location, setLocation] = useState(filters.location ?? '');
  const [company, setCompany] = useState(String(filters.company ?? ''));
  const [swabStatus, setSwabStatus] = useState(filters.swab_status ?? '');
  const [certificateStatus, setCertificateStatus] = useState(filters.certificate_status ?? '');
  const [perPage, setPerPage] = useState<number>(filters.per_page ?? 50);
  const [sort, setSort] = useState<'cert_created_at' | 'created_at' | 'last_name'>((filters.sort as any) ?? 'cert_created_at');
  const [direction, setDirection] = useState<'asc' | 'desc'>((filters.direction as any) ?? 'desc');

  const debouncedSearch = useDebounced(search, 450);
  const firstRun = useRef(true);

  const toSelectValue = (v: string) => (v && v.trim() !== '' ? v : 'ALL');
  const fromSelectValue = (v: string) => (v === 'ALL' ? '' : v);

  const STORAGE_KEY = 'medicals_patients_visible_columns_v1_pneumo';
  const DEFAULT_VISIBLE_KEYS = COLUMN_DEFS.map((c) => c.key);
  const [visibleColumns, setVisibleColumns] = useState<string[]>(() => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        const parsed = JSON.parse(saved);
        if (Array.isArray(parsed) && parsed.length) return parsed;
      }
    } catch {}
    return DEFAULT_VISIBLE_KEYS;
  });
  useEffect(() => { try { localStorage.setItem(STORAGE_KEY, JSON.stringify(visibleColumns)); } catch {} }, [visibleColumns]);

  const toggleColumn = (key: string, next: boolean) => {
    setVisibleColumns((prev) => (next ? (prev.includes(key) ? prev : [...prev, key]) : prev.filter((k) => k !== key)));
  };

  const visibleDefs = useMemo(() => COLUMN_DEFS.filter((c) => c.always || visibleColumns.includes(c.key)), [visibleColumns]);

  const applyQuery = (page = 1, overrides?: Partial<Record<string, string | number>>) => {
    router.get(
      baseRoute,
      {
        page,
        search: overrides?.search ?? search,
        location: overrides?.location ?? location,
        company: overrides?.company ?? company,
        swab_status: overrides?.swab_status ?? swabStatus,
        certificate_status: overrides?.certificate_status ?? certificateStatus,
        per_page: overrides?.per_page ?? perPage,
        sort: overrides?.sort ?? sort,
        direction: overrides?.direction ?? direction,
      },
      { preserveState: true, replace: true, preserveScroll: true },
    );
  };

  useEffect(() => {
    if (firstRun.current) { firstRun.current = false; return; }
    applyQuery(1, { search: debouncedSearch });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  const handlePageChange = (selected: { selected: number }) => { applyQuery(selected.selected + 1); };

  const handleReset = () => {
    setSearch(''); setLocation(''); setCompany(''); setSwabStatus(''); setCertificateStatus('');
    setPerPage(50); setSort('cert_created_at'); setDirection('desc');
    router.get(baseRoute, { page: 1 }, { preserveState: true, replace: true, preserveScroll: true });
  };

  const handleExport = () => {
    const exportKeys = visibleDefs.map((c) => c.label).filter((label) => label !== 'Actions');
    const flat = flattenForExport(patients?.data ?? [], exportKeys);
    exportToExcel(flat, 'pneumoconiosis-patients.xlsx');
  };

  const confirmDelete = async (id: number) => {
    const res = await Swal.fire({
      title: 'Delete patient?', text: 'This will remove the patient and linked records.', icon: 'warning',
      showCancelButton: true, confirmButtonText: 'Yes, delete', cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626', cancelButtonColor: '#6b7280', reverseButtons: true, focusCancel: true,
    });
    if (res.isConfirmed) {
      router.delete(route('medicals.patients.destroy', id), { preserveScroll: true });
    }
  };
  confirmDeleteGlobal = confirmDelete;

  const startConsultation = async (patientId: number) => {
    const res = await Swal.fire({
      title: 'Start New Examination',
      html: `
        <div style="text-align:left;">
          <div style="font-size:.9rem; color:#6b7280; margin-bottom:.75rem;">Choose the medical category for this examination.</div>
          <div style="display:flex; flex-direction:column; gap:.5rem;">
            <label for="medical-category-select" style="font-weight:600; font-size:.9rem;">Medical Category</label>
            <select id="medical-category-select" class="swal2-input" style="margin:0; height:44px; border-radius:.5rem; padding:.5rem .75rem; border:1px solid #e5e7eb;">
              <option value="" disabled selected>Select a category…</option>
              <option value="Food Handler (COH)">Food Handler</option>
              <option value="Pneumoconiosis">Pneumoconiosis</option>
              <option value="Pre-Employement">Pre-Employment</option>
              <option value="Exit-Pneumoconiosis">Exit (Pneumoconiosis)</option>
              <option value="Exit-Employement">Exit-Employement</option>
            </select>
            <div style="font-size:.8rem; color:#6b7280;">You can change category later if needed.</div>
          </div>
        </div>
      `,
      width: 680, icon: 'question', focusConfirm: false, showCancelButton: true,
      confirmButtonText: 'Start Examination', cancelButtonText: 'Cancel',
      confirmButtonColor: '#2563eb', cancelButtonColor: '#6b7280', buttonsStyling: false,
      customClass: {
        popup: 'rounded-xl', title: 'text-left',
        confirmButton: 'swal2-confirm rounded-lg px-4 py-2 font-medium bg-blue-600 text-white hover:bg-blue-700',
        cancelButton: 'swal2-cancel rounded-lg px-4 py-2 font-medium bg-gray-200 text-gray-800 hover:bg-gray-300',
      },
      preConfirm: () => {
        const el = document.getElementById('medical-category-select') as HTMLSelectElement | null;
        const selectedValue = el?.value?.trim() || '';
        if (!selectedValue) { Swal.showValidationMessage('Please select a medical category.'); return false; }
        return { selectedValue };
      },
    });
    if (!res.isConfirmed) return;
    const { selectedValue } = (res.value as any) || {};
    Swal.fire({ icon: 'success', title: 'Consultation Started', text: `The consultation has been initiated (${selectedValue}).`, confirmButtonColor: '#2563eb' });
  };
  startConsultationGlobal = startConsultation;

  const pageOffset = (patients.current_page - 1) * patients.per_page;

  const handleEnterSearch = (e: React.KeyboardEvent<HTMLInputElement>) => { if (e.key === 'Enter') applyQuery(1); };
  const clearSearch = () => setSearch('');

  const safeCompanies = (companies || []).filter((c) => {
    if (!c) return false;
    const id = (c.id as any); if (id === null || id === undefined) return false;
    const s = String(id); return s.trim().length > 0;
  });

  return (
    <AppLayout breadcrumbs={[
      { title: 'Medicals', href: `${API}/medicals` },
      { title: 'Pneumoconiosis', href: baseRoute },
    ]}>
      <Head title={`Medicals • ${PAGE_TITLE}`} />

      <div className="bg-background mx-4 my-6 rounded-xl p-4 shadow-sm md:mx-8 md:p-6">
        {/* Top: BIG search, columns, export */}
        <div className="mb-4 space-y-3">
          <div className="relative rounded-xl border bg-white/70 p-2 shadow-sm ring-1 ring-border dark:bg-neutral-900/60">
            <div className="relative">
              <SearchIcon className="text-muted-foreground absolute left-4 top-1/2 h-5 w-5 -translate-y-1/2" />
              <Input
                value={search} onChange={(e) => setSearch(e.target.value)} onKeyDown={handleEnterSearch}
                placeholder="Search patients by first/last name or ID number…" className="h-12 w-full rounded-xl pl-12 pr-[12.5rem] text-base"
              />
              {search && (
                <Button type="button" variant="ghost" className="absolute right-[9.25rem] top-1/2 -translate-y-1/2 h-8 w-8 rounded-full" onClick={clearSearch} title="Clear">
                  <X className="h-4 w-4" />
                </Button>
              )}
              <div className="absolute right-2 top-1/2 -translate-y-1/2 flex gap-2">
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="outline" className="h-10 rounded-lg px-3" title="Columns">
                      <SlidersHorizontal className="mr-2 h-4 w-4" /> Columns
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end" className="w-56">
                    <div className="px-2 py-1.5 text-xs text-muted-foreground">Toggle columns</div>
                    <Separator />
                    {COLUMN_DEFS.map((col) => {
                      const checked = col.always || visibleColumns.includes(col.key);
                      const disabled = !!col.always;
                      return (
                        <DropdownMenuCheckboxItem
                          key={col.key} checked={checked} disabled={disabled}
                          onCheckedChange={(v) => toggleColumn(col.key, Boolean(v))} className="capitalize"
                        >
                          {col.label}
                        </DropdownMenuCheckboxItem>
                      );
                    })}
                  </DropdownMenuContent>
                </DropdownMenu>
                <Button type="button" className="h-10 rounded-lg px-4" onClick={() => applyQuery(1)} title="Search">
                  <SearchIcon className="mr-2 h-4 w-4" /> Search
                </Button>
                <Button onClick={handleExport} className="h-10 rounded-lg px-3">
                  <FileSpreadsheet className="mr-2 h-4 w-4" /> Export
                </Button>
              </div>
            </div>
            <div className="mt-1 px-1 text-xs text-muted-foreground">
              Tip: Press <kbd className="rounded border px-1">Enter</kbd> to search • Hyphens/spaces in IDs are ignored.
            </div>
          </div>
        </div>

        {/* Filters */}
        <Card className="mb-4">
          <CardContent className="pt-6">
            <div className="grid grid-cols-1 gap-4 md:grid-cols-3 lg:grid-cols-6">
              <div>
                <Label>Location</Label>
                <Select value={toSelectValue(location)} onValueChange={(v) => { const real = fromSelectValue(v); setLocation(real); applyQuery(1, { location: real }); }}>
                  <SelectTrigger><SelectValue placeholder="All Locations" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Locations</SelectItem>
                    <SelectItem value="HARARE">HARARE</SelectItem>
                    <SelectItem value="MUTARE">MUTARE</SelectItem>
                    <SelectItem value="BULAWAYO">BULAWAYO</SelectItem>
                    <SelectItem value="PICKSTONE">PICKSTONE</SelectItem>
                    <SelectItem value="EUREKA">EUREKA</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Company</Label>
                <Select value={toSelectValue(company)} onValueChange={(v) => { const real = fromSelectValue(v); setCompany(real); applyQuery(1, { company: real }); }}>
                  <SelectTrigger><SelectValue placeholder="All Companies" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Companies</SelectItem>
                    { (companies || []).filter(Boolean).map((c) => {
                      const idStr = String((c as any).id);
                      if (!idStr.trim()) return null;
                      return (
                        <SelectItem key={idStr} value={idStr}>
                          <div className="inline-flex items-center gap-2"><Building2 className="h-4 w-4" />{(c as any).company_name}</div>
                        </SelectItem>
                      );
                    })}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Swab Status</Label>
                <Select value={toSelectValue(swabStatus)} onValueChange={(v) => { const real = fromSelectValue(v); setSwabStatus(real); applyQuery(1, { swab_status: real }); }}>
                  <SelectTrigger><SelectValue placeholder="All Swab Status" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All</SelectItem>
                    <SelectItem value="PENDING">PENDING</SelectItem>
                    <SelectItem value="DONE">DONE</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Certificate Status</Label>
                <Select value={toSelectValue(certificateStatus)} onValueChange={(v) => { const real = fromSelectValue(v); setCertificateStatus(real); applyQuery(1, { certificate_status: real }); }}>
                  <SelectTrigger><SelectValue placeholder="All Certificate Status" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All</SelectItem>
                    <SelectItem value="PENDING">PENDING</SelectItem>
                    <SelectItem value="MONITORING">MONITORING</SelectItem>
                    <SelectItem value="READY">READY</SelectItem>
                    <SelectItem value="RELEASED">RELEASED</SelectItem>
                    <SelectItem value="FAILED">FAILED</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Sort by</Label>
                <Select value={sort} onValueChange={(v: 'cert_created_at' | 'created_at' | 'last_name') => { setSort(v); applyQuery(1, { sort: v }); }}>
                  <SelectTrigger><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="cert_created_at">Latest Certificate</SelectItem>
                    <SelectItem value="created_at">Patient Created</SelectItem>
                    <SelectItem value="last_name">Last Name</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Direction</Label>
                <Select value={direction} onValueChange={(v: 'asc' | 'desc') => { setDirection(v); applyQuery(1, { direction: v }); }}>
                  <SelectTrigger><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="asc">Ascending</SelectItem>
                    <SelectItem value="desc">Descending</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="mt-4 flex items-center justify-end">
              <div className="w-48">
                <Label>Items per page</Label>
                <Select value={String(perPage)} onValueChange={(v) => { const n = parseInt(v, 10); setPerPage(n); applyQuery(1, { per_page: n }); }}>
                  <SelectTrigger><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="10">10 per page</SelectItem>
                    <SelectItem value="25">25 per page</SelectItem>
                    <SelectItem value="50">50 per page</SelectItem>
                    <SelectItem value="100">100 per page</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-[980px]">
            <TableHeader>
              <TableRow>{visibleDefs.map((col) => (<TableHead key={col.key} className="uppercase">{col.label}</TableHead>))}</TableRow>
            </TableHeader>
            <TableBody>
              {patients.data.map((p, idx) => {
                const rowHasPhysical = (p.physical_records?.length ?? 0) > 0;
                return (
                  <TableRow key={p.id} className={rowHasPhysical ? 'hover:bg-primary/5' : 'hover:bg-yellow-50'}>
                    {visibleDefs.map((col) => (<TableCell key={col.key}>{col.render(p, idx, pageOffset)}</TableCell>))}
                  </TableRow>
                );
              })}
              {patients.data.length === 0 && (
                <TableRow>
                  <TableCell colSpan={visibleDefs.length} className="py-8 text-center text-sm text-muted-foreground">No patients match your filters.</TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={patients.last_page}
            forcePage={patients.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="…"
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>

        {/* Footer quick actions (mobile) */}
        <div className="mt-4 flex items-center justify-between gap-2 md:hidden">
          <Button variant="outline" onClick={handleReset}>Reset</Button>
          <div className="flex gap-2">
            <Button onClick={handleExport} className="gap-2"><FileSpreadsheet className="h-4 w-4" />Export</Button>
            <Button onClick={() => applyQuery(1)} className="gap-2"><SearchIcon className="h-4 w-4" />Search</Button>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
