// resources/js/Pages/Medicals/Patients/Show.tsx
import React, { useRef } from 'react';
import { Head, Link, router, usePage } from '@inertiajs/react';
import moment from 'moment';
import Swal from 'sweetalert2';
import { useReactToPrint } from 'react-to-print';

import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { Badge } from '@/components/ui/badge';
import {
  Table,
  TableHead,
  TableHeader,
  TableRow,
  TableCell,
  TableBody,
} from '@/components/ui/table';

import {
  ArrowLeft,
  BadgeCheck,
  Edit3,
  ImageIcon,
  Printer,
  ShieldAlert,
  FileText,
  Clock,
  Stethoscope,
  Trash2
} from 'lucide-react';

import {
  ResponsiveContainer,
  LineChart,
  Line,
  CartesianGrid,
  XAxis,
  YAxis,
  Tooltip,
  Legend,
  AreaChart,
  Area,
} from 'recharts';

// ──────────────────────────────────────────────────────────
// Types
// ──────────────────────────────────────────────────────────
type Cert = {
  id: number;
  status: string;
  certificate_location?: string | null;
  created_at?: string | null;
  updated_at?: string | null;
  validity?: number | null;
  released_at?: string | null;
  certificate_number?: string | null;
  update_reason?: string | null;
};

type Swab = {
  id: number;
  status: string;
  file?: string | null;
  comment?: string | null;
  created_at?: string | null;
  updated_at?: string | null;
};

type PhysicalExam = {
  id: number;
  bmi: string;
  bmi_status?: string | null;
  bp_dia?: string | null;
  bp_repeat_dia?: string | null;
  bp_repeat_sys?: string | null;
  bp_status?: string | null;
  bp_sys?: string | null;
  created_at?: string | null;
  first_bp_time?: string | null;
  height?: string | null;
  referral_comment?: string | null;
  temp?: string | null;
  pulse?: string | null;
  last_bp_time?: string | null;
  left_vision?: string | null;
  patient_id: number;
  right_vision?: string | null;
  updated_at?: string | null;
  weight?: string | null;
};

type Company = {
  id: number;
  name: string;
  email?: string | null;
  phone?: string | null;
};

// NEW: category type coming from backend
type Category = {
  id: number;
  slug: string;
  name: string;
  form_key?: string | null;
};

type Patient = {
  id: number;
  first_name?: string | null;
  last_name?: string | null;
  age?: number | null;
  employee_number?: string | null;
  national_id?: string | null;
  date_of_birth?: string | null;
  gender?: string | null;
  phone_number?: string | null;
  company?: Company | null;

  // legacy single (kept)
  category?: string | null;

  // NEW: multi-categories for latest planned exam
  categories?: Category[];
  latest_exam_categories?: Category[];

  last_x_ray?: string | null;
  created_at?: string | null;
  updated_at?: string | null;
  latest_certificate?: Cert | null;
  certificates: Cert[];
  swabs: Swab[];
  physical_exams: PhysicalExam[];
};

type PageProps = { patient: Patient };

// ──────────────────────────────────────────────────────────
// Helpers
// ──────────────────────────────────────────────────────────
const prettyDate = (d?: string | null, fallback = '—') =>
  d && moment(d).isValid() ? moment(d).format('DD MMM YYYY, HH:mm') : fallback;

const statusVariant = (
  s?: string | null
): 'default' | 'secondary' | 'destructive' | 'outline' => {
  const v = (s || '').toUpperCase();
  if (['READY', 'RELEASED', 'VALID'].includes(v)) return 'default';
  if (['PENDING', 'MONITORING'].includes(v)) return 'secondary';
  if (v === 'FAILED') return 'destructive';
  return 'outline';
};

const toNumber = (v: string | number | undefined | null): number | null => {
  if (v == null) return null;
  const n = typeof v === 'number' ? v : parseFloat(String(v));
  return Number.isFinite(n) ? n : null;
};

const Info = ({ label, value }: { label: string; value?: string | null }) => (
  <div>
    <div className="text-sm text-muted-foreground">{label}</div>
    <div className="font-medium text-foreground">{value ?? '—'}</div>
  </div>
);

const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
  <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
    {items.map((i) => (
      <Info key={i.label} {...i} />
    ))}
  </div>
);

const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
  <section className="rounded-lg border bg-card p-6 shadow-sm">
    <h2 className="mb-4 text-xl font-semibold text-foreground">{title}</h2>
    {children}
  </section>
);

// ──────────────────────────────────────────────────────────
// Component
// ──────────────────────────────────────────────────────────
export default function MedicalsPatientsShow() {
  const { patient } = usePage<PageProps>().props;

  const fullName =
    [patient.first_name, patient.last_name].filter(Boolean).join(' ') ||
    `Patient #${patient.id}`;
  const ageShort = patient?.age != null ? `${Math.floor(patient.age)}y` : '—';
  const created = prettyDate(patient.created_at);

  const certStatus = patient.latest_certificate?.status ?? null;
  const latestPhysical = patient.physical_exams?.[0]; // already desc from controller

  // Build chart data (ascending by created_at)
  const examsAsc = [...(patient.physical_exams || [])].sort((a, b) => {
    const ta = new Date(a.created_at || '').getTime() || 0;
    const tb = new Date(b.created_at || '').getTime() || 0;
    return ta - tb;
  });

  const chartData = examsAsc.map((e) => ({
    date: e.created_at ? moment(e.created_at).format('DD MMM') : '',
    bp_sys: toNumber(e.bp_sys),
    bp_dia: toNumber(e.bp_dia),
    bmi: toNumber(e.bmi),
  }));

  // Print
  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  const onDelete = async () => {
    const res = await Swal.fire({
      title: 'Delete patient?',
      text: 'This will remove the patient and related records.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
      focusCancel: true,
    });
    if (!res.isConfirmed) return;
    router.delete(route('medicals.patients.destroy', patient.id), {
      onSuccess: () =>
        Swal.fire({
          icon: 'success',
          title: 'Deleted',
          timer: 1500,
          showConfirmButton: false,
        }),
    });
  };

  const manualUpdate = async () => {
    const res = await Swal.fire({
      title: 'Update Certificate Status',
      width: 680,
      icon: 'question',
      html: `
        <div style="text-align:left;">
          <div style="font-size:.9rem; color:#6b7280; margin-bottom:.75rem;">
            Choose a new status, add a reason, and confirm with your password.
          </div>
          <div class="swal2-field" style="display:flex; flex-direction:column; gap:.5rem; margin-bottom:.75rem;">
            <label for="status-select" style="font-weight:600; font-size:.9rem;">Status</label>
            <select id="status-select" class="swal2-input" style="margin:0; height:44px; border-radius:.5rem; padding:.5rem .75rem; border:1px solid #e5e7eb;">
              <option value="PENDING">PENDING</option>
              <option value="MONITORING">MONITORING</option>
              <option value="READY">READY</option>
              <option value="FAILED">FAILED</option>
            </select>
          </div>
          <div class="swal2-field" style="display:flex; flex-direction:column; gap:.5rem; margin-bottom:.75rem;">
            <label for="update-reason" style="font-weight:600; font-size:.9rem;">Update Reason</label>
            <textarea id="update-reason" class="swal2-textarea" placeholder="Enter reason..." style="height:80px; border-radius:.5rem;"></textarea>
          </div>
          <div class="swal2-field" style="display:flex; flex-direction:column; gap:.5rem;">
            <label for="password-input" style="font-weight:600; font-size:.9rem;">User Password</label>
            <input type="password" id="password-input" class="swal2-input" placeholder="Password" style="margin:0; height:44px; border-radius:.5rem;" />
          </div>
        </div>
      `,
      showCancelButton: true,
      confirmButtonText: 'Save',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#2563eb',
      cancelButtonColor: '#6b7280',
      buttonsStyling: false,
      customClass: {
        popup: 'rounded-xl',
        title: 'text-left',
        confirmButton:
          'swal2-confirm rounded-lg px-4 py-2 font-medium bg-blue-600 text-white hover:bg-blue-700',
        cancelButton:
          'swal2-cancel rounded-lg px-4 py-2 font-medium bg-gray-200 text-gray-800 hover:bg-gray-300',
      },
      preConfirm: () => {
        const status = (document.getElementById('status-select') as HTMLSelectElement)?.value || '';
        const reason = (document.getElementById('update-reason') as HTMLTextAreaElement)?.value || '';
        const password = (document.getElementById('password-input') as HTMLInputElement)?.value || '';
        if (!status || !password) {
          Swal.showValidationMessage('Status and Password are required.');
          return false;
        }
        return { status, reason, password };
      },
    });
    if (!res.isConfirmed) return;

    // TODO: implement actual route call
    Swal.fire({
      icon: 'success',
      title: 'Updated',
      text: 'Certificate status updated (placeholder).',
    });
  };

  const addToBatch = async () => {
    if ((patient.latest_certificate?.status || '').toUpperCase() !== 'READY') {
      Swal.fire('Not Ready', 'Certificate must be READY to add to a batch.', 'info');
      return;
    }
    // TODO: implement route
    Swal.fire({ icon: 'success', title: 'Added', text: 'Certificate added to batch (placeholder).' });
  };

  const renewCertificate = async () => {
    // TODO: implement actual route call
    Swal.fire({ icon: 'success', title: 'Renewed', text: 'Certificate renewal initiated (placeholder).' });
  };

  const printCreated = moment().format('MMMM Do YYYY, h:mm A');

  // Convenience
  const latestExamCats: Category[] = patient.latest_exam_categories ?? patient.categories ?? [];

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.dashboard') },
        { title: 'Patients', href: route('medicals.patients.index') },
        { title: fullName, href: route('medicals.patients.show', patient.id) },
      ]}
    >
      <Head title={`Patient: ${fullName}`} />

      {/* Hidden print area */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="p-6 text-sm text-gray-800">
          <style>{`@page{size:auto;margin:12mm 10mm}@media print{body{-webkit-print-color-adjust:exact;print-color-adjust:exact}}`}</style>
          <h2 className="mb-2 text-lg font-bold">Medical Snapshot</h2>
          <Separator className="mb-4" />
          <div className="grid gap-2">
            <div><strong>Name:</strong> {fullName}</div>
            <div><strong>National ID:</strong> {patient.national_id || '-'}</div>
            <div><strong>Company:</strong> {patient.company?.name || '-'}</div>
            {/* Show all planned exam categories in print */}
            <div>
              <strong>Exam Categories:</strong>{' '}
              {latestExamCats.length
                ? latestExamCats.map((c) => c.name).join(', ')
                : (patient.category || '-')}
            </div>
            <div><strong>Created:</strong> {created}</div>
          </div>
          <div className="mt-8 text-xs text-gray-500">Printed on {printCreated}</div>
        </div>
      </div>

      <div className="w-full space-y-8 px-4 py-8 sm:px-6">
        {/* Header row */}
        <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
          <div className="flex items-center gap-4">
            <div className="flex h-20 w-20 items-center justify-center rounded-full border bg-muted text-muted-foreground">
              <ImageIcon className="h-8 w-8" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-foreground">{fullName}</h1>
              <div className="mt-1 text-sm text-muted-foreground">
                {patient.gender || '—'} {ageShort !== '—' ? `• ${ageShort}` : ''}
              </div>

              {/* NEW: planned exam categories (badges) under the name */}
              <div className="mt-2 flex flex-wrap gap-1">
                {latestExamCats.length > 0 ? (
                  latestExamCats.map((c) => (
                    <Badge key={c.slug} variant="secondary">
                      {c.name}
                    </Badge>
                  ))
                ) : patient.category ? (
                  <Badge variant="secondary">{patient.category}</Badge>
                ) : null}
              </div>
            </div>
          </div>

          <div className="flex flex-wrap gap-2">
            <Button asChild variant="outline">
              <Link href={route('medicals.patients.index')}>
                <ArrowLeft className="mr-2 h-4 w-4" />
                Back
              </Link>
            </Button>

            {/* If your edit route is patients.edit, adjust below accordingly */}
            <Button asChild variant="secondary">
              <Link href={route('medicals.attendees.edit', patient.id)}>
                <Edit3 className="mr-2 h-4 w-4" />
                Edit
              </Link>
            </Button>

            <Button asChild>
              <Link href={route('medicals.exams.start', patient.id)}>
                <Stethoscope className="mr-2 h-4 w-4" />
                Start Medical Examination
              </Link>
            </Button>

            <Button variant="outline" onClick={reactToPrintFn}>
              <Printer className="mr-2 h-4 w-4" />
              Print Snapshot
            </Button>
          </div>
        </div>

        {/* Optional alert for pending certificate */}
        {certStatus && certStatus.toUpperCase() === 'PENDING' && (
          <Alert className="border-amber-300 bg-amber-50">
            <ShieldAlert className="h-4 w-4" />
            <AlertTitle>Certificate Pending</AlertTitle>
            <AlertDescription>This patient’s certificate is still pending.</AlertDescription>
          </Alert>
        )}

        <div className="grid grid-cols-12 gap-8">
          {/* LEFT: details & lists */}
          <div className="col-span-12 space-y-6 lg:col-span-8">
            <Section title="Personal Information">
              <InfoGrid
                items={[
                  { label: 'Full Name', value: fullName },
                  { label: 'National ID', value: patient.national_id ?? null },
                  {
                    label: 'Date of Birth',
                    value: patient.date_of_birth ? moment(patient.date_of_birth).format('DD MMM YYYY') : null,
                  },
                  { label: 'Gender', value: patient.gender ?? null },
                  { label: 'Phone', value: patient.phone_number ?? null },
                  { label: 'Employee Number', value: patient.employee_number ?? null },
                ]}
              />
            </Section>

            <Section title="Company & Exam">
              <InfoGrid
                items={[
                  { label: 'Company', value: patient.company?.name ?? null },
                  // legacy single category left visible:
                  { label: 'Primary Category', value: patient.category ?? null },
                  { label: 'Last Chest X-Ray', value: patient.last_x_ray ?? null },
                  { label: 'Created', value: created },
                ]}
              />
              {/* NEW: show full list of planned exam categories */}
              <div className="mt-3">
                <div className="text-sm text-muted-foreground mb-1">
                  Planned Exam Categories
                </div>
                {latestExamCats.length > 0 ? (
                  <div className="flex flex-wrap gap-1">
                    {latestExamCats.map((c) => (
                      <Badge key={c.slug} variant="outline">
                        {c.name}
                      </Badge>
                    ))}
                  </div>
                ) : (
                  <div className="text-sm text-muted-foreground">—</div>
                )}
              </div>
            </Section>

            {/* Certificates */}
            <Section title="Certificates">
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>#</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Location</TableHead>
                      <TableHead>Cert #</TableHead>
                      <TableHead>Created</TableHead>
                      <TableHead>Updated</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {patient.certificates.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={6} className="py-8 text-center text-sm text-muted-foreground">
                          No certificates found.
                        </TableCell>
                      </TableRow>
                    )}
                    {patient.certificates.map((c, idx) => (
                      <TableRow key={c.id}>
                        <TableCell>{idx + 1}</TableCell>
                        <TableCell>
                          <Badge variant={statusVariant(c.status)}>{c.status}</Badge>
                        </TableCell>
                        <TableCell>{c.certificate_location || '—'}</TableCell>
                        <TableCell>{c.certificate_number || '—'}</TableCell>
                        <TableCell>{prettyDate(c.created_at)}</TableCell>
                        <TableCell>{prettyDate(c.updated_at)}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            </Section>

            {/* Trends */}
            <div className="grid grid-cols-1 gap-6 md:grid-cols-2">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Blood Pressure Trend</CardTitle>
                </CardHeader>
                <CardContent className="h-64">
                  {chartData.length > 0 ? (
                    <ResponsiveContainer width="100%" height="100%">
                      <LineChart data={chartData} margin={{ top: 8, right: 16, left: 0, bottom: 0 }}>
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="date" />
                        <YAxis />
                        <Tooltip />
                        <Legend />
                        <Line type="monotone" dataKey="bp_sys" name="Systolic" dot={false} />
                        <Line type="monotone" dataKey="bp_dia" name="Diastolic" dot={false} />
                      </LineChart>
                    </ResponsiveContainer>
                  ) : (
                    <div className="flex h-full items-center justify-center text-sm text-muted-foreground">
                      No blood pressure data.
                    </div>
                  )}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">BMI Trend</CardTitle>
                </CardHeader>
                <CardContent className="h-64">
                  {chartData.length > 0 ? (
                    <ResponsiveContainer width="100%" height="100%">
                      <AreaChart data={chartData} margin={{ top: 8, right: 16, left: 0, bottom: 0 }}>
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="date" />
                        <YAxis />
                        <Tooltip />
                        <Area type="monotone" dataKey="bmi" name="BMI" strokeOpacity={1} fillOpacity={0.2} />
                      </AreaChart>
                    </ResponsiveContainer>
                  ) : (
                    <div className="flex h-full items-center justify-center text-sm text-muted-foreground">
                      No BMI data.
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>

            <Section title="Swabs">
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>#</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>File</TableHead>
                      <TableHead>Comment</TableHead>
                      <TableHead>Created</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {patient.swabs.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={5} className="py-8 text-center text-sm text-muted-foreground">
                          No swabs found.
                        </TableCell>
                      </TableRow>
                    )}
                    {patient.swabs.map((s, idx) => (
                      <TableRow key={s.id}>
                        <TableCell>{idx + 1}</TableCell>
                        <TableCell>
                          <Badge variant={statusVariant(s.status)}>{s.status}</Badge>
                        </TableCell>
                        <TableCell>
                          {s.file ? (
                            <a className="text-blue-600 underline" href={s.file} target="_blank" rel="noreferrer">
                              View
                            </a>
                          ) : (
                            '—'
                          )}
                        </TableCell>
                        <TableCell className="max-w-[320px] truncate">{s.comment || '—'}</TableCell>
                        <TableCell>{prettyDate(s.created_at)}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            </Section>
          </div>

          {/* RIGHT: actions + vitals */}
          <div className="col-span-12 space-y-6 lg:col-span-4">
            <Card>
              <CardHeader>
                <CardTitle className="text-xl">Actions</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <Button asChild className="w-full">
                  <Link href={route('medicals.attendees.edit', patient.id)}>
                    <Edit3 className="mr-2 h-4 w-4" />
                    Edit Bio
                  </Link>
                </Button>

                <Button variant="outline" className="w-full" onClick={reactToPrintFn}>
                  <Printer className="mr-2 h-4 w-4" />
                  Print Snapshot
                </Button>

                {certStatus && certStatus.toUpperCase() === 'READY' ? (
                  <Button className="w-full" onClick={addToBatch}>
                    <FileText className="mr-2 h-4 w-4" />
                    Add to Batch
                  </Button>
                ) : certStatus && certStatus.toUpperCase() === 'RELEASED' ? (
                  <Button className="w-full" onClick={renewCertificate}>
                    <Clock className="mr-2 h-4 w-4" />
                    Renew Certificate
                  </Button>
                ) : (
                  <Button variant="outline" className="w-full" onClick={manualUpdate}>
                    <FileText className="mr-2 h-4 w-4" />
                    Manual Status Update
                  </Button>
                )}

                <Button variant="destructive" className="w-full" onClick={onDelete}>
                  <Trash2 className="mr-2 h-4 w-4" />
                  Delete Patient
                </Button>

                <Separator />

                <div className="rounded-md border p-3 text-sm">
                  <div className="mb-1 flex items-center gap-2 font-medium">
                    <BadgeCheck className="h-4 w-4 text-emerald-600" />
                    Certificate
                  </div>
                  <div className="text-muted-foreground">
                    {certStatus ? `Current status: ${certStatus}` : 'No certificate available.'}
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Vitals / Physical Examination */}
            <Card>
              <CardHeader>
                <CardTitle className="text-xl">Physical Examination</CardTitle>
              </CardHeader>
              <CardContent className="grid grid-cols-2 gap-4 text-sm">
                <Info label="Height" value={latestPhysical?.height ? `${latestPhysical.height} m` : '—'} />
                <Info label="Weight" value={latestPhysical?.weight ? `${latestPhysical.weight} kg` : '—'} />
                <Info
                  label="Blood Pressure"
                  value={
                    latestPhysical?.bp_sys && latestPhysical?.bp_dia
                      ? `${latestPhysical.bp_sys}/${latestPhysical.bp_dia} mmHg`
                      : '—'
                  }
                />
                <Info label="Pulse" value={latestPhysical?.pulse ?? '—'} />
                <Info
                  label="BMI"
                  value={
                    latestPhysical?.bmi
                      ? `${latestPhysical.bmi}${latestPhysical?.bmi_status ? ` (${latestPhysical.bmi_status})` : ''}`
                      : '—'
                  }
                />
                <Info
                  label="Vision (L/R)"
                  value={`${latestPhysical?.left_vision || '—'} / ${latestPhysical?.right_vision || '—'}`}
                />
                <div className="col-span-2">
                  <div className="text-sm text-muted-foreground">Last Updated</div>
                  <div className="font-medium">{prettyDate(latestPhysical?.updated_at)}</div>
                </div>
                <div className="col-span-2">
                  <div className="text-sm text-muted-foreground">Remarks</div>
                  <div className="font-medium">{latestPhysical?.referral_comment || '—'}</div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
