// resources/js/Pages/Medicals/PhysicalExams/Start.tsx
import React, { useEffect, useMemo } from 'react';
import { Head, Link, useForm, usePage } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import moment from 'moment';

// shadcn/ui
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';

// icons
import { Stethoscope, Save, Undo2, Info } from 'lucide-react';

type Patient = {
  id: number;
  first_name?: string | null;
  last_name?: string | null;
  gender?: string | null;
  date_of_birth?: string | null;
  phone_number?: string | null;
  category?: string | null;
  company?: string | null;
};

type Exam = {
  id: number;
  created_at?: string | null;
  // Vital pieces we now also expect from backend to enable prefill:
  height?: number | string | null;
  weight?: number | string | null;

  bmi?: number | string | null;
  bmi_status?: string | null;
  bp_sys?: number | string | null;
  bp_dia?: number | string | null;
  bp_status?: string | null;
  left_vision?: number | string | null;
  right_vision?: number | string | null;
  pulse?: number | string | null;
  temp?: number | string | null;
  blood_sugar?: string | null;
  first_bp_time?: string | null;
  last_bp_time?: string | null;
  referral_comment?: string | null;
};

type PageProps = {
  patient: Patient;
  latestExam: Exam | null;
  flashExam?: Exam | null;
};

function toStr(v: number | string | null | undefined): string {
  if (v === null || v === undefined) return '';
  const s = String(v);
  return s === 'null' || s === 'undefined' ? '' : s;
}

/**
 * Normalize height input:
 * - If user types a decimal already (e.g. "1.8"), keep it (limited to two decimals).
 * - If user types a whole number with >= 3 digits (e.g. "177"), treat it as cm => "1.77".
 * - Strip non-numeric except one dot.
 */
function normalizeHeightInput(raw: string): string {
  // Keep digits and at most one dot
  let value = raw.replace(/[^\d.]/g, '');

  // If there are multiple dots, keep only the first
  const firstDot = value.indexOf('.');
  if (firstDot !== -1) {
    value =
      value.slice(0, firstDot + 1) + value.slice(firstDot + 1).replace(/\./g, '');
  }

  // If there's a dot, leave as-is (trim to 2 decimals)
  if (value.includes('.')) {
    const [intPart, decPart = ''] = value.split('.');
    const limitedDec = decPart.slice(0, 2);
    return intPart + (limitedDec.length ? '.' + limitedDec : '');
  }

  // No dot present. If 3+ digits, treat as centimeters, insert dot after first digit.
  if (value.length >= 3) {
    const intPart = value.slice(0, 1);
    const decPart = value.slice(1, 3); // at most two decimals after conversion
    return `${intPart}.${decPart}`;
  }

  // Otherwise (1-2 digits) just return it. (Uncommon, but user might finish typing soon.)
  return value;
}

export default function MedicalsPhysicalExamsStart() {
  const { patient, latestExam, flashExam } = usePage<PageProps>().props;

  // Prefer the just-saved exam from flash, else fallback to latest from DB.
  const sidebarExam = useMemo<Exam | null>(() => flashExam ?? latestExam ?? null, [flashExam, latestExam]);
  const fullName =
    [patient.first_name, patient.last_name].filter(Boolean).join(' ') || `Patient #${patient.id}`;

  // Inertia form state
  const form = useForm({
    height: '',        // meters (string)
    weight: '',        // kg (string)
    bp_sys: '',
    bp_dia: '',
    left_vision: '',
    right_vision: '',
    temp: '',
    pulse: '',
    blood_sugar: '',
  });

  // Prefill height & weight from existing physical exam (prefer flashExam, else latestExam)
  useEffect(() => {
    const prior = sidebarExam;
    if (!prior) return;

    // height might already be meters; if numeric, convert to string
    const priorHeight = toStr(prior.height);
    const priorWeight = toStr(prior.weight);

    // Only set if the user hasn't typed yet
    if (!form.data.height && priorHeight) {
      // Ensure we show it in "meters" style (e.g., 1.77)
      const normalized = normalizeHeightInput(priorHeight);
      form.setData('height', normalized);
    }
    if (!form.data.weight && priorWeight) {
      form.setData('weight', priorWeight);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [sidebarExam?.id]); // run if the visible exam changes (e.g., after save)

  // Live-normalize height as the user types (auto "." for cm-like input)
  const onHeightChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const next = normalizeHeightInput(e.target.value);
    form.setData('height', next);
  };

  // Submit
  const submit = (e: React.FormEvent) => {
    e.preventDefault();
    form.post(route('medicals.exams.store', patient.id), { preserveScroll: true });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.dashboard') },
        { title: 'Patients', href: route('medicals.patients.index') },
        { title: fullName, href: route('medicals.patients.show', patient.id) },
        { title: 'Start Exam', href: route('medicals.exams.start', patient.id) },
      ]}
    >
      <Head title={`Start Exam • ${fullName}`} />

      <div className="mx-4 my-6 grid grid-cols-12 gap-6 md:mx-8">
        {/* LEFT: Start Exam form */}
        <div className="col-span-12 space-y-4 lg:col-span-8">
          <Card>
            <CardHeader className="flex items-center justify-between gap-2 sm:flex-row">
              <CardTitle className="flex items-center gap-2">
                <Stethoscope className="h-5 w-5" />
                Start Medical Examination
              </CardTitle>
              <Button asChild variant="outline">
                <Link href={route('medicals.patients.show', patient.id)}>Back to Patient</Link>
              </Button>
            </CardHeader>

            <CardContent>
              <form className="grid grid-cols-1 gap-4 md:grid-cols-2" onSubmit={submit}>
                {/* Height (m) */}
                <div>
                  <div className="flex items-center justify-between">
                    <Label htmlFor="height">Height (m)</Label>
                    <TooltipProvider>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <span className="inline-flex items-center text-xs text-muted-foreground cursor-help">
                            <Info className="mr-1 h-3.5 w-3.5" />
                            help
                          </span>
                        </TooltipTrigger>
                        <TooltipContent side="top" align="end" className="max-w-xs">
                          <p className="text-xs">
                            You can type meters (e.g., <b>1.77</b>) or centimeters (e.g., <b>177</b>).
                            We’ll automatically format <b>177</b> → <b>1.77 m</b>.
                          </p>
                        </TooltipContent>
                      </Tooltip>
                    </TooltipProvider>
                  </div>

                  <Input
                    id="height"
                    inputMode="decimal"
                    value={form.data.height}
                    onChange={onHeightChange}
                    placeholder="e.g. 1.75 or 175"
                  />
                  <p className="mt-1 text-xs text-muted-foreground">
                    Tip: typing <b>177</b> becomes <b>1.77</b> automatically (meters).
                  </p>
                  {form.errors.height && <p className="mt-1 text-xs text-red-600">{form.errors.height}</p>}
                </div>

                {/* Weight (kg) */}
                <div>
                  <Label htmlFor="weight">Weight (kg)</Label>
                  <Input
                    id="weight"
                    inputMode="decimal"
                    type="number"
                    step="0.1"
                    value={form.data.weight}
                    onChange={(e) => form.setData('weight', e.target.value)}
                    placeholder="e.g. 73.5"
                  />
                  {form.errors.weight && <p className="mt-1 text-xs text-red-600">{form.errors.weight}</p>}
                </div>

                {/* BP SYS */}
                <div>
                  <Label htmlFor="bp_sys">Systolic BP</Label>
                  <Input
                    id="bp_sys"
                    type="number"
                    value={form.data.bp_sys}
                    onChange={(e) => form.setData('bp_sys', e.target.value)}
                    placeholder="e.g. 120"
                  />
                  {form.errors.bp_sys && <p className="mt-1 text-xs text-red-600">{form.errors.bp_sys}</p>}
                </div>

                {/* BP DIA */}
                <div>
                  <Label htmlFor="bp_dia">Diastolic BP</Label>
                  <Input
                    id="bp_dia"
                    type="number"
                    value={form.data.bp_dia}
                    onChange={(e) => form.setData('bp_dia', e.target.value)}
                    placeholder="e.g. 80"
                  />
                  {form.errors.bp_dia && <p className="mt-1 text-xs text-red-600">{form.errors.bp_dia}</p>}
                </div>

                {/* Pulse */}
                <div>
                  <Label htmlFor="pulse">Pulse</Label>
                  <Input
                    id="pulse"
                    type="number"
                    value={form.data.pulse}
                    onChange={(e) => form.setData('pulse', e.target.value)}
                    placeholder="e.g. 72"
                  />
                  {form.errors.pulse && <p className="mt-1 text-xs text-red-600">{form.errors.pulse}</p>}
                </div>

                {/* Temperature */}
                <div>
                  <Label htmlFor="temp">Temperature (°C)</Label>
                  <Input
                    id="temp"
                    type="number"
                    step="0.1"
                    value={form.data.temp}
                    onChange={(e) => form.setData('temp', e.target.value)}
                    placeholder="e.g. 36.8"
                  />
                  {form.errors.temp && <p className="mt-1 text-xs text-red-600">{form.errors.temp}</p>}
                </div>

                {/* Left Vision */}
                <div>
                  <Label htmlFor="left_vision">Left Vision (out of 6)</Label>
                  <Input
                    id="left_vision"
                    type="number"
                    value={form.data.left_vision}
                    onChange={(e) => form.setData('left_vision', e.target.value)}
                    placeholder="e.g. 6"
                  />
                </div>

                {/* Right Vision */}
                <div>
                  <Label htmlFor="right_vision">Right Vision (out of 6)</Label>
                  <Input
                    id="right_vision"
                    type="number"
                    value={form.data.right_vision}
                    onChange={(e) => form.setData('right_vision', e.target.value)}
                    placeholder="e.g. 6"
                  />
                </div>

                {/* Blood sugar */}
                <div className="md:col-span-2">
                  <Label htmlFor="blood_sugar">Blood Sugar</Label>
                  <Input
                    id="blood_sugar"
                    value={form.data.blood_sugar}
                    onChange={(e) => form.setData('blood_sugar', e.target.value)}
                    placeholder="optional"
                  />
                </div>

                {/* Actions */}
                <div className="md:col-span-2 mt-2 flex items-center gap-2">
                  <Button type="submit" disabled={form.processing}>
                    <Save className="mr-2 h-4 w-4" />
                    {form.processing ? 'Saving…' : 'Save & Continue'}
                  </Button>
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => form.reset()}
                    disabled={form.processing}
                  >
                    <Undo2 className="mr-2 h-4 w-4" />
                    Reset
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </div>

        {/* RIGHT: Sidebar — shows latest or just-saved exam */}
        <div className="col-span-12 lg:col-span-4">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Latest Physical Exam</CardTitle>
            </CardHeader>
            <CardContent>
              {!sidebarExam ? (
                <div className="text-sm text-muted-foreground">
                  No exam yet. Fill the form to create one.
                </div>
              ) : (
                <div className="space-y-3 text-sm">
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">Date</span>
                    <span className="font-medium">
                      {sidebarExam.created_at ? moment(sidebarExam.created_at).format('DD MMM YYYY, HH:mm') : '—'}
                    </span>
                  </div>
                  <Separator />

                  <div className="grid grid-cols-2 gap-2">
                    <div>
                      <div className="text-muted-foreground">Height</div>
                      <div className="font-medium">
                        {sidebarExam.height != null && sidebarExam.height !== ''
                          ? `${normalizeHeightInput(toStr(sidebarExam.height))} m`
                          : '—'}
                      </div>
                    </div>
                    <div>
                      <div className="text-muted-foreground">Weight</div>
                      <div className="font-medium">
                        {sidebarExam.weight != null && sidebarExam.weight !== '' ? `${sidebarExam.weight} kg` : '—'}
                      </div>
                    </div>

                    <div>
                      <div className="text-muted-foreground">BP</div>
                      <div className="font-medium">
                        {sidebarExam.bp_sys ?? '—'} / {sidebarExam.bp_dia ?? '—'}
                      </div>
                      {sidebarExam.bp_status && <Badge className="mt-1" variant="outline">{sidebarExam.bp_status}</Badge>}
                    </div>
                    <div>
                      <div className="text-muted-foreground">BMI</div>
                      <div className="font-medium">{sidebarExam.bmi ?? '—'}</div>
                      {sidebarExam.bmi_status && <Badge className="mt-1">{sidebarExam.bmi_status}</Badge>}
                    </div>
                    <div>
                      <div className="text-muted-foreground">Pulse</div>
                      <div className="font-medium">{sidebarExam.pulse ?? '—'}</div>
                    </div>
                    <div>
                      <div className="text-muted-foreground">Temp (°C)</div>
                      <div className="font-medium">{sidebarExam.temp ?? '—'}</div>
                    </div>
                    <div>
                      <div className="text-muted-foreground">Vision (L)</div>
                      <div className="font-medium">{sidebarExam.left_vision ?? '—'}</div>
                    </div>
                    <div>
                      <div className="text-muted-foreground">Vision (R)</div>
                      <div className="font-medium">{sidebarExam.right_vision ?? '—'}</div>
                    </div>
                    <div className="col-span-2">
                      <div className="text-muted-foreground">Blood Sugar</div>
                      <div className="font-medium">{sidebarExam.blood_sugar ?? '—'}</div>
                    </div>
                    <div className="col-span-2">
                      <div className="text-muted-foreground">Referral Comment</div>
                      <div className="font-medium">{sidebarExam.referral_comment || '—'}</div>
                    </div>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </AppLayout>
  );
}
