import AppLayout from '@/layouts/app-layout';
import { Head, router, usePage } from '@inertiajs/react';
import React, { useMemo, useState } from 'react';
import moment from 'moment';

// shadcn/ui
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogTrigger, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Separator } from '@/components/ui/separator';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Calendar as CalendarIcon, Clock } from 'lucide-react';

type Row = {
  id: number;
  entity: 'certificate' | 'patient';
  created_at: string;
  patient_id: number | null;
  first_name?: string | null;
  last_name?: string | null;
  company_name?: string | null;
  category?: string | null;
  status?: string | null;
};

type PageProps = {
  now: string;
  certificates: Row[];
  patients: Row[];
};

export default function Corrections() {
  const { now, certificates = [], patients = [] } = usePage<PageProps>().props;
  const [query, setQuery] = useState('');

  const filterMatch = (r: Row) => {
    const q = query.trim().toLowerCase();
    if (!q) return true;
    return (
      (r.first_name || '').toLowerCase().includes(q) ||
      (r.last_name || '').toLowerCase().includes(q) ||
      (r.company_name || '').toLowerCase().includes(q) ||
      String(r.patient_id ?? '').includes(q)
    );
  };

  const filteredCerts = useMemo(() => certificates.filter(filterMatch), [certificates, query]);
  const filteredPatients = useMemo(() => patients.filter(filterMatch), [patients, query]);

  const [editOpen, setEditOpen] = useState(false);
  const [editRow, setEditRow] = useState<Row | null>(null);
  const [newDateTime, setNewDateTime] = useState<string>('');

  const openEdit = (row: Row) => {
    setEditRow(row);
    // default to "now" rounded to minutes
    const suggestion = moment().format('YYYY-MM-DDTHH:mm');
    setNewDateTime(suggestion);
    setEditOpen(true);
  };

  const submitFix = () => {
    if (!editRow || !newDateTime) return;
    router.patch(
      route('medicals.reports.corrections.fix'),
      {
        entity: editRow.entity,
        id: editRow.id,
        new_created_at: newDateTime, // 'YYYY-MM-DDTHH:mm'
      },
      {
        preserveScroll: true,
        onSuccess: () => setEditOpen(false),
      }
    );
  };

  const bulkFixAll = (rows: Row[]) => {
    const dt = moment().format('YYYY-MM-DDTHH:mm');
    // naive sequential calls (keeps it simple)
    rows.forEach((r) => {
      router.patch(route('medicals.reports.corrections.fix'), {
        entity: r.entity,
        id: r.id,
        new_created_at: dt,
      }, { preserveScroll: true });
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.reports.index') },
        { title: 'Corrections', href: route('medicals.reports.corrections') },
      ]}
    >
      <Head title="Medicals • Date Corrections" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        <div className="flex flex-col items-start justify-between gap-3 md:flex-row md:items-center">
          <h1 className="text-2xl font-bold">Future-dated Records</h1>
          <div className="text-sm text-muted-foreground">
            Server time now:&nbsp;
            <Badge variant="outline">{moment(now).format('YYYY-MM-DD HH:mm:ss')}</Badge>
          </div>
        </div>

        <Card>
          <CardHeader className="flex flex-col gap-2 md:flex-row md:items-center md:justify-between">
            <CardTitle>Search & Bulk Actions</CardTitle>
            <div className="flex flex-col gap-2 md:flex-row md:items-center">
              <Input
                placeholder="Search by first/last name, company, or patient id…"
                value={query}
                onChange={(e) => setQuery(e.target.value)}
                className="w-[280px]"
              />
              <Separator orientation="vertical" className="hidden h-8 md:block" />
              <div className="flex gap-2">
                <Button
                  variant="secondary"
                  disabled={filteredCerts.length === 0}
                  onClick={() => bulkFixAll(filteredCerts)}
                  title="Set all certificate timestamps to now"
                >
                  Fix All Certificates to Now
                </Button>
                <Button
                  variant="secondary"
                  disabled={filteredPatients.length === 0}
                  onClick={() => bulkFixAll(filteredPatients)}
                  title="Set all patient timestamps to now"
                >
                  Fix All Patients to Now
                </Button>
              </div>
            </div>
          </CardHeader>

          <CardContent className="space-y-6">
            <Alert>
              <AlertDescription>
                These rows have <strong>created_at</strong> in the future. Use the edit button to set a correct timestamp
                (it cannot be in the future). Bulk actions set timestamps to the current time.
              </AlertDescription>
            </Alert>

            {/* Certificates */}
            <section className="space-y-3">
              <div className="flex items-center justify-between">
                <h2 className="text-lg font-semibold">
                  Certificates <Badge variant="secondary">{filteredCerts.length}</Badge>
                </h2>
              </div>

              <div className="overflow-x-auto rounded-lg border">
                <Table className="min-w-[960px]">
                  <TableHeader>
                    <TableRow>
                      <TableHead>ID</TableHead>
                      <TableHead>Patient</TableHead>
                      <TableHead>Company</TableHead>
                      <TableHead>Category</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Created At</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredCerts.map((r) => (
                      <TableRow key={`cert-${r.id}`}>
                        <TableCell>{r.id}</TableCell>
                        <TableCell>
                          {(r.first_name ?? '—') + ' ' + (r.last_name ?? '')}
                        </TableCell>
                        <TableCell>{r.company_name ?? '—'}</TableCell>
                        <TableCell>{r.category ?? '—'}</TableCell>
                        <TableCell>
                          <Badge variant="outline">{r.status ?? '—'}</Badge>
                        </TableCell>
                        <TableCell>
                          <Badge variant="destructive" title="Future date">
                            {moment(r.created_at).format('YYYY-MM-DD HH:mm')}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right">
                          <Button size="sm" onClick={() => openEdit(r)}>
                            Edit Date
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}

                    {filteredCerts.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={7} className="py-8 text-center text-sm text-muted-foreground">
                          No future-dated certificates found.
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                </Table>
              </div>
            </section>

            {/* Patients */}
            <section className="space-y-3">
              <div className="flex items-center justify-between">
                <h2 className="text-lg font-semibold">
                  Patients <Badge variant="secondary">{filteredPatients.length}</Badge>
                </h2>
              </div>

              <div className="overflow-x-auto rounded-lg border">
                <Table className="min-w-[960px]">
                  <TableHeader>
                    <TableRow>
                      <TableHead>ID</TableHead>
                      <TableHead>Patient</TableHead>
                      <TableHead>Company</TableHead>
                      <TableHead>Category</TableHead>
                      <TableHead>Created At</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredPatients.map((r) => (
                      <TableRow key={`pat-${r.id}`}>
                        <TableCell>{r.id}</TableCell>
                        <TableCell>
                          {(r.first_name ?? '—') + ' ' + (r.last_name ?? '')}
                        </TableCell>
                        <TableCell>{r.company_name ?? '—'}</TableCell>
                        <TableCell>{r.category ?? '—'}</TableCell>
                        <TableCell>
                          <Badge variant="destructive" title="Future date">
                            {moment(r.created_at).format('YYYY-MM-DD HH:mm')}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right">
                          <Button size="sm" onClick={() => openEdit(r)}>
                            Edit Date
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}

                    {filteredPatients.length === 0 && (
                      <TableRow>
                        <TableCell colSpan={6} className="py-8 text-center text-sm text-muted-foreground">
                          No future-dated patients found.
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                </Table>
              </div>
            </section>
          </CardContent>
        </Card>
      </div>

      {/* Edit dialog */}
      <Dialog open={editOpen} onOpenChange={setEditOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Update timestamp</DialogTitle>
          </DialogHeader>
          <div className="space-y-2">
            <div className="text-sm text-muted-foreground">
              {editRow?.entity === 'certificate' ? 'Certificate' : 'Patient'} ID:&nbsp;
              <Badge variant="outline">{editRow?.id}</Badge>
            </div>
            <div className="grid gap-2">
              <label className="text-sm font-medium">New created_at (cannot be in the future)</label>
              <div className="flex items-center gap-2">
                <CalendarIcon className="h-4 w-4 text-muted-foreground" />
                <Input
                  type="datetime-local"
                  value={newDateTime}
                  onChange={(e) => setNewDateTime(e.target.value)}
                  className="w-[260px]"
                />
              </div>
              <div className="text-xs text-muted-foreground flex items-center gap-1">
                <Clock className="h-3.5 w-3.5" />
                Current server time: {moment(now).format('YYYY-MM-DD HH:mm:ss')}
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="ghost" onClick={() => setEditOpen(false)}>Cancel</Button>
            <Button onClick={submitFix}>Save</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  );
}
