import { Head, Link, usePage } from '@inertiajs/react';
import React, { Fragment, useMemo, useState } from 'react';
import * as XLSX from 'xlsx';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Table, TableBody, TableCell, TableHead, TableHeader, TableRow,
} from '@/components/ui/table';
import {
  Select, SelectContent, SelectItem, SelectTrigger, SelectValue,
} from '@/components/ui/select';
import { Download, Building2, Search, CalendarDays } from 'lucide-react';
import moment from 'moment';

type Patient = {
  first_name: string;
  last_name: string;
  date_of_birth?: string | null;
  gender?: string | null;
  national_id?: string | null;
  phone_number?: string | null;
  company?: string | null;
  category?: string | null;
  patient_id: number;
};

type DayReport = {
  day: string; // YYYY-MM-DD
  certificates_count: number;
  [category: string]: any; // category arrays
};

type PageProps = {
  day: string;
  report: DayReport;               // same structure returned by controller for that day
  companies: string[];             // list of company names present that day
};

const CATEGORIES = [
  'Pneumoconiosis',
  'Food Handler (COH)',
  'Pre-Employement',
  'Exit-Employement',
  'Exit-Pneumoconiosis',
];

export default function ReportsDay() {
  const { day, report, companies } = usePage<PageProps>().props;

  const [search, setSearch] = useState('');
  const [company, setCompany] = useState<string>('ALL');

  // Build a flat array of all patients (with category kept)
  const allPatients: Patient[] = useMemo(() => {
    if (!report) return [];
    const arr: Patient[] = [];
    CATEGORIES.forEach((cat) => {
      const list: Patient[] = report?.[cat] ?? [];
      list?.forEach((p) => arr.push({ ...p, category: cat }));
    });
    return arr;
  }, [report]);

  // Summary counts by company (for cards)
  const companyCounts = useMemo(() => {
    const acc: Record<string, number> = {};
    allPatients.forEach((p) => {
      const key = p.company ?? '—';
      acc[key] = (acc[key] || 0) + 1;
    });
    return acc;
  }, [allPatients]);

  // Filtering
  const filteredByCompany = useMemo(() => {
    if (company === 'ALL') return allPatients;
    return allPatients.filter((p) => (p.company ?? '') === company);
  }, [allPatients, company]);

  const filtered = useMemo(() => {
    const s = search.trim().toLowerCase();
    if (!s) return filteredByCompany;
    return filteredByCompany.filter((p) => {
      const first = (p.first_name ?? '').toLowerCase();
      const last = (p.last_name ?? '').toLowerCase();
      return first.includes(s) || last.includes(s);
    });
  }, [filteredByCompany, search]);

  // Build per-category buckets AFTER searching/filtering
  const perCategory: Record<string, Patient[]> = useMemo(() => {
    const map: Record<string, Patient[]> = {};
    CATEGORIES.forEach((cat) => (map[cat] = []));
    filtered.forEach((p) => {
      const cat = p.category ?? 'Unknown';
      if (!map[cat]) map[cat] = [];
      map[cat].push(p);
    });
    return map;
  }, [filtered]);

  const exportToExcel = () => {
    const dataToExport = filtered.map((p) => ({
      Category: p.category ?? '',
      'First Name': p.first_name,
      'Last Name': p.last_name,
      'Date Of Birth': p.date_of_birth ?? '',
      Gender: p.gender ?? '',
      'National ID': p.national_id ?? '',
      'Phone Number': p.phone_number ?? '',
      Company: p.company ?? '',
    }));

    if (!dataToExport.length) return;

    const worksheet = XLSX.utils.json_to_sheet(dataToExport);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, 'SingleDayReport');
    const title = moment(day, 'YYYY-MM-DD').format('DD-MMMM-YYYY');
    XLSX.writeFile(workbook, `Single_Report_${title}.xlsx`);
  };

  const renderCategoryTable = (categoryName: string) => {
    const data: Patient[] = perCategory[categoryName] ?? [];
    if (!data.length) return null;

    return (
      <Fragment key={categoryName}>
        <div className="flex items-center justify-between">
          <h4 className="text-base font-semibold uppercase">
            {categoryName}{' '}
            <Badge variant="secondary" className="align-middle">
              {data.length} {data.length === 1 ? 'Patient' : 'Patients'}
            </Badge>
          </h4>
        </div>

        <div className="overflow-x-auto rounded-lg border">
          <Table className="min-w-[880px]">
            <TableHeader>
              <TableRow>
                <TableHead>First Name</TableHead>
                <TableHead>Surname</TableHead>
                <TableHead>Date Of Birth</TableHead>
                <TableHead>Gender</TableHead>
                <TableHead>National ID</TableHead>
                <TableHead>Phone Number</TableHead>
                <TableHead>Company</TableHead>
                <TableHead className="text-right">Action</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {data.map((patient, idx) => (
                <TableRow key={`${categoryName}-${idx}-${patient.patient_id}`}>
                  <TableCell>{patient.first_name}</TableCell>
                  <TableCell>{patient.last_name}</TableCell>
                  <TableCell>{patient.date_of_birth ?? '—'}</TableCell>
                  <TableCell>{patient.gender ?? '—'}</TableCell>
                  <TableCell>{patient.national_id ?? '—'}</TableCell>
                  <TableCell>{patient.phone_number ?? '—'}</TableCell>
                  <TableCell>{patient.company ?? '—'}</TableCell>
                  <TableCell className="text-right">
                    {/* Adjust route name if your edit path differs */}
                    <Button asChild variant="secondary" size="sm" title="Edit Patient">
                      <Link href={route('medicals.attendees.edit', patient.patient_id)}>Edit</Link>
                    </Button>
                  </TableCell>
                </TableRow>
              ))}
              {data.length === 0 && (
                <TableRow>
                  <TableCell colSpan={8} className="text-center text-sm text-muted-foreground">
                    No patients in this category.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>
      </Fragment>
    );
  };

  const prettyDay = moment(day, 'YYYY-MM-DD').format('dddd D MMMM YYYY');

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.dashboard') },
        { title: 'Reports', href: route('medicals.reports.index') },
        { title: prettyDay, href: route('medicals.reports.day', { day }) },
      ]}
    >
      <Head title={`Medicals • Report • ${prettyDay}`} />

      <div className="mx-4 my-6 space-y-6 md:mx-8">
        {/* Header / Title + Export */}
        <div className="flex flex-col items-start justify-between gap-3 md:flex-row md:items-center">
          <div className="flex items-center gap-2">
            <CalendarDays className="h-5 w-5 text-primary" />
            <h1 className="text-xl font-semibold">
              {prettyDay} <span className="text-muted-foreground">Daily Report</span>
            </h1>
          </div>
          <div className="flex gap-2">
            <Button onClick={exportToExcel} className="gap-2">
              <Download className="h-4 w-4" />
              Export to Excel
            </Button>
          </div>
        </div>

        {/* Filters */}
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-base">Filters</CardTitle>
          </CardHeader>
          <CardContent className="grid grid-cols-1 gap-4 md:grid-cols-3">
            <div className="col-span-2">
              <Label>Search by name</Label>
              <div className="relative mt-1">
                <Search className="text-muted-foreground absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2" />
                <Input
                  placeholder="Search first name or surname…"
                  className="pl-9"
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                />
              </div>
            </div>
            <div>
              <Label>Company</Label>
              <Select value={company} onValueChange={(v) => setCompany(v)}>
                <SelectTrigger className="mt-1">
                  <SelectValue placeholder="All Companies" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ALL">All Companies</SelectItem>
                  {companies.map((c) => (
                    <SelectItem key={c || '—'} value={c || '—'}>
                      <div className="inline-flex items-center gap-2">
                        <Building2 className="h-4 w-4" />
                        {c || '—'}
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        </Card>

        {/* Company Summary Cards */}
        <div className="grid grid-cols-1 gap-3 sm:grid-cols-2 lg:grid-cols-4">
          {Object.entries(companyCounts).map(([c, count], i) => (
            <Card key={`${c}-${i}`}>
              <CardHeader className="pb-2">
                <CardTitle className="text-sm font-medium">{c}</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-semibold">{count}</div>
                <div className="text-xs text-muted-foreground">Patients</div>
              </CardContent>
            </Card>
          ))}
          {Object.keys(companyCounts).length === 0 && (
            <Card>
              <CardContent className="py-6 text-sm text-muted-foreground">
                No patients to summarize.
              </CardContent>
            </Card>
          )}
        </div>

        {/* Category Tables */}
        <div className="space-y-6">
          {CATEGORIES.map((cat) => renderCategoryTable(cat))}
        </div>
      </div>
    </AppLayout>
  );
}
