import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useMemo, useState } from 'react';
import moment from 'moment';
import * as XLSX from 'xlsx';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';
import {
  Select, SelectTrigger, SelectValue, SelectContent, SelectItem,
} from '@/components/ui/select';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';

import { FileSpreadsheet, Search as SearchIcon } from 'lucide-react';

type Company = { id: number | string; company_name: string };

type PatientRow = {
  id: number;
  category?: string | null;
  created_at?: string | null;            // medical date (latest certificate or fallback)
  certificate_status?: string | null;
  attendee: {
    employee_number?: string | null;
    first_name?: string | null;
    last_name?: string | null;
    date_of_birth?: string | null;
    gender?: string | null;
    phone_number?: string | null;
    national_id?: string | null;
    company?: {
      id?: number | string | null;
      company_name?: string | null;
    } | null;
  };
};

type PageProps = {
  patients: PatientRow[];
  companies: Company[];
  filters: {
    year?: number | null;
    month?: number | null;
    company?: number | string | null;
    category?: string | null;
    date_from?: string | null;
    date_to?: string | null;
  };
};

const categories = [
  'Food Handler (COH)',
  'Pre-Employement',
  'Exit-Employement',
  'Pneumoconiosis',
  'Exit-Pneumoconiosis',
];

function exportToExcel(rows: PatientRow[]) {
  if (!rows.length) return;
  const flat = rows.map((item) => ({
    MEDICAL_DATE: item.created_at ? moment(item.created_at).format('DD/MM/YYYY') : '',
    EMPLOYEE_NUMBER: item.attendee.employee_number ?? '',
    FIRST_NAME: item.attendee.first_name ?? '',
    LAST_NAME: item.attendee.last_name ?? '',
    DATE_OF_BIRTH: item.attendee.date_of_birth ?? '',
    GENDER: item.attendee.gender ?? '',
    PHONE_NUMBER: item.attendee.phone_number ?? '',
    NATIONAL_ID: item.attendee.national_id ?? '',
    COMPANY: item.attendee.company?.company_name ?? '',
    CERTIFICATE_STATUS: item.certificate_status ?? '',
    EXAMINATION_TYPE: '', // (optional field in your legacy UI; left blank here)
    MEDICAL_CATEGORY: item.category ?? '',
  }));

  const ws = XLSX.utils.json_to_sheet(flat);
  const cols: { wch: number }[] = [];
  Object.keys(flat[0]).forEach(() => cols.push({ wch: 20 }));
  (ws as any)['!cols'] = cols;

  const wb = XLSX.utils.book_new();
  XLSX.utils.book_append_sheet(wb, ws, 'Report');
  XLSX.writeFile(wb, `Report_${moment().format('YYYY-MM-DD_HH-mm')}.xlsx`);
}

export default function GeneratedReports() {
  const { patients = [], companies = [], filters } = usePage<PageProps>().props;

  // Controlled (defaults from server)
  const [year, setYear] = useState<string>(filters.year ? String(filters.year) : '');
  const [month, setMonth] = useState<string>(filters.month ? String(filters.month) : '');
  const [company, setCompany] = useState<string>(filters.company ? String(filters.company) : '');
  const [category, setCategory] = useState<string>(filters.category ?? '');
  const [dateFrom, setDateFrom] = useState<string>(filters.date_from ?? '');
  const [dateTo, setDateTo] = useState<string>(filters.date_to ?? '');

  const apply = () => {
    router.get(
      route('medicals.reports.generated'),
      {
        year: year || undefined,
        month: month || undefined,
        company: company || undefined,
        category: category || undefined,
        date_from: dateFrom || undefined,
        date_to: dateTo || undefined,
      },
      { preserveState: true, replace: true, preserveScroll: true },
    );
  };

  const reset = () => {
    setYear('');
    setMonth('');
    setCompany('');
    setCategory('');
    setDateFrom('');
    setDateTo('');
    router.get(route('medicals.reports.generated'), {}, { preserveState: true, replace: true, preserveScroll: true });
  };

  const yearOptions = useMemo(() => {
    const now = new Date().getFullYear();
    const start = 2019; // 2019..now
    const years: number[] = [];
    for (let y = now; y >= start; y--) years.push(y);
    return years;
  }, []);

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.reports.generated') },
        { title: 'Generated Reports', href: route('medicals.reports.generated') },
      ]}
    >
      <Head title="Medicals • Generated Reports" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        <Card>
          <CardHeader>
            <CardTitle>Reports Generation</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {/* Filters row 1 */}
            <div className="grid grid-cols-1 gap-4 md:grid-cols-6">
              {/* Year */}
              <div>
                <Label>Year</Label>
                <Select value={year || 'ALL'} onValueChange={(v) => setYear(v === 'ALL' ? '' : v)}>
                  <SelectTrigger><SelectValue placeholder="All Years" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Years</SelectItem>
                    {yearOptions.map((y) => (
                      <SelectItem key={y} value={String(y)}>{y}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Month */}
              <div>
                <Label>Month</Label>
                <Select value={month || 'ALL'} onValueChange={(v) => setMonth(v === 'ALL' ? '' : v)}>
                  <SelectTrigger><SelectValue placeholder="All Months" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Months</SelectItem>
                    <SelectItem value="1">January</SelectItem>
                    <SelectItem value="2">February</SelectItem>
                    <SelectItem value="3">March</SelectItem>
                    <SelectItem value="4">April</SelectItem>
                    <SelectItem value="5">May</SelectItem>
                    <SelectItem value="6">June</SelectItem>
                    <SelectItem value="7">July</SelectItem>
                    <SelectItem value="8">August</SelectItem>
                    <SelectItem value="9">September</SelectItem>
                    <SelectItem value="10">October</SelectItem>
                    <SelectItem value="11">November</SelectItem>
                    <SelectItem value="12">December</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {/* Company */}
              <div>
                <Label>Company</Label>
                <Select value={company || 'ALL'} onValueChange={(v) => setCompany(v === 'ALL' ? '' : v)}>
                  <SelectTrigger><SelectValue placeholder="All Companies" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Companies</SelectItem>
                    {companies.map((c) => (
                      <SelectItem key={String(c.id)} value={String(c.id)}>
                        {c.company_name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Category */}
              <div className="md:col-span-2">
                <Label>Category</Label>
                <Select value={category || 'ALL'} onValueChange={(v) => setCategory(v === 'ALL' ? '' : v)}>
                  <SelectTrigger><SelectValue placeholder="All Categories" /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Categories</SelectItem>
                    {categories.map((c) => (<SelectItem key={c} value={c}>{c}</SelectItem>))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            {/* Filters row 2 (date range) */}
            <div className="grid grid-cols-1 gap-4 md:grid-cols-6">
              <div className="md:col-span-3">
                <Label>Date From</Label>
                <Input type="date" value={dateFrom} onChange={(e) => setDateFrom(e.target.value)} />
              </div>
              <div className="md:col-span-3">
                <Label>Date To</Label>
                <Input type="date" value={dateTo} onChange={(e) => setDateTo(e.target.value)} />
              </div>
            </div>

            {/* Actions */}
            <div className="flex flex-wrap gap-2 pt-2">
              <Button onClick={apply} className="gap-2">
                <SearchIcon className="h-4 w-4" />
                Generate Report
              </Button>
              <Button variant="outline" onClick={reset}>Reset</Button>
              <Button variant="secondary" onClick={() => exportToExcel(patients)} className="gap-2">
                <FileSpreadsheet className="h-4 w-4" />
                Download Excel (xlsx)
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Results */}
        <Card>
          <CardHeader>
            <CardTitle>
              Generated Results{' '}
              <Badge variant="secondary" className="ml-2">{patients.length}</Badge>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto rounded-lg border">
              <Table className="min-w-[900px]">
                <TableHeader>
                  <TableRow>
                    <TableHead>Medical Date</TableHead>
                    <TableHead>First Name</TableHead>
                    <TableHead>Last Name</TableHead>
                    <TableHead>Gender</TableHead>
                    <TableHead>Company</TableHead>
                    <TableHead>Category</TableHead>
                    <TableHead>Phone Number</TableHead>
                    <TableHead>Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {patients.map((p) => (
                    <TableRow key={p.id}>
                      <TableCell>{p.created_at ? moment(p.created_at).format('D MMMM YYYY') : '—'}</TableCell>
                      <TableCell>{p.attendee.first_name ?? '—'}</TableCell>
                      <TableCell>{p.attendee.last_name ?? '—'}</TableCell>
                      <TableCell>{p.attendee.gender ?? '—'}</TableCell>
                      <TableCell>{p.attendee.company?.company_name ?? '—'}</TableCell>
                      <TableCell>{p.category ?? '—'}</TableCell>
                      <TableCell>{p.attendee.phone_number ?? '—'}</TableCell>
                      <TableCell>
                        <Link
                          href={route('medicals.patients.show', p.id)}
                          className="text-primary hover:underline"
                        >
                          View
                        </Link>
                      </TableCell>
                    </TableRow>
                  ))}

                  {patients.length === 0 && (
                    <TableRow>
                      <TableCell colSpan={8} className="py-8 text-center text-sm text-muted-foreground">
                        No results. Adjust filters and try again.
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
