import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useMemo, useState } from 'react';
import moment from 'moment';
import ActivityCalendar, { Activity } from 'react-activity-calendar';

// shadcn/ui
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { Popover, PopoverTrigger, PopoverContent } from '@/components/ui/popover';
import { Calendar } from '@/components/ui/calendar';
import { Calendar as CalendarIcon, ChevronLeft, ChevronRight } from 'lucide-react';

type ActivityRow = { day: string; activity: number };

type ReportDayRow = {
  day: string;
  certificates_count: number;
  patients_count: number;
  [category: string]: any; // category arrays live here
};

type PageProps = {
  activityByDay: ActivityRow[];
  reportByDay: ReportDayRow[];
  filters: { from: string; to: string };
};

export default function ReportsIndex() {
  const { activityByDay, reportByDay, filters } = usePage<PageProps>().props;

  // ──────────────────────────────────────────────────────────
  // react-activity-calendar data preparation
  // It requires: { date, count, level } with level in [0..4]
  // ──────────────────────────────────────────────────────────
  const calendarData: Activity[] = useMemo(() => {
    const rows = activityByDay || [];
    const max = rows.reduce((m, r) => Math.max(m, r.activity || 0), 0);
    const toLevel = (count: number) => {
      if (!count || max === 0) return 0;
      // Map linearly to 1..4, leaving 0 for "no activity"
      const lvl = Math.ceil((count / max) * 4);
      return Math.min(4, Math.max(1, lvl));
    };
    return rows.map((d) => ({
      date: d.day,          // yyyy-MM-dd
      count: d.activity,    // numeric count (not strictly required for theming, but useful for tooltip)
      level: toLevel(d.activity),
    }));
  }, [activityByDay]);

  // Custom color theme (5 swatches: [level0, level1, level2, level3, level4])
  const theme = {
    light: ['#ebedf0', '#9be9a8', '#40c463', '#30a14e', '#216e39'],
    dark:  ['#161b22', '#0e4429', '#006d32', '#26a641', '#39d353'],
  };

  // ──────────────────────────────────────────────────────────
  // Date range controls
  // ──────────────────────────────────────────────────────────
  const [from, setFrom] = useState<Date | undefined>(filters?.from ? new Date(filters.from) : undefined);
  const [to, setTo] = useState<Date | undefined>(filters?.to ? new Date(filters.to) : undefined);
  const [open, setOpen] = useState(false);

  const formattedRange = useMemo(() => {
    if (from && to) return `${moment(from).format('DD MMM YYYY')} — ${moment(to).format('DD MMM YYYY')}`;
    if (from) return `${moment(from).format('DD MMM YYYY')} — …`;
    return 'Pick a date range';
  }, [from, to]);

  const applyRange = () => {
    const qs: Record<string, string> = {};
    if (from) qs.from = moment(from).format('YYYY-MM-DD');
    if (to) qs.to = moment(to).format('YYYY-MM-DD');
    router.get(route('medicals.reports.index'), qs, { preserveState: true, replace: true, preserveScroll: true });
    setOpen(false);
  };

  // ──────────────────────────────────────────────────────────
  // Client-side pagination for the day table
  // ──────────────────────────────────────────────────────────
  const pageSize = 8;
  const [page, setPage] = useState(0);
  const pageCount = Math.max(1, Math.ceil((reportByDay?.length ?? 0) / pageSize));
  const pagedRows = useMemo(
    () => (reportByDay || []).slice(page * pageSize, page * pageSize + pageSize),
    [reportByDay, page]
  );

  const nextPage = () => setPage((p) => Math.min(pageCount - 1, p + 1));
  const prevPage = () => setPage((p) => Math.max(0, p - 1));

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.reports.index') },
        { title: 'Reports', href: route('medicals.reports.index') },
      ]}
    >
      <Head title="Medicals • Reports" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        {/* Top controls */}
        <div className="flex flex-col items-start justify-between gap-3 md:flex-row md:items-center">
          <h1 className="text-2xl font-bold">Reports</h1>

          <div className="flex items-center gap-2">
            <Popover open={open} onOpenChange={setOpen}>
              <PopoverTrigger asChild>
                <Button variant="outline" className="inline-flex items-center gap-2">
                  <CalendarIcon className="h-4 w-4" />
                  {formattedRange}
                </Button>
              </PopoverTrigger>
              <PopoverContent align="end" className="w-auto p-2">
                <Calendar
                  mode="range"
                  numberOfMonths={2}
                  selected={{ from, to }}
                  onSelect={(range: any) => {
                    setFrom(range?.from);
                    setTo(range?.to);
                  }}
                  initialFocus
                />
                <div className="mt-2 flex justify-end gap-2">
                  <Button
                    variant="ghost"
                    onClick={() => {
                      setFrom(undefined);
                      setTo(undefined);
                    }}
                  >
                    Clear
                  </Button>
                  <Button onClick={applyRange}>Apply</Button>
                </div>
              </PopoverContent>
            </Popover>

            <Button
              variant="outline"
              onClick={() => {
                router.get(route('medicals.reports.index'), {}, { preserveState: true, replace: true, preserveScroll: true });
              }}
            >
              This Year
            </Button>
          </div>
        </div>

        {/* Heatmap (react-activity-calendar) */}
        <Card>
          <CardHeader>
            <CardTitle>Activity by Day</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <ActivityCalendar
                data={calendarData}
                theme={theme}
                // keep default maxLevel = 4; we've already mapped to [0..4]
                labels={{
                  legend: { less: 'Less', more: 'More' },
                  months: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                  totalCount: '{{count}} activities in selected period',
                  tooltip: '<strong>{{count}} activities</strong> on {{date}}',
                  weekdays: ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
                }}
              />
            </div>
          </CardContent>
        </Card>

        {/* Reports Table */}
        <Card>
          <CardHeader>
            <CardTitle>Daily Medical Report</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto rounded-lg border">
              <Table className="min-w-[980px]">
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-center">No.</TableHead>
                    <TableHead>Date of Entry</TableHead>
                    <TableHead className="text-center">Pneumoconiosis</TableHead>
                    <TableHead className="text-center">Food Handler (COH)</TableHead>
                    <TableHead className="text-center">Pre-Employment</TableHead>
                    <TableHead className="text-center">Exit-Employment</TableHead>
                    <TableHead className="text-center">Exit-Pneumoconiosis</TableHead>
                    <TableHead className="text-right">Action</TableHead>
                  </TableRow>
                </TableHeader>

                <TableBody>
                  {pagedRows.map((row, idx) => {
                    const index = page * pageSize + idx + 1;

                    const pneumo = row['Pneumoconiosis']?.length ?? 0;
                    const coh = row['Food Handler (COH)']?.length ?? 0;
                    const pre = row['Pre-Employement']?.length ?? 0;
                    const exitE = row['Exit-Employement']?.length ?? 0;
                    const exitP = row['Exit-Pneumoconiosis']?.length ?? 0;

                    return (
                      <TableRow key={row.day} className="align-top">
                        <TableCell className="text-center font-medium">{index}</TableCell>

                        <TableCell>
                          <div className="flex flex-col">
                            <span className="text-primary font-semibold uppercase">
                              {moment(row.day).format('dddd, D MMMM YYYY')}
                            </span>
                            <span className="text-muted-foreground text-xs mt-1">
                              Total Patients:{' '}
                              <Badge variant="secondary">{row.patients_count ?? row.certificates_count}</Badge>
                            </span>
                          </div>
                        </TableCell>

                        <TableCell className="text-center">
                          <Badge variant="outline">{pneumo} Patients</Badge>
                        </TableCell>
                        <TableCell className="text-center">
                          <Badge variant="outline">{coh} Patients</Badge>
                        </TableCell>
                        <TableCell className="text-center">
                          <Badge variant="outline">{pre} Patients</Badge>
                        </TableCell>
                        <TableCell className="text-center">
                          <Badge variant="outline">{exitE} Patients</Badge>
                        </TableCell>
                        <TableCell className="text-center">
                          <Badge variant="outline">{exitP} Patients</Badge>
                        </TableCell>

                        <TableCell className="text-right">
                          {/* If you have a dedicated day route, prefer that.
                             Otherwise this keeps your existing querystring approach. */}
                          <Button asChild size="sm">
                            <Link href={route('medicals.reports.day', { day: row.day })}>
                              View Report
                            </Link>
                          </Button>
                        </TableCell>
                      </TableRow>
                    );
                  })}

                  {pagedRows.length === 0 && (
                    <TableRow>
                      <TableCell colSpan={8} className="py-8 text-center text-sm text-muted-foreground">
                        No data for the selected range.
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>

            {/* Pagination */}
            <div className="mt-4 flex items-center justify-between">
              <div className="text-sm text-muted-foreground">
                Page <span className="font-medium">{page + 1}</span> of <span className="font-medium">{pageCount}</span>
              </div>
              <div className="flex gap-2">
                <Button variant="outline" size="sm" onClick={prevPage} disabled={page === 0}>
                  <ChevronLeft className="mr-1 h-4 w-4" /> Prev
                </Button>
                <Button variant="outline" size="sm" onClick={nextPage} disabled={page >= pageCount - 1}>
                  Next <ChevronRight className="ml-1 h-4 w-4" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
