import AppLayout from '@/layouts/app-layout';
import { Head, Link, usePage } from '@inertiajs/react';
import React, { useMemo, useState } from 'react';
import axios from 'axios';
import moment from 'moment';

// shadcn/ui
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { Popover, PopoverTrigger, PopoverContent } from '@/components/ui/popover';
import { Calendar } from '@/components/ui/calendar';
import {
  Select, SelectTrigger, SelectContent, SelectValue, SelectItem,
} from '@/components/ui/select';
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuCheckboxItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
} from '@/components/ui/dropdown-menu';
import { Calendar as CalendarIcon, Loader2, Eye, Pencil, Settings2 } from 'lucide-react';

type Company = { id: number; company_name: string };

type LatestCert = {
  id?: number;
  patient_id?: number;
  status?: string | null;
  certificate_location?: string | null;
  created_at?: string | null;
} | null;

type Attendee = {
  id?: number;
  first_name?: string;
  last_name?: string;
  date_of_birth?: string | null;
  gender?: string | null;
  national_id?: string | null;
  phone_number?: string | null;
  time_of_entry?: string | null;
  company?: { id: number; company_name: string } | null;
};

type Swab = {
  id: number;
  patient_id?: number;
  status: string;
  file?: string | null;
  comment?: string | null;
  created_at?: string | null;
  updated_at?: string | null;
};

type PatientRow = {
  id: number;
  created_at?: string | null;
  category?: string | null;
  exam_purpose?: string | null;
  attendee: Attendee;
  company?: { id: number; company_name: string } | null;
  latest_certificate?: LatestCert;
  certificates_count?: number;
  swabs?: Swab[];
};

type PageProps = {
  companies: Company[];
  filters?: { start_date?: string; end_date?: string; company?: number | string | null };
};

type ColumnKey =
  | 'no'
  | 'date_time'
  | 'first_name'
  | 'last_name'
  | 'company'
  | 'category'
  | 'latest_certificate'
  | 'swab_status'
  | 'phone'
  | 'national_id'
  | 'actions';

const DEFAULT_VISIBLE: Record<ColumnKey, boolean> = {
  no: true,
  date_time: true, // Certificate Date / Time
  first_name: true,
  last_name: true,
  company: true,
  category: true,
  latest_certificate: true,
  swab_status: true,
  phone: true,
  national_id: true,
  actions: true,
};

export default function MedicalsSwabsIndex() {
  const { companies = [], filters } = usePage<PageProps>().props;

  // filters
  const [startDate, setStartDate] = useState<Date | undefined>(
    filters?.start_date ? new Date(filters.start_date) : undefined
  );
  const [endDate, setEndDate] = useState<Date | undefined>(
    filters?.end_date ? new Date(filters.end_date) : undefined
  );
  const [company, setCompany] = useState<string>(filters?.company ? String(filters.company) : '');

  // results
  const [loading, setLoading] = useState(false);
  const [rows, setRows] = useState<PatientRow[]>([]);
  const [error, setError] = useState<string | null>(null);

  // column visibility (persist to localStorage)
  const [visibleCols, setVisibleCols] = useState<Record<ColumnKey, boolean>>(() => {
    if (typeof window === 'undefined') return DEFAULT_VISIBLE;
    try {
      const raw = localStorage.getItem('medicals.swabs.columns.v1');
      if (!raw) return DEFAULT_VISIBLE;
      const parsed = JSON.parse(raw);
      return { ...DEFAULT_VISIBLE, ...parsed };
    } catch {
      return DEFAULT_VISIBLE;
    }
  });

  const updateCol = (key: ColumnKey, val: boolean) => {
    setVisibleCols(prev => {
      const next = { ...prev, [key]: val };
      try {
        localStorage.setItem('medicals.swabs.columns.v1', JSON.stringify(next));
      } catch {}
      return next;
    });
  };

  const startStr = startDate ? moment(startDate).format('YYYY-MM-DD') : '';
  const endStr = endDate ? moment(endDate).format('YYYY-MM-DD') : '';

  const latestSwabStatus = (swabs?: Swab[]) => {
    if (!swabs || swabs.length === 0) return '-';
    const sorted = [...swabs].sort((a, b) =>
      new Date(b.created_at || '').getTime() - new Date(a.created_at || '').getTime()
    );
    return sorted[0]?.status ?? '-';
  };

  const handleFetch = async () => {
    setLoading(true);
    setError(null);
    try {
      const payload: Record<string, any> = {};
      if (startStr) payload.start_date = startStr;
      if (endStr) payload.end_date = endStr;
      if (company) payload.company = Number(company);

      // Make sure route name matches your backend
      const { data } = await axios.post(route('medicals.swabs.filter'), payload);

      // backend returns a plain array; fallback to {data: …} if any wrapper
      const list: PatientRow[] = Array.isArray(data) ? data : (data?.data ?? []);
      setRows(list);
    } catch (e: any) {
      const msg = e?.response?.data?.message || 'Failed to fetch results.';
      setError(msg);
      setRows([]);
    } finally {
      setLoading(false);
    }
  };

  const total = rows.length;

  // Column helpers
  const columnsOrder: ColumnKey[] = useMemo(
    () => [
      'no',
      'date_time', // Certificate Date / Time (from latest_certificate.created_at)
      'first_name',
      'last_name',
      'company',
      'category',
      'latest_certificate',
      'swab_status',
      'phone',
      'national_id',
      'actions',
    ],
    []
  );

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.swabs.index') },
        { title: 'Daily Swab Register', href: route('medicals.swabs.index') },
      ]}
    >
      <Head title="Medicals • Daily Swab Register" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        <div className="flex flex-col items-start justify-between gap-3 md:flex-row md:items-center">
          <h1 className="text-2xl font-bold">Daily Swab Register</h1>

          {/* Column chooser */}
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="outline" className="inline-flex items-center gap-2">
                <Settings2 className="h-4 w-4" />
                Columns
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="w-56">
              <DropdownMenuLabel>Toggle columns</DropdownMenuLabel>
              <DropdownMenuSeparator />
              {columnsOrder.map((key) => {
                // lock "no" and "actions" from being disabled if you prefer
                const lock = key === 'no' || key === 'actions';
                const label: Record<ColumnKey, string> = {
                  no: 'No.',
                  date_time: 'Certificate Date / Time',
                  first_name: 'First Name',
                  last_name: 'Last Name',
                  company: 'Company',
                  category: 'Category',
                  latest_certificate: 'Latest Certificate',
                  swab_status: 'Swab Status',
                  phone: 'Phone',
                  national_id: 'National ID',
                  actions: 'Actions',
                };
                return (
                  <DropdownMenuCheckboxItem
                    key={key}
                    checked={visibleCols[key]}
                    onCheckedChange={(v) => !lock && updateCol(key, Boolean(v))}
                    disabled={lock}
                  >
                    {label[key]}
                  </DropdownMenuCheckboxItem>
                );
              })}
            </DropdownMenuContent>
          </DropdownMenu>
        </div>

        {/* Filters */}
        <Card>
          <CardHeader>
            <CardTitle>Filter</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 gap-4 md:grid-cols-5">
              {/* Start Date */}
              <div>
                <label className="mb-1 block text-sm font-medium">Start Date</label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button variant="outline" className="w-full justify-start">
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {startDate ? moment(startDate).format('DD MMM YYYY') : 'Pick date'}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-2" align="start">
                    <Calendar
                      mode="single"
                      selected={startDate}
                      onSelect={(d: Date | undefined) => setStartDate(d)}
                      initialFocus
                    />
                  </PopoverContent>
                </Popover>
              </div>

              {/* End Date */}
              <div>
                <label className="mb-1 block text-sm font-medium">End Date</label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button variant="outline" className="w-full justify-start">
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {endDate ? moment(endDate).format('DD MMM YYYY') : 'Pick date'}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-2" align="start">
                    <Calendar
                      mode="single"
                      selected={endDate}
                      onSelect={(d: Date | undefined) => setEndDate(d)}
                      initialFocus
                    />
                  </PopoverContent>
                </Popover>
              </div>

              {/* Company */}
              <div className="md:col-span-1">
                <label className="mb-1 block text-sm font-medium">Company</label>
                <Select value={company || 'ALL'} onValueChange={(v) => setCompany(v === 'ALL' ? '' : v)}>
                  <SelectTrigger className="w-full">
                    <SelectValue placeholder="All Companies" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ALL">All Companies</SelectItem>
                    {companies.map((c) => (
                      <SelectItem key={c.id} value={String(c.id)}>
                        {c.company_name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Actions */}
              <div className="flex items-end gap-2">
                <Button className="w-full" onClick={handleFetch} disabled={loading}>
                  {loading ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" /> Loading…
                    </>
                  ) : (
                    'Fetch'
                  )}
                </Button>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => {
                    setStartDate(undefined);
                    setEndDate(undefined);
                    setCompany('');
                    setRows([]);
                    setError(null);
                  }}
                >
                  Reset
                </Button>
              </div>
            </div>

            {error && (
              <div className="mt-3 rounded-md border border-destructive bg-red-50 px-3 py-2 text-sm text-red-700">
                {error}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Results */}
        <Card>
          <CardHeader>
            <CardTitle>
              Results{' '}
              <Badge variant="secondary" className="ml-2">
                {total}
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto rounded-lg border">
              <Table className="min-w-[1100px]">
                <TableHeader>
                  <TableRow>
                    {visibleCols.no && <TableHead>No.</TableHead>}
                    {visibleCols.date_time && <TableHead>Certificate Date / Time</TableHead>}
                    {visibleCols.first_name && <TableHead>First Name</TableHead>}
                    {visibleCols.last_name && <TableHead>Last Name</TableHead>}
                    {visibleCols.company && <TableHead>Company</TableHead>}
                    {visibleCols.category && <TableHead>Category</TableHead>}
                    {visibleCols.latest_certificate && <TableHead>Latest Certificate</TableHead>}
                    {visibleCols.swab_status && <TableHead>Swab Status</TableHead>}
                    {visibleCols.phone && <TableHead>Phone</TableHead>}
                    {visibleCols.national_id && <TableHead>National ID</TableHead>}
                    {visibleCols.actions && <TableHead className="text-right">Actions</TableHead>}
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {rows.map((p, idx) => {
                    const a = p.attendee || {};

                    // Use latest certificate created_at for Date / Time
                    const dateSource =
                      p.latest_certificate?.created_at ||
                      p.created_at ||
                      a.time_of_entry ||
                      null;

                    const dateStr = dateSource ? moment(dateSource).format('YYYY-MM-DD') : '-';
                    const timeStr = dateSource ? moment(dateSource).format('HH:mm') : '-';

                    const certDate = p.latest_certificate?.created_at
                      ? moment(p.latest_certificate.created_at).format('YYYY-MM-DD')
                      : '-';
                    const certStatus = p.latest_certificate?.status ?? '-';

                    return (
                      <TableRow key={p.id}>
                        {visibleCols.no && <TableCell>{idx + 1}</TableCell>}

                        {visibleCols.date_time && (
                          <TableCell>
                            <div className="flex flex-col">
                              <span className="font-medium">{dateStr}</span>
                              <span className="text-xs text-muted-foreground">{timeStr}</span>
                            </div>
                          </TableCell>
                        )}

                        {visibleCols.first_name && <TableCell>{a.first_name ?? '-'}</TableCell>}
                        {visibleCols.last_name && <TableCell>{a.last_name ?? '-'}</TableCell>}
                        {visibleCols.company && (
                          <TableCell>{a.company?.company_name ?? p.company?.company_name ?? '-'}</TableCell>
                        )}
                        {visibleCols.category && <TableCell>{p.category ?? '-'}</TableCell>}

                        {visibleCols.latest_certificate && (
                          <TableCell>
                            <div className="flex flex-col">
                              <span className="text-xs text-muted-foreground">{certDate}</span>
                              <Badge variant="outline">{certStatus}</Badge>
                            </div>
                          </TableCell>
                        )}

                        {visibleCols.swab_status && (
                          <TableCell>
                            <Badge variant="outline">{latestSwabStatus(p.swabs)}</Badge>
                          </TableCell>
                        )}

                        {visibleCols.phone && <TableCell>{a.phone_number ?? '-'}</TableCell>}
                        {visibleCols.national_id && <TableCell>{a.national_id ?? '-'}</TableCell>}

                        {visibleCols.actions && (
                          <TableCell className="text-right">
                            <div className="inline-flex items-center gap-2">
                              {/* Update these routes if your names differ */}
                              <Button asChild size="sm" variant="outline" title="View patient">
                                <Link href={route('medicals.patients.show', p.id)}>
                                  <Eye className="h-4 w-4" />
                                </Link>
                              </Button>
                              <Button asChild size="sm" variant="outline" title="Edit patient">
                                <Link href={route('medicals.attendees.edit', p.id)}>
                                  <Pencil className="h-4 w-4" />
                                </Link>
                              </Button>
                            </div>
                          </TableCell>
                        )}
                      </TableRow>
                    );
                  })}
                  {rows.length === 0 && (
                    <TableRow>
                      <TableCell
                        colSpan={
                          Object.values(visibleCols).filter(Boolean).length || 1
                        }
                        className="py-8 text-center text-sm text-muted-foreground"
                      >
                        No results. Choose a filter and click Fetch.
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
