import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useRef, useState } from 'react';
import ReactPaginate from 'react-paginate';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableHeader, TableHead, TableRow, TableBody, TableCell } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Plus, Search as SearchIcon, Pencil, Trash2 } from 'lucide-react';
import moment from 'moment';

type Condition = {
  id: number;
  name: string;
  created_at?: string | null;
  updated_at?: string | null;
};

type PageProps = {
  conditions: {
    data: Condition[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search?: string;
    per_page?: number;
    sort?: 'name' | 'id' | 'created_at' | 'updated_at';
    direction?: 'asc' | 'desc';
  };
};

function useDebounced<T>(val: T, delay = 450) {
  const [v, setV] = useState(val);
  useEffect(() => {
    const id = setTimeout(() => setV(val), delay);
    return () => clearTimeout(id);
  }, [val, delay]);
  return v;
}

// Thursday 3 November 2025
function fmtNice(d?: string | null) {
  if (!d) return '—';
  const m = moment(d);
  return m.isValid() ? m.format('dddd D MMMM YYYY') : '—';
}

export default function TobaccoUsesIndex() {
  const { conditions, filters } = usePage<PageProps>().props;

  const [search, setSearch] = useState(filters.search ?? '');
  const [perPage, setPerPage] = useState(String(filters.per_page ?? 20));
  const [sort, setSort] = useState<'name' | 'id' | 'created_at' | 'updated_at'>(filters.sort ?? 'name');
  const [direction, setDirection] = useState<'asc' | 'desc'>(filters.direction ?? 'asc');

  const debouncedSearch = useDebounced(search, 450);
  const firstRun = useRef(true);

  const apply = (page = 1, override?: Partial<Record<string, string | number>>) => {
    router.get(
      route('medicals.tobacco-uses.index'),
      {
        page,
        search: override?.search ?? search,
        per_page: override?.per_page ?? perPage,
        sort: override?.sort ?? sort,
        direction: override?.direction ?? direction,
      },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    apply(1, { search: debouncedSearch });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  const handleDelete = (c: Condition) => {
    router.delete(route('medicals.tobacco-uses.destroy', c.id), {
      preserveScroll: true,
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.dashboard') },
        { title: 'Tobacco Uses', href: route('medicals.tobacco-uses.index') },
      ]}
    >
      <Head title="Tobacco Uses" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        {/* Header */}
        <div className="flex flex-col items-start justify-between gap-3 md:flex-row md:items-center">
          <h1 className="text-2xl font-bold">Tobacco Uses</h1>
          <div className="flex gap-2">
            <Button asChild>
              <Link href={route('medicals.tobacco-uses.create')}>
                <Plus className="mr-2 h-4 w-4" />
                New Tobacco Use
              </Link>
            </Button>
          </div>
        </div>

        {/* Filters */}
        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent className="grid grid-cols-1 gap-3 md:grid-cols-2 lg:grid-cols-6">
            {/* Big search */}
            <div className="col-span-full relative">
              <SearchIcon className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
              <Input
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && apply(1)}
                placeholder="Search by name…"
                className="h-12 pl-10"
              />
            </div>

            <div>
              <div className="mb-1 text-sm text-muted-foreground">Sort</div>
              <Select
                value={sort}
                onValueChange={(v: 'name' | 'id' | 'created_at' | 'updated_at') => {
                  setSort(v);
                  apply(1, { sort: v });
                }}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="name">Name</SelectItem>
                  <SelectItem value="id">ID</SelectItem>
                  <SelectItem value="created_at">Created</SelectItem>
                  <SelectItem value="updated_at">Updated</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <div className="mb-1 text-sm text-muted-foreground">Direction</div>
              <Select
                value={direction}
                onValueChange={(v: 'asc' | 'desc') => {
                  setDirection(v);
                  apply(1, { direction: v });
                }}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="asc">Ascending</SelectItem>
                  <SelectItem value="desc">Descending</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <div className="mb-1 text-sm text-muted-foreground">Per Page</div>
              <Select
                value={perPage}
                onValueChange={(v) => {
                  setPerPage(v);
                  apply(1, { per_page: v });
                }}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="10">10</SelectItem>
                  <SelectItem value="20">20</SelectItem>
                  <SelectItem value="50">50</SelectItem>
                  <SelectItem value="100">100</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        </Card>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-[780px]">
            <TableHeader>
              <TableRow>
                <TableHead>ID</TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Created</TableHead>
                <TableHead>Updated</TableHead>
                <TableHead className="text-right">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {conditions.data.map((c) => (
                <TableRow key={c.id}>
                  <TableCell>{c.id}</TableCell>
                  <TableCell>
                    <Badge variant="secondary">{c.name}</Badge>
                  </TableCell>
                  <TableCell>{fmtNice(c.created_at)}</TableCell>
                  <TableCell>{fmtNice(c.updated_at)}</TableCell>
                  <TableCell className="text-right">
                    <div className="flex justify-end gap-2">
                      <Button asChild size="icon" variant="secondary" title="Edit">
                        <Link href={route('medicals.tobacco-uses.edit', c.id)}>
                          <Pencil className="h-4 w-4" />
                        </Link>
                      </Button>
                      <Button size="icon" variant="destructive" onClick={() => handleDelete(c)} title="Delete">
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))}

              {conditions.data.length === 0 && (
                <TableRow>
                  <TableCell colSpan={5} className="py-8 text-center text-sm text-muted-foreground">
                    No items found.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={conditions.last_page}
            forcePage={conditions.current_page - 1}
            onPageChange={(p) => apply(p.selected + 1)}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="…"
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="rounded border px-3 py-1 hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="rounded border px-3 py-1 hover:bg-muted"
            nextClassName="rounded border px-3 py-1 hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
