import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Plus, Search, Trash } from 'lucide-react';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Medications', href: '/medications' }];

interface Medication {
    id: number;
    name: string;
    dosage: string | null;
    form: string | null;
    unit: string | null;
    instructions: string | null;
}

interface PagePropsWithMedications extends PageProps {
    medications: {
        data: Medication[];
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    filters: {
        search?: string;
    };
}

export default function MedicationList() {
    const { medications, filters } = usePage<PagePropsWithMedications>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');

    const handleSearch = () => {
        router.get(`${API}/medications`, { search: searchQuery }, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(`${API}/medications`, { page: selected.selected + 1, search: searchQuery }, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This medication will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('medications.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The medication has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Medications" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Medications</h1>

                    <div className="flex flex-wrap items-center gap-3">
                        <Link
                            href={route('central-stores.receive-batch.form', 1)}
                            className="inline-flex items-center gap-2 rounded-lg bg-yellow-600 px-5 py-2 text-sm font-medium text-white hover:bg-yellow-700"
                        >
                            <Plus className="h-4 w-4" />
                            Received Medication Batches
                        </Link>
                        <Link
                            href={route('medications.create')}
                            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700"
                        >
                            <Plus className="h-4 w-4" />
                            Add Medication
                        </Link>
                    </div>

                </div>

                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            placeholder="Search medications..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                    <button onClick={handleSearch} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                        Search
                    </button>
                </div>

                <div className="overflow-x-auto rounded-xl border">
                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {['Name', 'Dosage', 'Form', 'Unit', 'Instructions', 'Actions'].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {medications.data.map((medication) => (
                                <tr key={medication.id} className="hover:bg-muted/50 transition">
                                    <td className="px-4 py-3 font-medium">{medication.name}</td>
                                    <td className="px-4 py-3">{medication.dosage || '-'}</td>
                                    <td className="px-4 py-3">{medication.form || '-'}</td>
                                    <td className="px-4 py-3">{medication.unit || '-'}</td>
                                    <td className="px-4 py-3">{medication.instructions || '-'}</td>
                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <Link
                                            href={route('medications.show', medication.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                                            title="View"
                                        >
                                            <Eye className="h-4 w-4" />
                                        </Link>
                                        <Link
                                            href={route('medications.edit', medication.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                            title="Edit"
                                        >
                                            <Pencil className="h-4 w-4" />
                                        </Link>
                                        <button
                                            onClick={() => handleDelete(medication.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                            title="Delete"
                                        >
                                            <Trash className="h-4 w-4" />
                                        </button>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={medications.last_page}
                        forcePage={medications.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}
