// resources/js/Pages/Medications/MedicationDispense.tsx
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { ClinicMedicationStock, Consultation } from '@/types/medication';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import { Pill, Plus, Search, Info as InfoIcon, Trash2 } from 'lucide-react';
import moment from 'moment';
import { useEffect, useMemo, useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import Select, { MultiValue, SingleValue, StylesConfig, GroupBase } from 'react-select';
import { Modal, Form, Select as AntdSelect, InputNumber, Input, Radio, message } from 'antd';

type MaybePaginated<T> =
  | T[]
  | {
    data: T[];
    current_page: number;
    last_page: number;
    first_page_url: string;
    last_page_url: string;
    links: Array<{ url: string | null; label: string; active: boolean }>;
    next_page_url: string | null;
    path: string;
    per_page: number;
    prev_page_url: string | null;
    to: number;
    total: number;
  };

interface Props extends PageProps {
  consultation: Consultation;
  clinicMedicationStocks: MaybePaginated<ClinicMedicationStock>;
  nurseMedicationStocks: MaybePaginated<ClinicMedicationStock>;
  filters: { search?: string };
  dispensations: any[];
  frequencyOptions: Record<string, string>;
  routeOptions: Record<string, string>;
}

/** ===== theme for react-select ===== */
function useThemeTokens() {
  const read = (name: string, fallback: string) =>
    typeof window !== 'undefined'
      ? getComputedStyle(document.documentElement).getPropertyValue(name).trim() || fallback
      : fallback;

  const bg = read('--background', '#fff');
  const fg = read('--foreground', '#0a0a0a');
  const border = read('--border', '220 13% 91%');
  const muted = read('--muted', '220 14% 96%');
  const ring = read('--ring', '221 83% 53%');

  const hsl = (triple: string) => (triple.includes('%') ? `hsl(${triple})` : triple);
  return { bg, fg, border: hsl(border), muted: hsl(muted), ring: hsl(ring) };
}
function makeSelectStyles<T, M extends boolean = false, G extends GroupBase<T> = GroupBase<T>>(
  tokens: ReturnType<typeof useThemeTokens>
): StylesConfig<T, M, G> {
  return {
    container: (base) => ({ ...base, width: '100%' }),
    control: (base, state) => ({
      ...base,
      minHeight: 36,
      borderRadius: 8,
      backgroundColor: tokens.bg,
      borderColor: state.isFocused ? tokens.ring : tokens.border,
      boxShadow: state.isFocused ? `0 0 0 2px ${tokens.ring}22` : 'none',
      ':hover': { borderColor: state.isFocused ? tokens.ring : tokens.border },
    }),
    valueContainer: (base) => ({ ...base, paddingBlock: 2, paddingInline: 8 }),
    input: (base) => ({ ...base, color: tokens.fg }),
    singleValue: (base) => ({ ...base, color: tokens.fg }),
    placeholder: (base) => ({ ...base, color: `${tokens.fg}99` }),
    menu: (base) => ({
      ...base,
      zIndex: 60,
      backgroundColor: tokens.bg,
      border: `1px solid ${tokens.border}`,
      boxShadow: `0 8px 24px rgba(0,0,0,.12)`,
      overflow: 'hidden',
      borderRadius: 8,
    }),
    menuList: (base) => ({ ...base, paddingBlock: 4 }),
    option: (base, state) => ({
      ...base,
      backgroundColor: state.isSelected ? `${tokens.ring}22` : state.isFocused ? `${tokens.muted}` : 'transparent',
      color: tokens.fg,
      cursor: 'pointer',
    }),
  };
}

function isPaginated<T>(v: MaybePaginated<T>): v is Extract<MaybePaginated<T>, { data: T[] }> {
  return !!v && typeof v === 'object' && 'data' in v && Array.isArray((v as any).data);
}

/** ===== form row (no medication select here) ===== */
type RowFields = {
  quantity: string;
  frequency: string;
  route: string;
  duration: string;
  notes: string;
};

/** Utility */
const uuid = () =>
  (crypto as any)?.randomUUID ? (crypto as any).randomUUID() : `${Date.now()}-${Math.random().toString(16).slice(2)}`;

const PAGE_SIZE_ITEMS = 10;

export default function MedicationDispense() {
  const {
    consultation,
    clinicMedicationStocks,
    nurseMedicationStocks,
    filters,
    dispensations,
    frequencyOptions,
    routeOptions,
  } = usePage<Props>().props;

  const tokens = useThemeTokens();
  const selectStyles = makeSelectStyles(tokens);

  const [searchQuery, setSearchQuery] = useState(filters.search || '');

  // Stocks
  const clinicStocks = useMemo<ClinicMedicationStock[]>(
    () => (isPaginated(clinicMedicationStocks) ? clinicMedicationStocks.data : clinicMedicationStocks),
    [clinicMedicationStocks]
  );
  const nurseStocks = useMemo<ClinicMedicationStock[]>(
    () => (isPaginated(nurseMedicationStocks) ? nurseMedicationStocks.data : nurseMedicationStocks),
    [nurseMedicationStocks]
  );

  // Global stock source (single dropdown)
  const stockTypeOptions = [
    { value: 'clinic', label: 'Clinic Stock' },
    { value: 'nurse', label: 'Nurse Stock' },
  ];
  const [globalStockType, setGlobalStockType] = useState<'clinic' | 'nurse'>('nurse');
  const currentStocks = globalStockType === 'clinic' ? clinicStocks : nurseStocks;

  // Single (multi-select) medication dropdown — drives rows
  type MedOpt = { value: string; label: string };
  const medicationOptions: MedOpt[] = useMemo(
    () =>
      currentStocks.map((s) => ({
        value: String(s.id),
        label: `${s.medication_batch.medication.name} (${s.quantity} avail) — ${s.medication_batch.medication.dosage ?? ''}${s.medication_batch.medication.unit ?? ''}`,
      })),
    [currentStocks]
  );
  const [selectedMeds, setSelectedMeds] = useState<MedOpt[]>([]);

  // Row fields keyed by stock_id
  const [rowsByStockId, setRowsByStockId] = useState<Record<string, RowFields>>({});

  // When changing stock source, clear selection and rows
  const onChangeGlobalStockType = (opt: SingleValue<{ value: string; label: string }>) => {
    const value = (opt?.value || 'nurse') as 'clinic' | 'nurse';

    setGlobalStockType(value);
    setSelectedMeds([]);
    setRowsByStockId({});
  };

  // Sync rows state with selected meds
  useEffect(() => {
    setRowsByStockId((prev) => {
      const next: Record<string, RowFields> = {};
      selectedMeds.forEach((m) => {
        next[m.value] = prev[m.value] || {
          quantity: '',
          frequency: '',
          route: '',
          duration: '',
          notes: '',
        };
      });
      return next;
    });
  }, [selectedMeds]);

  const updateRow = (stockId: string, patch: Partial<RowFields>) =>
    setRowsByStockId((prev) => ({ ...prev, [stockId]: { ...prev[stockId], ...patch } }));

  const removeSelectedMed = (stockId: string) =>
    setSelectedMeds((prev) => prev.filter((m) => m.value !== stockId));

  const findStock = (stock_id: string): ClinicMedicationStock | undefined =>
    currentStocks.find((s) => String(s.id) === stock_id);

  /** ===== Search & paging ===== */
  const handleSearch = () => {
    router.get(
      route('consultations.dispense', consultation.id),
      { search: searchQuery },
      {
        preserveState: true,
        preserveScroll: true,
        replace: true,
        only: ['clinicMedicationStocks', 'nurseMedicationStocks', 'filters'],
      }
    );
  };
  const handleResetSearch = () => {
    setSearchQuery('');
    router.get(route('consultations.dispense', consultation.id), {}, {
      preserveState: true,
      preserveScroll: true,
      replace: true,
      only: ['clinicMedicationStocks', 'nurseMedicationStocks', 'filters'],
    });
  };
  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      route('consultations.dispense', consultation.id),
      { page: selected.selected + 1, search: searchQuery },
      {
        preserveState: true,
        preserveScroll: true,
        replace: true,
        only: ['clinicMedicationStocks', 'nurseMedicationStocks', 'filters'],
      }
    );
  };

  const showClinicPagination = isPaginated(clinicMedicationStocks);
  const showNursePagination = isPaginated(nurseMedicationStocks);

  /** ===== Submit (batch) ===== */
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleSubmitBatch = async (e: React.FormEvent) => {
    e.preventDefault();
    if (isSubmitting) return;

    const items = selectedMeds
      .map((m) => {
        const f = rowsByStockId[m.value] || {};
        const quantity = f.quantity === '' ? 0 : parseInt(f.quantity, 10);
        const duration = f.duration === '' ? null : parseInt(f.duration, 10);

        return {
          stock_type: globalStockType,
          stock_id: Number(m.value),
          quantity,
          frequency: f.frequency || null,
          route: f.route || null,
          duration,
          notes: f.notes || null,
          request_id: uuid(),
        };
      })
      .filter((it) => it.stock_id && it.quantity && it.quantity > 0);

    if (items.length === 0) {
      return Swal.fire('Nothing to dispense', 'Pick medication(s) and enter valid quantity.', 'warning');
    }

    // Validate against current stock availability
    for (const it of items) {
      const stock = currentStocks.find((s) => s.id === it.stock_id);
      if (!stock) return Swal.fire('Invalid selection', 'One of the selected stocks no longer exists.', 'error');
      if (stock.quantity < (it.quantity ?? 0)) {
        return Swal.fire(
          'Insufficient stock',
          `Only ${stock.quantity} available for ${stock.medication_batch.medication.name}.`,
          'warning'
        );
      }
      if (it.duration !== null && it.duration !== undefined && it.duration <= 0) {
        return Swal.fire('Invalid duration', 'Duration must be greater than 0 days.', 'warning');
      }
    }

    setIsSubmitting(true);
    Swal.fire({
      title: 'Dispensing…',
      text: 'Please wait while we record all medications.',
      allowOutsideClick: false,
      allowEscapeKey: false,
      didOpen: () => Swal.showLoading(),
    });

    router.post(
      route('consultations.dispensations.store.batch', consultation.id),
      { patient_id: consultation.patient.id, items },
      {
        preserveScroll: true,
        onSuccess: async () => {
          Swal.close();
          await Swal.fire('Success', 'Medications dispensed successfully.', 'success');
          // Clear selection & rows
          setSelectedMeds([]);
          setRowsByStockId({});
          // Reload stocks & dispensed list
          router.reload({
            only: ['dispensations', 'clinicMedicationStocks', 'nurseMedicationStocks'],
            preserveScroll: true,
          });
        },
        onError: async (errors) => {
          Swal.close();
          const msg =
            (errors && typeof errors === 'object'
              ? Object.values(errors).join('\n')
              : 'Failed to dispense medications.') || 'Failed to dispense medications.';
          await Swal.fire('Error', msg, 'error');
        },
        onFinish: () => setIsSubmitting(false),
      }
    );
  };

  /** ===== Referrals + Close ===== */
  const handleGoToReferralsPage = async () => {
    const { isConfirmed } = await Swal.fire({
      title: 'Open Referrals & Follow-up',
      html: `
        <div style="text-align:left">
          <p>You'll be taken to a page to manage:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>Follow-up</b> (required + date)</li>
            <li><b>Referrals</b> (radiology, lab, specialist, casualty, outpatient)</li>
            <li><b>Sick leave</b> (days, reason, notes)</li>
            <li><b>Referral notes</b></li>
          </ul>
        </div>
      `,
      icon: 'info',
      showCancelButton: true,
      confirmButtonText: 'Proceed',
      cancelButtonText: 'Stay here',
      focusCancel: true,
    });
    if (isConfirmed) router.visit(route('consultations.referrals.edit', consultation.id));
  };

  const handleCloseConsultation = async (consultationId: number | string) => {
    const result = await Swal.fire({
      title: 'What would you like to do?',
      html: `
        <div style="text-align:left">
          <p>You can either:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>Close Consultation</b> — finalize and lock this consultation.</li>
            <li><b>Go to Referrals Page</b> — update follow-up, referrals, and sick leave before closing.</li>
          </ul>
        </div>
      `,
      icon: 'question',
      showDenyButton: true,
      showCancelButton: true,
      confirmButtonText: 'Close Consultation',
      denyButtonText: 'Go to Referrals Page',
      cancelButtonText: 'Cancel',
      reverseButtons: true,
    });

    if (result.isConfirmed) {
      Swal.fire({
        title: 'Closing Consultation...',
        text: 'Please wait',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      });

      router.post(route('consultations.close', consultationId), {}, {
        onSuccess: () => Swal.fire('Closed!', 'Consultation closed successfully.', 'success'),
        onError: () => Swal.fire('Error', 'Failed to close consultation.', 'error'),
      });
    } else if (result.isDenied) {
      await Swal.fire({
        title: 'Referrals & Follow-up',
        html: `
          <div style="text-align:left">
            <p>You’re heading to the Referrals page to manage:</p>
            <ul style="margin-left:1rem; list-style:disc;">
              <li>Follow-up requirements & date</li>
              <li>Radiology, Lab & Specialist referrals</li>
              <li>Casualty/Outpatient referrals</li>
              <li>Sick leave details & notes</li>
            </ul>
          </div>
        `,
        icon: 'info',
        confirmButtonText: 'Continue',
      });
      router.visit(route('consultations.referrals.edit', consultationId));
    }
  };

  /** ===== Correction Modal ===== */
  const [corrOpen, setCorrOpen] = useState(false);
  const [corrForm] = Form.useForm();
  const [corrDisp, setCorrDisp] = useState<any | null>(null);
  const [corrSubmitting, setCorrSubmitting] = useState(false);
  const [corrToType, setCorrToType] = useState<'clinic' | 'nurse'>('clinic');

  const clinicAntdOptions = useMemo(
    () =>
      clinicStocks.map((s) => ({
        value: s.id,
        label: `${s.medication_batch.medication.name} — batch ${s.medication_batch.batch_number || '-'} (${s.quantity} avail)`,
      })),
    [clinicStocks]
  );
  const nurseAntdOptions = useMemo(
    () =>
      nurseStocks.map((s) => ({
        value: s.id,
        label: `${s.medication_batch.medication.name} — batch ${s.medication_batch.batch_number || '-'} (${s.quantity} avail)`,
      })),
    [nurseStocks]
  );

  const openCorrectionModal = (d: any) => {
    setCorrDisp(d);
    const fromType: 'clinic' | 'nurse' = d.clinic_medication_stock ? 'clinic' : 'nurse';
    const defaultToType: 'clinic' | 'nurse' = fromType === 'clinic' ? 'nurse' : 'clinic';
    setCorrToType(defaultToType);

    corrForm.setFieldsValue({
      quantity: d.quantity || 1,
      to_stock_type: defaultToType,
      to_stock_id: undefined,
      reason: '',
    });
    setCorrOpen(true);
  };
  const closeCorrectionModal = () => {
    setCorrOpen(false);
    setCorrDisp(null);
    corrForm.resetFields();
  };
  const submitCorrection = (values: any) => {
    if (!corrDisp) return;

    const fromType: 'clinic' | 'nurse' = corrDisp.clinic_medication_stock ? 'clinic' : 'nurse';
    const fromId: number = fromType === 'clinic' ? corrDisp.clinic_medication_stock?.id : corrDisp.nurse_medication_stock?.id;

    if (values.to_stock_type === fromType && Number(values.to_stock_id) === Number(fromId)) {
      message.error('Please choose a different stock as the correction target.');
      return;
    }

    const payload = {
      to_stock_type: values.to_stock_type,
      to_stock_id: values.to_stock_id,
      quantity: values.quantity,
      reason: values.reason || '',
      request_uuid: uuid(),
    };

    setCorrSubmitting(true);

    router.post(
      route('dispensations.corrections.store', corrDisp.id),
      payload,
      {
        preserveScroll: true,
        onSuccess: () => {
          message.success('Correction request submitted. Awaiting approval.');
          closeCorrectionModal();
          router.reload({ only: ['dispensations'] });
        },
        onError: (errors) => {
          const msg = Object.values(errors || {}).join('\n') || 'Failed to submit correction.';
          message.error(msg);
        },
        onFinish: () => setCorrSubmitting(false),
      }
    );
  };

  /** ===== Delete & Return to Stock ===== */
  const handleDeleteAndReturn = async (d: any) => {
    // Determine source stock & current qty
    const fromClinic = !!d.clinic_medication_stock;
    const stockObj = fromClinic ? d.clinic_medication_stock : d.nurse_medication_stock;
    const stockTypeLabel = fromClinic ? 'Clinic' : 'Nurse';
    const medName =
      d.clinic_medication_stock?.medication_batch.medication.name ||
      d.nurse_medication_stock?.medication_batch.medication.name ||
      'Medication';
    const batchNo = stockObj?.medication_batch?.batch_number ?? '-';
    const currentQty: number = Number(stockObj?.quantity ?? 0);
    const returnQty: number = Number(d.quantity ?? 0);
    const newQty = currentQty + returnQty;

    const { isConfirmed } = await Swal.fire({
      title: 'Delete & Return to Stock?',
      html: `
        <div style="text-align:left">
          <p><strong>Medication:</strong> ${medName}</p>
          <p><strong>From:</strong> ${stockTypeLabel} stock #${stockObj?.id ?? '-'}</p>
          <p><strong>Batch:</strong> ${batchNo}</p>
          <hr />
          <p><strong>Current Stock:</strong> ${currentQty}</p>
          <p><strong>Quantity to Return:</strong> ${returnQty}</p>
          <p><strong>New Stock After Return:</strong> ${newQty}</p>
          <hr />
          <p>This will <strong>delete</strong> the dispensation record and <strong>return</strong> the quantity back to stock.</p>
        </div>
      `,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete & return',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      reverseButtons: true,
      focusCancel: true,
    });

    if (!isConfirmed) return;

    Swal.fire({
      title: 'Processing…',
      allowOutsideClick: false,
      didOpen: () => Swal.showLoading(),
    });

    router.delete(route('dispensations.delete_return', d.id), {
      preserveScroll: true,
      onSuccess: async () => {
        Swal.close();
        await Swal.fire('Done', 'Dispensation deleted and quantity returned to stock.', 'success');
        // Reload lists & stocks
        router.reload({
          only: ['dispensations', 'clinicMedicationStocks', 'nurseMedicationStocks'],
          preserveScroll: true,
        });
      },
      onError: async (errors) => {
        Swal.close();
        const msg =
          (errors && typeof errors === 'object' ? Object.values(errors).join('\n') : 'Failed to delete and return.') ||
          'Failed to delete and return.';
        await Swal.fire('Error', msg, 'error');
      },
    });
  };

  /** ===== UI ===== */
  const breadcrumbs = [
    { title: 'Consultations', href: '/consultations' },
    { title: `Consultation #${consultation.id}`, href: `/consultations/${consultation.id}` },
    { title: 'Dispense Medication', href: '#' },
  ];
  const stockBannerText = globalStockType === 'clinic' ? 'Clinic Stock' : 'Nurse Stock';

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Dispense Medication" />



      <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">
        <h1 className="text-2xl font-bold">
          Dispense Medication to{' '}
          <span style={{ textTransform: 'capitalize' }}>
            {consultation.patient.first_name} {consultation.patient.surname}
          </span>
        </h1>

        {/* Banner */}
        <div className="flex items-start gap-3 rounded-md border border-blue-200 bg-blue-50 p-3 text-sm dark:border-blue-900/50 dark:bg-blue-900/20">
          <InfoIcon className="mt-0.5 h-4 w-4" />
          <div>
            You are dispensing from <strong>{stockBannerText}</strong>. Choose medications once, then fill their details below.
          </div>
        </div>

        {/* Search */}
        <div className="mb-2 flex gap-2">
          <div className="relative w-72">
            <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
            <input
              type="text"
              value={searchQuery}
              onChange={(e) => {
                const v = e.target.value;
                setSearchQuery(v);
                if (v === '') handleResetSearch();
              }}
              onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
              placeholder="Search medications..."
              className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:outline-none focus:ring-2 focus:ring-blue-600"
            />
          </div>
          <button
            onClick={handleSearch}
            className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
          >
            Search
          </button>
          <button
            onClick={handleResetSearch}
            className="rounded-lg border px-4 py-2 text-sm font-medium hover:bg-muted"
          >
            Reset
          </button>
        </div>

        {/* ONE stock source dropdown */}
        <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
          <div className="max-w-xs">
            <label className="mb-1 block text-sm font-medium">Stock Source</label>
            <Select
              instanceId="global-stock-type"
              styles={selectStyles}
              options={stockTypeOptions}
              value={stockTypeOptions.find((o) => o.value === globalStockType) || null}
              onChange={onChangeGlobalStockType}
              classNamePrefix="rs"
              menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
              menuPosition="fixed"
            />
          </div>

          {/* ONE (multi) medication dropdown */}
          <div>
            <label className="mb-1 block text-sm font-medium">Medications to dispense (multi-select)</label>
            <Select
              instanceId="meds-multi"
              isMulti
              styles={selectStyles}
              options={medicationOptions}
              value={selectedMeds}
              onChange={(vals: MultiValue<MedOpt>) => setSelectedMeds(vals as MedOpt[])}
              classNamePrefix="rs"
              placeholder="Select one or more medications…"
              menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
              menuPosition="fixed"
            />
          </div>
        </div>

        {/* Compact rows (no medication dropdown here) */}
        <form onSubmit={handleSubmitBatch} className="space-y-3">
          <div className="overflow-x-auto">
            <table className="min-w-full border text-sm">
              <thead className="bg-muted">
                <tr>
                  <th className="px-3 py-2 text-left">Medication</th>
                  <th className="px-3 py-2 text-left">Qty</th>
                  <th className="px-3 py-2 text-left">Freq</th>
                  <th className="px-3 py-2 text-left">Route</th>
                  <th className="px-3 py-2 text-left">Days</th>
                  <th className="px-3 py-2 text-left">Notes</th>
                  <th className="px-3 py-2"></th>
                </tr>
              </thead>
              <tbody className="divide-y">
                {selectedMeds.map((med) => {
                  const stock = findStock(med.value);
                  const row = rowsByStockId[med.value] || {
                    quantity: '',
                    frequency: '',
                    route: '',
                    duration: '',
                    notes: '',
                  };

                  return (
                    <tr key={med.value} className="align-top">
                      <td className="px-2 py-2 min-w-[260px]">
                        <div className="font-medium">{med.label.split(' — ')[0]}</div>
                        {stock && (
                          <div className="mt-1 text-xs text-muted-foreground">
                            Available: <strong>{stock.quantity}</strong> — batch {stock.medication_batch.batch_number || '-'}
                          </div>
                        )}
                      </td>

                      <td className="w-[90px] px-2 py-2">
                        <input
                          type="text"
                          value={row.quantity}
                          onChange={(e) => {
                            const v = e.target.value;
                            if (v === '' || /^[0-9\b]+$/.test(v)) updateRow(med.value, { quantity: v });
                          }}
                          className="w-full rounded-md border px-2 py-1"
                          placeholder="0"
                        />
                      </td>

                      <td className="w-[160px] px-2 py-2">
                        <Select
                          instanceId={`freq-${med.value}`}
                          styles={selectStyles}
                          options={Object.entries(frequencyOptions).map(([value, label]) => ({ value, label }))}
                          value={
                            row.frequency
                              ? { value: row.frequency, label: frequencyOptions[row.frequency] ?? row.frequency }
                              : null
                          }
                          onChange={(opt) => updateRow(med.value, { frequency: opt?.value || '' })}
                          isClearable
                          classNamePrefix="rs"
                          menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                          menuPosition="fixed"
                        />
                      </td>

                      <td className="w-[160px] px-2 py-2">
                        <Select
                          instanceId={`route-${med.value}`}
                          styles={selectStyles}
                          options={Object.entries(routeOptions).map(([value, label]) => ({ value, label }))}
                          value={
                            row.route
                              ? { value: row.route, label: routeOptions[row.route] ?? row.route }
                              : null
                          }
                          onChange={(opt) => updateRow(med.value, { route: opt?.value || '' })}
                          isClearable
                          classNamePrefix="rs"
                          menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                          menuPosition="fixed"
                        />
                      </td>

                      <td className="w-[90px] px-2 py-2">
                        <input
                          type="text"
                          value={row.duration}
                          onChange={(e) => {
                            const v = e.target.value;
                            if (v === '' || /^[0-9\b]+$/.test(v)) updateRow(med.value, { duration: v });
                          }}
                          className="w-full rounded-md border px-2 py-1"
                          placeholder="0"
                        />
                      </td>

                      <td className="px-2 py-2">
                        <input
                          type="text"
                          value={row.notes}
                          onChange={(e) => updateRow(med.value, { notes: e.target.value })}
                          className="w-full rounded-md border px-2 py-1"
                          placeholder="Notes"
                        />
                      </td>

                      <td className="w-[46px] px-2 py-2">
                        <button
                          type="button"
                          onClick={() => removeSelectedMed(med.value)}
                          className="inline-flex items-center rounded border px-2 py-1 text-xs hover:bg-muted"
                          title="Remove"
                        >
                          <Trash2 className="h-4 w-4" />
                        </button>
                      </td>
                    </tr>
                  );
                })}
                {selectedMeds.length === 0 && (
                  <tr>
                    <td colSpan={7} className="px-3 py-6 text-center text-sm text-muted-foreground">
                      Select one or more medications above to start dispensing.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>

          <div className="flex items-center justify-end">
            <Button
              type="submit"
              disabled={isSubmitting}
              className="inline-flex items-center gap-2"
              style={{ cursor: isSubmitting ? 'not-allowed' : 'pointer' }}
            >
              {isSubmitting ? 'Processing…' : 'Dispense Selected'}
            </Button>
          </div>
        </form>

        <div className="bg-card rounded-lg border p-4 shadow-sm" style={{ marginTop: '10px', marginBottom: '10px' }}>
          <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
            <Pill className="text-muted-foreground h-5 w-5" />
            Dispensed Medications
          </h2>
          {dispensations?.length > 0 ? (
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y text-sm">
                <thead className="bg-muted">
                  <tr>
                    <th className="px-3 py-2 text-left">Medication</th>
                    <th className="px-3 py-2 text-left">Qty</th>
                    <th className="px-3 py-2 text-left">Frequency</th>
                    <th className="px-3 py-2 text-left">Route</th>
                    <th className="px-3 py-2 text-left">Date</th>
                    <th className="px-3 py-2 text-left">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {dispensations.map((d) => {
                    const medName =
                      d.clinic_medication_stock?.medication_batch.medication.name ||
                      d.nurse_medication_stock?.medication_batch.medication.name;

                    // NEW: dosage/form/unit + batch for display
                    const medObj =
                      d.clinic_medication_stock?.medication_batch?.medication ||
                      d.nurse_medication_stock?.medication_batch?.medication ||
                      {};
                    const dosage = medObj?.dosage ?? '';
                    const unit = medObj?.unit ?? '';
                    const form = medObj?.form ?? '';
                    const batch =
                      d.clinic_medication_stock?.medication_batch?.batch_number ??
                      d.nurse_medication_stock?.medication_batch?.batch_number ??
                      '-';

                    return (
                      <tr key={d.id} className="hover:bg-muted/50">
                        <td className="px-3 py-2">
                          <span className="inline-block rounded-full bg-blue-100 px-3 py-1 text-xs font-semibold text-blue-800 dark:bg-blue-800/20 dark:text-blue-300">
                            {medName}
                          </span>
                          {/* NEW subline with more info */}
                          <div className="mt-1 text-xs text-muted-foreground">
                            {(dosage || unit) ? `${dosage}${unit ? ` ${unit}` : ''}` : '-'}
                            {form ? ` • ${form}` : ''}
                            {batch ? ` • Batch ${batch}` : ''}
                          </div>
                        </td>
                        <td className="px-3 py-2">{d.quantity}</td>
                        <td className="px-3 py-2">{d.frequency || '-'}</td>
                        <td className="px-3 py-2">{d.route || '-'}</td>
                        <td className="px-3 py-2">{moment(d.dispensed_at).format('ddd D MMM YYYY')}</td>
                        <td className="px-3 py-2 flex gap-2">
                          <Button
                            variant="outline"
                            className="text-xs"
                            onClick={() => router.visit(route('dispensations.corrections.create', d.id))}
                            style={{ cursor: 'pointer' }}
                          >
                            Request Correction
                          </Button>

                          {/* <Button
                            variant="destructive"
                            className="text-xs"
                            onClick={() => handleDeleteAndReturn(d)}
                            style={{ cursor: 'pointer' }}
                          >
                            Delete & Return
                          </Button> */}
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          ) : (
            <p className="text-muted-foreground text-sm">No medications dispensed yet.</p>
          )}
        </div>

        {/* Stocks table + pagination */}
        <div className="grid grid-cols-12 gap-8 mt-6">
          <div className="col-span-8">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y border text-sm">
                <thead className="bg-muted">
                  <tr>
                    {['Medication', 'Dosage', 'Form', 'Available Qty', 'Batch', 'Expiry'].map((header) => (
                      <th key={header} className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                        {header}
                      </th>
                    ))}
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {currentStocks.map((stock) => (
                    <tr key={stock.id} className="transition hover:bg-muted/50">
                      <td className="px-4 py-3">{stock.medication_batch.medication.name}</td>
                      <td className="px-4 py-3">{stock.medication_batch.medication.dosage || '-'}</td>
                      <td className="px-4 py-3">{stock.medication_batch.medication.form || '-'}</td>
                      <td className="px-4 py-3">{stock.quantity}</td>
                      <td className="px-4 py-3">{stock.medication_batch.batch_number || '-'}</td>
                      <td className="px-4 py-3">{stock.medication_batch.expiry_date || '-'}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>

            {globalStockType === 'clinic' && isPaginated(clinicMedicationStocks) && (
              <div className="mt-6 flex justify-center">
                <ReactPaginate
                  pageCount={clinicMedicationStocks.last_page}
                  forcePage={clinicMedicationStocks.current_page - 1}
                  onPageChange={handlePageChange}
                  marginPagesDisplayed={1}
                  pageRangeDisplayed={3}
                  previousLabel="← Prev"
                  nextLabel="Next →"
                  breakLabel="..."
                  containerClassName="flex items-center gap-2 text-sm"
                  pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  activeClassName="bg-blue-600 text-white"
                  previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  breakClassName="px-2"
                />
              </div>
            )}
            {globalStockType === 'nurse' && isPaginated(nurseMedicationStocks) && (
              <div className="mt-6 flex justify-center">
                <ReactPaginate
                  pageCount={nurseMedicationStocks.last_page}
                  forcePage={nurseMedicationStocks.current_page - 1}
                  onPageChange={handlePageChange}
                  marginPagesDisplayed={1}
                  pageRangeDisplayed={3}
                  previousLabel="← Prev"
                  nextLabel="Next →"
                  breakLabel="..."
                  containerClassName="flex items-center gap-2 text-sm"
                  pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  activeClassName="bg-blue-600 text-white"
                  previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  breakClassName="px-2"
                />
              </div>
            )}
          </div>

          {/* Right pane */}
          <div className="col-span-4 space-y-6">
            <div>
              <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px', marginBottom: '10px' }}>
                <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                <Button onClick={() => handleCloseConsultation(consultation.id)} className="mb-3 w-full" style={{ cursor: 'pointer' }}>
                  Close Consultation
                </Button>
                <Button className="mb-3 w-full" onClick={handleGoToReferralsPage} style={{ cursor: 'pointer' }}>
                  Go to Patient Referral Page
                </Button>
              </div>

              <section className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="text-foreground mb-4 text-xl font-semibold">Latest Consultation</h2>
                {consultation ? (
                  <div className="space-y-4">
                    <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 md:grid-cols-3">
                      <div>
                        <div className="text-muted-foreground text-sm">Date</div>
                        <div className="text-foreground font-medium">
                          {moment(consultation.consultation_date).format('DD MMM YYYY, h:mm a')}
                        </div>
                      </div>
                      <div>
                        <div className="text-muted-foreground text-sm">Doctor/RGN</div>
                        <div className="text-foreground font-medium">{consultation.doctor?.name || '-'}</div>
                      </div>
                      <div>
                        <div className="text-muted-foreground text-sm">Clinic</div>
                        <div className="text-foreground font-medium">{consultation.clinic?.name || '-'}</div>
                      </div>
                    </div>

                    {consultation.presenting_complaint && (
                      <div>
                        <div className="text-muted-foreground text-sm">Presenting Complaint</div>
                        <div className="text-foreground rounded border p-3 font-medium">
                          {consultation?.presenting_complaint}
                        </div>
                      </div>
                    )}
                    {consultation.diagnosis && (
                      <div>
                        <div className="text-muted-foreground text-sm">Diagnosis</div>
                        <div className="text-foreground rounded border p-3 font-medium">
                          {consultation.diagnosis}
                        </div>
                      </div>
                    )}

                    <Button
                      variant="outline"
                      className="w-full"
                      onClick={() => {
                        if (consultation?.id) router.visit(route('consultations.show', consultation.id));
                      }}
                      style={{ cursor: 'pointer' }}
                    >
                      View Full Consultation
                    </Button>
                  </div>
                ) : (
                  <p className="text-muted-foreground">No consultation data available.</p>
                )}
              </section>
            </div>
          </div>
        </div>
      </div>

      {/* Correction Modal (antd) */}
      <Modal
        title="Request Dispensation Correction"
        open={corrOpen}
        onCancel={closeCorrectionModal}
        onOk={() => corrForm.submit()}
        okText={corrSubmitting ? 'Submitting…' : 'Submit Request'}
        okButtonProps={{ loading: corrSubmitting }}
        destroyOnHidden
      >
        {corrDisp && (
          <div className="mb-3 rounded border p-3 text-sm">
            <div className="mb-1 font-medium">Original Dispensation</div>
            <div>
              <strong>Medication: </strong>
              {corrDisp.clinic_medication_stock?.medication_batch.medication.name ||
                corrDisp.nurse_medication_stock?.medication_batch.medication.name}
            </div>
            <div>
              <strong>From Stock: </strong>
              {corrDisp.clinic_medication_stock ? 'Clinic' : 'Nurse'} #{corrDisp.clinic_medication_stock?.id || corrDisp.nurse_medication_stock?.id}
            </div>
            <div>
              <strong>Dispensed Qty: </strong>
              {corrDisp.quantity}
            </div>
          </div>
        )}

        <Form form={corrForm} layout="vertical" onFinish={submitCorrection}>
          <Form.Item
            name="to_stock_type"
            label="Correct To — Stock Type"
            rules={[{ required: true, message: 'Please select a target stock type' }]}
            initialValue={corrToType}
          >
            <Radio.Group
              onChange={(e) => {
                setCorrToType(e.target.value);
                corrForm.setFieldValue('to_stock_id', undefined);
              }}
            >
              <Radio.Button value="clinic">Clinic</Radio.Button>
              <Radio.Button value="nurse">Nurse</Radio.Button>
            </Radio.Group>
          </Form.Item>

          <Form.Item
            name="to_stock_id"
            label="Target Stock Item"
            rules={[{ required: true, message: 'Please choose a stock item' }]}
          >
            <AntdSelect
              showSearch
              allowClear
              placeholder="Search stock…"
              optionFilterProp="label"
              options={corrToType === 'clinic' ? clinicAntdOptions : nurseAntdOptions}
              filterOption={(input, option) => (option?.label as string)?.toLowerCase().includes(input.toLowerCase())}
            />
          </Form.Item>

          <Form.Item
            name="quantity"
            label="Quantity to Correct"
            tooltip="Must be ≤ the originally dispensed quantity for this item"
            rules={[
              { required: true, message: 'Please enter a quantity' },
              { type: 'number', min: 1, message: 'Quantity must be at least 1' },
            ]}
            initialValue={corrDisp?.quantity || 1}
          >
            <InputNumber className="w-full" min={1} />
          </Form.Item>

          <Form.Item name="reason" label="Reason (optional)">
            <Input.TextArea rows={3} placeholder="Brief reason for the correction" />
          </Form.Item>
        </Form>
      </Modal>
    </AppLayout>
  );
}
