import { Button } from '@/components/ui/button';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, router, usePage } from '@inertiajs/react';
import { Pencil, Printer } from 'lucide-react';
import React from 'react';

interface Medication {
    id: number;
    name: string;
    dosage: string | null;
    form: string | null;
    unit: string | null;
    instructions: string | null;
    created_at: string;
    updated_at: string;
}

export default function MedicationShow() {
    const { medication } = usePage<{ medication: Medication }>().props;

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Medications', href: route('medications.index') },
        { title: medication.name, href: route('medications.show', medication.id) },
    ];

    const Info = ({ label, value }: { label: string; value?: string | null }) => (
        <div>
            <div className="text-muted-foreground text-sm">{label}</div>
            <div className="text-foreground font-medium">{value || '-'}</div>
        </div>
    );

    const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
            {items.map((item) => (
                <Info key={item.label} {...item} />
            ))}
        </div>
    );

    const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
        <section className="bg-card rounded-lg border p-6 shadow-sm">
            <h2 className="text-foreground mb-4 text-xl font-semibold">{title}</h2>
            {children}
        </section>
    );

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`Medication: ${medication.name}`} />
            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                {/* Header: Name + Buttons */}
                <div className="flex flex-col justify-between gap-4 sm:flex-row sm:items-center">
                    <h1 className="text-foreground text-2xl font-bold">{medication.name}</h1>

                    {/* Action Buttons */}
                    <div className="flex flex-wrap gap-2">
                        <Button variant="outline" onClick={() => router.visit(route('medications.edit', medication.id))}>
                            <Pencil className="mr-2 h-4 w-4" />
                            Edit
                        </Button>
                        <Button variant="outline">
                            <Printer className="mr-2 h-4 w-4" />
                            Print
                        </Button>
                    </div>
                </div>

                <div className="space-y-6">
                    <Section title="Basic Information">
                        <InfoGrid
                            items={[
                                { label: 'Name', value: medication.name },
                                { label: 'Dosage', value: medication.dosage },
                                { label: 'Form', value: medication.form },
                                { label: 'Unit', value: medication.unit },
                            ]}
                        />
                    </Section>

                    <Section title="Usage Instructions">
                        <div className="rounded bg-gray-50 p-4 whitespace-pre-wrap dark:bg-gray-800 dark:text-gray-200">
                            {medication.instructions || 'No instructions provided'}
                        </div>
                    </Section>

                    <Section title="Metadata">
                        <InfoGrid
                            items={[
                                { label: 'Created At', value: new Date(medication.created_at).toLocaleString() },
                                { label: 'Last Updated', value: new Date(medication.updated_at).toLocaleString() },
                            ]}
                        />
                    </Section>
                </div>
            </div>
        </AppLayout>
    );
}
