import React, { useMemo } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Calendar } from 'lucide-react';

import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  Tooltip,
  CartesianGrid,
} from 'recharts';

type DayPoint = { date: string; label: string; count: number };
type ClinicCount = { clinic_id: number; clinic_name: string; count: number };
type DiagnosisCount = { name: string; count: number };
type PaymentMethodCount = { payment_method: string; count: number };

type Props = {
  day: string;
  nurseName: string;
  totalToday: number;
  last7: DayPoint[]; // last 7 days ending on selected date
  referrals: {
    radiology: number;
    lab: number;
    specialist: number;
    casualty: number;
    out_patient: number;
    sick_leave: number;
    total_referrals: number;
  }; // ALL TIME
  iodCount: number; // ALL TIME
  perClinic: ClinicCount[]; // TODAY (selected date)
  diagnoses: DiagnosisCount[]; // ALL TIME (Top)
  paymentMethods: PaymentMethodCount[]; // TODAY (selected date)
  avgVitals: { temperature: number; systolic: number; diastolic: number }; // TODAY (selected date)
  allowedClinics: { id: number; name: string }[];
};

export interface BreadcrumbItem {
  title: string;
  href: string;
}

export default function NurseReports({
  day,
  nurseName,
  totalToday,
  last7,
  referrals,
  iodCount,
  perClinic,
  diagnoses,
  paymentMethods,
  avgVitals,
  allowedClinics,
}: Props) {
  const totalDiagnoses = useMemo(
    () => diagnoses.reduce((s, d) => s + d.count, 0),
    [diagnoses]
  );

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Triage', href: '#' },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`Nurse Reports • ${nurseName}`} />

      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 flex flex-wrap items-start justify-between gap-4">
          <div>
            <h1 className="text-2xl font-semibold">Nurse Reports</h1>
            <p className="text-sm text-muted-foreground">
              {nurseName} • {new Date(day).toLocaleDateString()}
            </p>
          </div>

          {/* Bigger date picker with helper note */}
          <div className="flex flex-col items-end gap-2">
            <label htmlFor="report-date" className="text-sm font-medium">
              Report Date
            </label>
            <div className="relative">
              <Calendar className="pointer-events-none absolute left-3 top-1/2 h-5 w-5 -translate-y-1/2 text-muted-foreground" />
              <input
                id="report-date"
                type="date"
                value={day}
                onChange={(e) => {
                  const d = e.target.value;
                  router.visit(route('nurse.reports', { date: d }), { preserveScroll: true });
                }}
                className="h-12 w-[240px] rounded-md border px-10 text-base"
              />
            </div>
            <p className="max-w-md text-right text-xs text-muted-foreground">
              The selected date anchors this report. <b>Consultations Today</b>, <b>Per-clinic</b>,{' '}
              <b>Payment Methods</b>, and <b>Average Vitals</b> show data for that day. The{' '}
              <b>Last 7 Days</b> chart shows the six days before and the selected date.{' '}
              <b>Referrals/Sick Leave</b> and <b>Injury On Duty</b> are all-time totals for you.
            </p>
          </div>
        </div>

        {/* SUMMARY CARDS */}
        <div className="mb-6 grid gap-4 sm:grid-cols-12">
          <Card className="sm:col-span-3">
            <CardHeader>
              <CardTitle>Consultations Today</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-4xl font-bold">{totalToday}</div>
              <div className="mt-2 text-xs text-muted-foreground">
                Across clinics you can access.
              </div>
            </CardContent>
          </Card>

          <Card className="sm:col-span-5">
            <CardHeader>
              <CardTitle>Referrals & Sick Leave (All Time)</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 gap-3 text-sm">
                <div className="flex items-center justify-between">
                  <span>Radiology</span>
                  <Badge variant="secondary">{referrals.radiology}</Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span>Lab</span>
                  <Badge variant="secondary">{referrals.lab}</Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span>Specialist</span>
                  <Badge variant="secondary">{referrals.specialist}</Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span>Casualty</span>
                  <Badge variant="secondary">{referrals.casualty}</Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span>Out-patient</span>
                  <Badge variant="secondary">{referrals.out_patient}</Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span>Sick Leave</span>
                  <Badge className="bg-amber-600 text-white hover:bg-amber-600">
                    {referrals.sick_leave}
                  </Badge>
                </div>
              </div>

              <div className="mt-4 flex items-center justify-between rounded-md border bg-muted/40 p-3 text-sm">
                <span className="font-medium">Total Referrals</span>
                <Badge className="bg-blue-600 hover:bg-blue-600">
                  {referrals.total_referrals}
                </Badge>
              </div>
            </CardContent>
          </Card>

          <Card className="sm:col-span-4">
            <CardHeader>
              <CardTitle>Injury On Duty (All Time)</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-4xl font-bold">{iodCount}</div>
              <div className="mt-2 text-xs text-muted-foreground">Recorded by you.</div>
            </CardContent>
          </Card>
        </div>

        {/* CHARTS ROW */}
        <div className="mb-6 grid gap-4 sm:grid-cols-12">
          {/* Last 7 days blue-ish chart */}
          <Card className="sm:col-span-8">
            <CardHeader>
              <CardTitle>Consultations — Last 7 Days</CardTitle>
            </CardHeader>
            <CardContent style={{ height: 320 }}>
              <ResponsiveContainer width="100%" height="100%">
                <BarChart data={last7}>
                  <defs>
                    <linearGradient id="blueFade" x1="0" y1="0" x2="0" y2="1">
                      <stop offset="0%" stopColor="#60a5fa" stopOpacity={0.95} />
                      <stop offset="100%" stopColor="#3b82f6" stopOpacity={0.85} />
                    </linearGradient>
                  </defs>
                  <CartesianGrid stroke="rgba(59,130,246,0.15)" strokeDasharray="3 3" />
                  <XAxis
                    dataKey="label"
                    tick={{ fill: '#1e40af' }}
                    axisLine={{ stroke: 'rgba(59,130,246,0.35)' }}
                    tickLine={{ stroke: 'rgba(59,130,246,0.35)' }}
                  />
                  <YAxis
                    allowDecimals={false}
                    tick={{ fill: '#1e40af' }}
                    axisLine={{ stroke: 'rgba(59,130,246,0.35)' }}
                    tickLine={{ stroke: 'rgba(59,130,246,0.35)' }}
                  />
                  <Tooltip contentStyle={{ borderColor: 'rgba(59,130,246,0.35)' }} />
                  <Bar dataKey="count" fill="url(#blueFade)" stroke="#1d4ed8" strokeWidth={1} radius={[6, 6, 0, 0]} />
                </BarChart>
              </ResponsiveContainer>
            </CardContent>
          </Card>

          {/* Per-clinic (badges list) – today */}
          <Card className="sm:col-span-4">
            <CardHeader>
              <CardTitle>Consulted Today by Clinic</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                {perClinic.length === 0 ? (
                  <div className="text-sm text-muted-foreground">No consultations recorded today.</div>
                ) : (
                  perClinic.map((c) => (
                    <div key={c.clinic_id} className="flex items-center justify-between">
                      <Badge className="border border-blue-200 bg-blue-50 text-blue-700 hover:bg-blue-50">
                        {c.clinic_name}
                      </Badge>
                      <span className="font-semibold">{c.count}</span>
                    </div>
                  ))
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* DIAGNOSES (horizontal) + EXTRAS */}
        <div className="grid gap-4 sm:grid-cols-12">
          {/* Horizontal diagnoses bar chart – ALL TIME */}
          <Card className="sm:col-span-8">
            <CardHeader>
              <CardTitle>Top Diagnoses (All Time)</CardTitle>
            </CardHeader>
            <CardContent style={{ height: 420 }}>
              {diagnoses.length === 0 ? (
                <div className="text-sm text-muted-foreground">No diagnoses recorded.</div>
              ) : (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart
                    data={[...diagnoses].reverse()}
                    layout="vertical"
                    margin={{ left: 60, right: 16, top: 10, bottom: 10 }}
                  >
                    <defs>
                      <linearGradient id="blueFadeH" x1="0" y1="0" x2="1" y2="0">
                        <stop offset="0%" stopColor="#93c5fd" stopOpacity={0.95} />
                        <stop offset="100%" stopColor="#3b82f6" stopOpacity={0.85} />
                      </linearGradient>
                    </defs>
                    <CartesianGrid
                      stroke="rgba(59,130,246,0.15)"
                      strokeDasharray="3 3"
                      horizontal={true}
                      vertical={false}
                    />
                    <XAxis
                      type="number"
                      allowDecimals={false}
                      tick={{ fill: '#1e40af' }}
                      axisLine={{ stroke: 'rgba(59,130,246,0.35)' }}
                      tickLine={{ stroke: 'rgba(59,130,246,0.35)' }}
                    />
                    <YAxis dataKey="name" type="category" width={160} tick={{ fill: '#1e40af' }} />
                    <Tooltip contentStyle={{ borderColor: 'rgba(59,130,246,0.35)' }} />
                    <Bar dataKey="count" fill="url(#blueFadeH)" stroke="#1d4ed8" strokeWidth={1} radius={[0, 6, 6, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              )}
              {totalDiagnoses > 0 && (
                <div className="mt-2 text-xs text-muted-foreground">
                  Total diagnosis entries counted: {totalDiagnoses}
                </div>
              )}
            </CardContent>
          </Card>

          {/* Extras: payment methods & vitals – TODAY */}
          <Card className="sm:col-span-4">
            <CardHeader>
              <CardTitle>Extras (Today)</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="mb-4">
                <div className="mb-2 text-sm font-medium">Top Payment Methods</div>
                <div className="space-y-2">
                  {paymentMethods.length === 0 ? (
                    <div className="text-sm text-muted-foreground">No data.</div>
                  ) : (
                    paymentMethods.map((p, i) => (
                      <div key={`${p.payment_method}-${i}`} className="flex items-center justify-between text-sm">
                        <Badge variant="outline">{p.payment_method || 'N/A'}</Badge>
                        <span className="font-semibold">{p.count}</span>
                      </div>
                    ))
                  )}
                </div>
              </div>

              <div>
                <div className="mb-2 text-sm font-medium">Average Vitals</div>
                <div className="grid grid-cols-3 gap-2 text-center">
                  <div className="rounded-md border p-2">
                    <div className="text-xs text-muted-foreground">Temp (°C)</div>
                    <div className="text-lg font-semibold">
                      {avgVitals.temperature ? avgVitals.temperature.toFixed(1) : '—'}
                    </div>
                  </div>
                  <div className="rounded-md border p-2">
                    <div className="text-xs text-muted-foreground">Sys</div>
                    <div className="text-lg font-semibold">
                      {avgVitals.systolic ? avgVitals.systolic.toFixed(0) : '—'}
                    </div>
                  </div>
                  <div className="rounded-md border p-2">
                    <div className="text-xs text-muted-foreground">Dia</div>
                    <div className="text-lg font-semibold">
                      {avgVitals.diastolic ? avgVitals.diastolic.toFixed(0) : '—'}
                    </div>
                  </div>
                </div>
              </div>

              <div className="mt-4">
                <div className="mb-2 text-sm font-medium">Clinics in Scope</div>
                <div className="flex flex-wrap gap-2">
                  {allowedClinics.length === 0 ? (
                    <span className="text-xs text-muted-foreground">None</span>
                  ) : (
                    allowedClinics.map((c) => (
                      <Badge
                        key={c.id}
                        className="border border-blue-200 bg-blue-50 text-blue-700 hover:bg-blue-50"
                      >
                        {c.name}
                      </Badge>
                    ))
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </AppLayout>
  );
}
