import React, { useMemo, useRef, useState, useMemo as useMemoAlias } from 'react'
import AppLayout from '@/layouts/app-layout'
import { Head, usePage } from '@inertiajs/react'
import Swal from 'sweetalert2'

// shadcn/ui
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card'
import { Tabs, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Badge } from '@/components/ui/badge'
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table'
import { Separator } from '@/components/ui/separator'
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert'
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select'
import { Label } from '@/components/ui/label'

import {
  Upload,
  FileSpreadsheet,
  CheckCircle2,
  XCircle,
  Download,
  Building2,
  FileWarning,
  Files
} from 'lucide-react'

type Mode = 'patients' | 'dependents'
type Company = { id: number; name: string }

type InertiaPageProps = {
  templates: {
    patients: { headers: string[] }
    dependents: { headers: string[] }
  }
  companies: Company[]
}

type PreviewRow = {
  index: number
  data: Record<string, any>
  errors: string[]
  raw: Record<string, any>
}

type PreviewResponse = {
  columns: string[]
  rows: PreviewRow[]
  stats: { total: number; valid: number; invalid: number }
}

type ReasonCount = { reason: string; count: number }

type CommitResponse = {
  ok: boolean
  created: number
  failed: number
  errors?: Array<{ row: number; errors: string[] }>
  invalid_download_url?: string | null
  duplicates_download_url?: string | null
  invalid_reasons_summary?: ReasonCount[]
  duplicate_reasons_summary?: ReasonCount[]
  invalid_examples?: Array<{ row_index: number; errors: string[] }>
  duplicate_examples?: Array<{ row_index: number; reason: string }>
  message?: string
}

// --- helpers (new) ---
function getCookie(name: string): string | null {
  return document.cookie
    .split('; ')
    .find(row => row.startsWith(name + '='))
    ?.split('=')[1] ?? null
}

function sameOriginUrl(u: string): string {
  // Defensive: if a route returns absolute URL with localhost vs 127.0.0.1 mismatch,
  // convert to current origin to keep cookies working in dev.
  try {
    const url = new URL(u, window.location.origin)
    url.host = window.location.host
    url.protocol = window.location.protocol
    return url.toString()
  } catch {
    return u
  }
}

export default function BulkImportPage() {
  const { templates, companies } = usePage<InertiaPageProps>().props

  const [mode, setMode] = useState<Mode>('patients')
  const [companyId, setCompanyId] = useState<string>('')
  const [file, setFile] = useState<File | null>(null)

  const [preview, setPreview] = useState<PreviewResponse | null>(null)
  const [loadingPreview, setLoadingPreview] = useState(false)
  const [committing, setCommitting] = useState(false)

  const [artifactInvalidUrl, setArtifactInvalidUrl] = useState<string | null>(null)
  const [artifactDuplicatesUrl, setArtifactDuplicatesUrl] = useState<string | null>(null)

  const fileRef = useRef<HTMLInputElement | null>(null)

  const dataColumns = useMemo(() => {
    if (!preview?.rows?.length) return []
    const set = new Set<string>()
    preview.rows.forEach(r => Object.keys(r.data || {}).forEach(k => set.add(k)))
    return Array.from(set)
  }, [preview])

  const csrf = useMemoAlias(() => {
    const meta = document.querySelector<HTMLMetaElement>('meta[name="csrf-token"]')
    return meta?.content ?? ''
  }, [])

  const xsrf = useMemo(() => {
    const raw = getCookie('XSRF-TOKEN')
    return raw ? decodeURIComponent(raw) : ''
  }, [])

  // @ts-ignore (Ziggy global)
  const previewRoute: string = route('bulk.import.preview')
  // @ts-ignore
  const commitRoute: string = route('bulk.import.commit')

  const onPickFile = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      setFile(e.target.files[0])
      setPreview(null)
      setArtifactInvalidUrl(null)
      setArtifactDuplicatesUrl(null)
    }
  }

  const clearAll = () => {
    setFile(null)
    setPreview(null)
    setArtifactInvalidUrl(null)
    setArtifactDuplicatesUrl(null)
    if (fileRef.current) fileRef.current.value = ''
  }

  const ensureCompanyIfPatients = (): boolean => {
    if (mode === 'patients' && !companyId) {
      Swal.fire({
        icon: 'warning',
        title: 'Select a company',
        text: 'Please choose the target company for this upload.',
      })
      return false
    }
    return true
  }

  const doPreview = async () => {
    if (!file) {
      Swal.fire({ icon: 'warning', title: 'Choose a file', text: 'Please select an .xlsx or .csv file first.' })
      return
    }
    if (!ensureCompanyIfPatients()) return

    const fd = new FormData()
    fd.append('mode', mode)
    if (mode === 'patients') fd.append('company_id', companyId)
    fd.append('file', file)
    fd.append('_token', csrf)

    try {
      setLoadingPreview(true)
      Swal.fire({
        title: 'Parsing file…',
        html: 'Validating rows; please wait.',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      })

      const res = await fetch(sameOriginUrl(previewRoute), {
        method: 'POST',
        body: fd,
        headers: {
          'X-Requested-With': 'XMLHttpRequest',
          'X-CSRF-TOKEN': csrf,   // meta token
          'X-XSRF-TOKEN': xsrf,   // cookie token (important)
          'Accept': 'application/json',
        },
        credentials: 'same-origin',
      })

      if (!res.ok) {
        Swal.close()
        const text = await res.text()
        throw new Error(text || 'Failed to preview file.')
      }

      const json = (await res.json()) as PreviewResponse
      setPreview(json)

      Swal.close()
      Swal.fire({
        icon: 'success',
        title: 'Preview ready',
        text: `Found ${json.stats.total} rows; ${json.stats.valid} valid, ${json.stats.invalid} invalid.`,
      })
    } catch (err: any) {
      Swal.close()
      Swal.fire({ icon: 'error', title: 'Preview failed', text: err?.message ?? 'Could not parse the file.' })
    } finally {
      setLoadingPreview(false)
    }
  }

  const escapeHtml = (s: string) =>
    s.replace(/[&<>"']/g, (c) => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c]!))

  const renderReasonsHtml = (label: string, list?: ReasonCount[]) => {
    if (!list || list.length === 0) return ''
    const items = list
      .slice(0, 8)
      .map(rc => `<li><code>${escapeHtml(rc.reason)}</code> — <strong>${rc.count}</strong></li>`)
      .join('')
    return `<div style="margin-top:.5rem">
      <div style="font-weight:600">${label}</div>
      <ul style="margin:.25rem 0 .5rem 1rem; line-height:1.4">${items}</ul>
    </div>`
  }

  const doCommit = async () => {
    if (!file) {
      Swal.fire({ icon: 'warning', title: 'Choose a file', text: 'Please select a file first.' })
      return
    }
    if (!preview) {
      Swal.fire({ icon: 'info', title: 'No preview yet', text: 'Please preview the file before importing.' })
      return
    }
    if (!ensureCompanyIfPatients()) return

    const fd = new FormData()
    fd.append('mode', mode)
    if (mode === 'patients') fd.append('company_id', companyId)
    fd.append('file', file)
    fd.append('_token', csrf)

    try {
      setCommitting(true)
      Swal.fire({
        title: 'Importing…',
        html: 'Creating records; please wait.',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      })

      const res = await fetch(sameOriginUrl(commitRoute), {
        method: 'POST',
        body: fd,
        headers: {
          'X-Requested-With': 'XMLHttpRequest',
          'X-CSRF-TOKEN': csrf,   // meta token
          'X-XSRF-TOKEN': xsrf,   // cookie token (important)
          'Accept': 'application/json',
        },
        credentials: 'same-origin',
      })

      const text = await res.text()
      let json: CommitResponse | null = null
      try {
        json = text ? (JSON.parse(text) as CommitResponse) : null
      } catch {
        // non-JSON error body
      }

      Swal.close()

      if (res.status === 422) {
        let msg = 'Validation failed.'
        if (json && (json as any).errors) {
          const errs = (json as any).errors
          const flat = Object.entries(errs).flatMap(([k, arr]: any) => (arr || []).map((m: string) => `${k}: ${m}`))
          if (flat.length) msg = flat.join('\n')
        } else if (text) {
          msg = text
        }
        Swal.fire({ icon: 'error', title: 'Import failed', text: msg })
        return
      }

      if (!res.ok || !json || json.ok === false) {
        const msg = (json?.message || text || 'Import failed.')
        Swal.fire({ icon: 'error', title: 'Import failed', text: msg })
        return
      }

      const {
        created,
        failed,
        invalid_download_url,
        duplicates_download_url,
        invalid_reasons_summary,
        duplicate_reasons_summary,
      } = json

      setArtifactInvalidUrl(invalid_download_url ?? null)
      setArtifactDuplicatesUrl(duplicates_download_url ?? null)

      const invalidHtml = renderReasonsHtml('Top invalid reasons', invalid_reasons_summary)
      const duplicateHtml = renderReasonsHtml('Top duplicate reasons', duplicate_reasons_summary)

      Swal.fire({
        icon: failed > 0 ? 'warning' : 'success',
        title: 'Import finished',
        html: `<div style="text-align:left">
          <div><strong>Created:</strong> ${created}</div>
          <div><strong>Failed:</strong> ${failed}</div>
          ${invalidHtml}${duplicateHtml}
          ${(invalid_download_url || duplicates_download_url)
            ? '<div style="margin-top:.25rem">Artifacts are available below.</div>'
            : ''}
        </div>`,
      })
    } catch (err: any) {
      Swal.close()
      Swal.fire({ icon: 'error', title: 'Import failed', text: err?.message ?? 'Could not import the file.' })
    } finally {
      setCommitting(false)
    }
  }

  const downloadTemplate = (m: Mode) => {
    const headers = (m === 'patients' ? templates.patients.headers : templates.dependents.headers) || []
    const content = headers.join(',') + '\n'
    const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' })
    const url = URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url
    a.download = `${m}_template.csv`
    document.body.appendChild(a)
    a.click()
    a.remove()
    URL.revokeObjectURL(url)
  }

  const stats = preview?.stats

  return (
    <AppLayout breadcrumbs={[{ title: 'Patients', href: route('patients.index') }, { title: 'Bulk Import', href: '#' }]}>
      <Head title="Bulk Import" />

      {/* A. Prevent page-level horizontal scroll */}
      <div className="mx-8 my-6 overflow-x-hidden">
        <Card className="shadow-sm">
          <CardHeader>
            <CardTitle>Bulk Import</CardTitle>
            <CardDescription>Upload an .xlsx or .csv file, preview the rows, then import into the system.</CardDescription>
          </CardHeader>

          <CardContent className="space-y-6">
            {/* Mode & Template */}
            <div className="flex flex-col items-start gap-3 sm:flex-row sm:items-center sm:justify-between">
              <Tabs
                value={mode}
                onValueChange={(v) => { setMode(v as Mode); setPreview(null); setArtifactInvalidUrl(null); setArtifactDuplicatesUrl(null); }}
                className="w-full sm:w-auto"
              >
                <TabsList>
                  <TabsTrigger value="patients">Employees (Account Holders)</TabsTrigger>
                  <TabsTrigger value="dependents">Dependents</TabsTrigger>
                </TabsList>
              </Tabs>

              <div className="flex items-center gap-2">
                <Button variant="outline" size="sm" onClick={() => downloadTemplate('patients')}>
                  <Download className="mr-2 h-4 w-4" />
                  Patients Template
                </Button>
                <Button variant="outline" size="sm" onClick={() => downloadTemplate('dependents')}>
                  <Download className="mr-2 h-4 w-4" />
                  Dependents Template
                </Button>
              </div>
            </div>

            <Separator />

            {/* Company Select (required for Employees) */}
            {mode === 'patients' && (
              <div className="grid gap-2">
                <Label className="text-sm font-medium">Company to import into</Label>
                <div className="flex items-center gap-2">
                  <Building2 className="h-4 w-4 text-muted-foreground" />
                  <Select value={companyId} onValueChange={setCompanyId}>
                    <SelectTrigger className="w-full sm:w-[28rem]">
                      <SelectValue placeholder="Select a company..." />
                    </SelectTrigger>
                    <SelectContent className="max-h-[320px]">
                      {companies.map((c) => (
                        <SelectItem key={c.id} value={String(c.id)}>
                          {c.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <p className="text-xs text-muted-foreground">All new employees will be created under the selected company.</p>
              </div>
            )}

            {/* Upload */}
            <div className="grid gap-4 sm:grid-cols-[1fr_auto] sm:items-end">
              <div>
                <label className="mb-1 block text-sm font-medium">Choose file (.xlsx or .csv)</label>
                <div className="flex items-center gap-2">
                  <input
                    ref={fileRef}
                    type="file"
                    accept=".xlsx,.csv"
                    onChange={onPickFile}
                    className="w-full rounded-md border border-border bg-background p-2 text-sm"
                  />
                  {file ? (
                    <Badge variant="secondary" className="shrink-0 max-w-[50vw] truncate">{file.name}</Badge>
                  ) : null}
                </div>
              </div>

              <div className="flex gap-2">
                <Button onClick={doPreview} disabled={!file || loadingPreview || (mode === 'patients' && !companyId)} className="shrink-0">
                  <Upload className="mr-2 h-4 w-4" />
                  {loadingPreview ? 'Parsing…' : 'Preview'}
                </Button>
                <Button onClick={doCommit} disabled={!file || !preview || committing || (mode === 'patients' && !companyId)} className="shrink-0" variant="secondary">
                  <FileSpreadsheet className="mr-2 h-4 w-4" />
                  {committing ? 'Importing…' : 'Import'}
                </Button>
                <Button variant="outline" onClick={clearAll} disabled={!file && !preview && !artifactInvalidUrl && !artifactDuplicatesUrl} className="shrink-0">
                  Clear
                </Button>
              </div>
            </div>

            {/* Stats */}
            {preview && (
              <div className="flex flex-wrap items-center gap-3">
                <Badge variant="outline">Total: {stats?.total ?? 0}</Badge>
                <Badge className="bg-emerald-600 hover:bg-emerald-600">Valid: {stats?.valid ?? 0}</Badge>
                <Badge className="bg-rose-600 hover:bg-rose-600">Invalid: {stats?.invalid ?? 0}</Badge>
              </div>
            )}

            {/* No preview yet */}
            {!preview && !artifactInvalidUrl && !artifactDuplicatesUrl && (
              <Alert>
                <AlertTitle>No preview to show yet</AlertTitle>
                <AlertDescription>Choose a mode and file{mode === 'patients' ? ', select a company,' : ''} then click “Preview”.</AlertDescription>
              </Alert>
            )}

            {/* Preview table */}
            {preview && (
              <div className="rounded-md border overflow-hidden">
                <div className="max-h-[60vh] overflow-y-auto overflow-x-hidden">
                  <Table className="w-full table-fixed">
                    <TableHeader className="sticky top-0 z-10 bg-background">
                      <TableRow>
                        <TableHead className="w-[80px]">Row</TableHead>
                        {dataColumns.map((col) => (
                          <TableHead
                            key={col}
                            className="capitalize break-words whitespace-normal align-top"
                          >
                            {col.replaceAll('_', ' ')}
                          </TableHead>
                        ))}
                        <TableHead className="w-[220px]">Validation</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {preview.rows.map((r) => {
                        const isValid = !r.errors || r.errors.length === 0
                        return (
                          <TableRow key={r.index} className={isValid ? '' : 'bg-rose-50/40 dark:bg-rose-950/10'}>
                            <TableCell className="font-medium tabular-nums align-top">{r.index}</TableCell>
                            {dataColumns.map((col) => (
                              <TableCell
                                key={`${r.index}-${col}`}
                                className="align-top break-words whitespace-normal"
                              >
                                <div className="max-w-[24rem] break-words whitespace-normal">
                                  {formatCell(r.data?.[col])}
                                </div>
                              </TableCell>
                            ))}
                            <TableCell className="align-top">
                              {isValid ? (
                                <span className="inline-flex items-center gap-1 rounded-full bg-emerald-100 px-2 py-0.5 text-xs font-medium text-emerald-800 dark:bg-emerald-900/40 dark:text-emerald-200">
                                  <CheckCircle2 className="h-3.5 w-3.5" />
                                  OK
                                </span>
                              ) : (
                                <div className="flex flex-wrap gap-1">
                                  {r.errors.map((e, i) => (
                                    <span
                                      key={i}
                                      className="inline-flex items-center gap-1 rounded-full bg-rose-100 px-2 py-0.5 text-[11px] font-medium text-rose-800 dark:bg-rose-900/40 dark:text-rose-200"
                                    >
                                      <XCircle className="h-3.5 w-3.5" />
                                      {e}
                                    </span>
                                  ))}
                                </div>
                              )}
                            </TableCell>
                          </TableRow>
                        )
                      })}
                    </TableBody>
                  </Table>
                </div>
              </div>
            )}

            {/* Artifacts (appear after commit if any) */}
            {(artifactInvalidUrl || artifactDuplicatesUrl) && (
              <div className="rounded-md border p-4">
                <div className="flex items-center gap-2 mb-2">
                  <Files className="h-4 w-4" />
                  <div className="text-sm font-medium">Artifacts</div>
                </div>
                <div className="flex flex-wrap gap-2">
                  {artifactInvalidUrl && (
                    <a href={artifactInvalidUrl} target="_blank" rel="noopener noreferrer">
                      <Button variant="outline" size="sm">
                        <FileWarning className="mr-2 h-4 w-4" />
                        Download Invalid Rows
                      </Button>
                    </a>
                  )}
                  {artifactDuplicatesUrl && (
                    <a href={artifactDuplicatesUrl} target="_blank" rel="noopener noreferrer">
                      <Button variant="outline" size="sm">
                        <Download className="mr-2 h-4 w-4" />
                        Download Duplicates
                      </Button>
                    </a>
                  )}
                </div>
                <p className="mt-2 text-xs text-muted-foreground">
                  These files open in a new tab. If your browser blocks the download, please allow pop-ups.
                </p>
              </div>
            )}
          </CardContent>

          <CardFooter className="justify-between">
            <div className="text-xs text-muted-foreground">Only valid, non-duplicate rows will be created during import.</div>
            <div className="flex items-center gap-2">
              <Badge variant="secondary">{mode === 'patients' ? 'Mode: Patients' : 'Mode: Dependents'}</Badge>
              {mode === 'patients' && companyId && <Badge variant="outline">Company ID: {companyId}</Badge>}
            </div>
          </CardFooter>
        </Card>
      </div>
    </AppLayout>
  )
}

function formatCell(v: any) {
  if (v == null || v === '') return <span className="text-muted-foreground">—</span>
  if (typeof v === 'boolean') return v ? 'Yes' : 'No'
  return String(v)
}
