import { AddressInfoSection, ContactInfoSection, EmergencyContactSection, MedicalInfoSection, PersonalInfoSection } from '@/components/PatientForm';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, Link, usePage } from '@inertiajs/react';
import { useEffect, useMemo, useRef, useState } from 'react';
import Select from 'react-select';
import Swal from 'sweetalert2';

type FormShape = {
  first_name: string;
  middle_name: string;
  surname: string;
  employee_number: string;
  company_id: string;
  date_of_birth: string;
  email: string;
  gender: string;
  id_number: string;
  medical_aid_number: string;
  medical_aid_provider: string;
  phone: string;
  emergency_contact_name: string;
  emergency_contact_relation: string;
  emergency_contact_phone: string;
  allergies: string;
  is_smoker: boolean;
  occupation: string;

  home_address: string;
  work_area: string;
  suburb: string;

  marital_status: string;

  /** NEW: Optional chronic flag (not required) */
  is_chronic_patient: boolean;

  // Sent to backend, used only for redirect logic
  redirect_to_dependent: boolean;
};

const STORAGE_KEY = 'patient_form_draft_v1';

export default function PatientCreate() {
  const { companies } = usePage<{ companies: any[] }>().props;

  // initial form values
  const initialData: FormShape = useMemo(
    () => ({
      first_name: '',
      middle_name: '',
      surname: '',
      employee_number: '',
      company_id: '',
      date_of_birth: '',
      email: '',
      gender: '',
      id_number: '',
      medical_aid_number: '',
      medical_aid_provider: '',
      phone: '',
      emergency_contact_name: '',
      emergency_contact_relation: '',
      emergency_contact_phone: '',
      allergies: '',
      is_smoker: false,
      occupation: '',

      home_address: '',
      work_area: '',
      suburb: '',

      marital_status: '',

      // NEW
      is_chronic_patient: false,

      redirect_to_dependent: false,
    }),
    []
  );

  const { data, setData, post, errors, processing, reset } = useForm<FormShape>(initialData);

  const [isDarkMode, setIsDarkMode] = useState(false);
  const [selectedCompany, setSelectedCompany] = useState<any>(null);

  // NEW: ref to focus Employee Number on validation error
  const employeeNumberRef = useRef<HTMLInputElement | null>(null);

  // ── Parse duplicate candidates coming from the backend (if any) ─────────
  const duplicateCandidates = useMemo(() => {
    const raw = (errors as any)?.duplicate_candidates;
    if (!raw) return [];
    try {
      const str = Array.isArray(raw) ? raw[0] : raw;
      const parsed = JSON.parse(str);
      return Array.isArray(parsed) ? parsed : [];
    } catch {
      return [];
    }
  }, [errors]);
  // ────────────────────────────────────────────────────────────────────────────

  // Dark mode tracker
  useEffect(() => {
    const checkDarkMode = () => {
      setIsDarkMode(document.documentElement.classList.contains('dark'));
    };
    checkDarkMode();

    const observer = new MutationObserver(checkDarkMode);
    observer.observe(document.documentElement, { attributes: true, attributeFilter: ['class'] });
    return () => observer.disconnect();
  }, []);

  const companyOptions = companies.map((company) => ({
    value: String(company.id),
    label: company.name,
  }));

  // Load draft from localStorage on mount
  useEffect(() => {
    try {
      const raw = localStorage.getItem(STORAGE_KEY);
      if (raw) {
        const draft = JSON.parse(raw) as Partial<FormShape>;
        setData((prev) => ({
          ...prev,
          ...draft,
          // ensure booleans are proper when restoring
          is_smoker: Boolean(draft.is_smoker),
          is_chronic_patient: Boolean(draft.is_chronic_patient),
          redirect_to_dependent: Boolean(draft.redirect_to_dependent),
        }));
        if (draft.company_id) {
          const match = companyOptions.find((o) => String(o.value) === String(draft.company_id));
          setSelectedCompany(match ?? null);
        }
      }
    } catch {}
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Keep the react-select in sync with data.company_id (works for restored draft too)
  useEffect(() => {
    if (data.company_id) {
      const match = companyOptions.find((o) => String(o.value) === String(data.company_id));
      setSelectedCompany(match ?? null);
    } else {
      setSelectedCompany(null);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [data.company_id, companies]);

  const handleCompanyChange = (selectedOption: any) => {
    setSelectedCompany(selectedOption);
    setData('company_id', selectedOption ? String(selectedOption.value) : '');
  };

  // Save to localStorage whenever the form data changes
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(data));
    } catch {}
  }, [data]);

  const selectStyles = {
    control: (base: any) => ({
      ...base,
      backgroundColor: isDarkMode ? '#000' : '#fff',
      borderColor: isDarkMode ? '#374151' : '#d1d5db',
      color: isDarkMode ? '#fff' : '#000',
      '&:hover': { borderColor: isDarkMode ? '#4b5563' : '#9ca3af' },
    }),
    menu: (base: any) => ({ ...base, backgroundColor: isDarkMode ? '#000' : '#fff', color: isDarkMode ? '#fff' : '#000' }),
    option: (base: any, { isFocused }: any) => ({
      ...base,
      backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
      color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
      '&:active': { backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6' },
    }),
    singleValue: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    input: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    placeholder: (base: any) => ({ ...base, color: isDarkMode ? '#9ca3af' : '#6b7280' }),
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    // EARLY GUARD: Employee Number must be present — show Swal and focus input
    if (!String(data.employee_number || '').trim()) {
      await Swal.fire({
        icon: 'error',
        title: 'Employee Number is required',
        text: 'Please enter the Employee Number before saving the patient.',
        confirmButtonText: 'OK',
      });
      // Focus the field after the alert closes
      employeeNumberRef.current?.focus();
      return;
    }

    // Require: company_id, id_number, gender (generic list alert)
    const missing: string[] = [];
    if (!String(data.company_id || '').trim()) missing.push('Company');
    if (!String(data.id_number || '').trim()) missing.push('National ID Number');
    if (!String(data.gender || '').trim()) missing.push('Gender');

    if (missing.length) {
      await Swal.fire({
        icon: 'error',
        title: 'Missing required fields',
        html: `<div style="text-align:left">
                 <p>Please provide the following:</p>
                 <ul style="margin-left:1rem; list-style:disc;">
                   ${missing.map((m) => `<li><b>${m}</b></li>`).join('')}
                 </ul>
               </div>`,
        confirmButtonText: 'OK',
      });
      return;
    }

    post(route('patients.store'), {
      onSuccess: () => {
        localStorage.removeItem(STORAGE_KEY);
      },
    });
  };

  const handleResetDraft = async () => {
    const res = await Swal.fire({
      title: 'Clear form & draft?',
      text: 'This will clear the form and remove the saved draft from this browser.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Clear form',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
    });

    if (res.isConfirmed) {
      reset();
      setSelectedCompany(null);
      setData('redirect_to_dependent', false);
      localStorage.removeItem(STORAGE_KEY);
      await Swal.fire({ icon: 'success', title: 'Cleared', timer: 1200, showConfirmButton: false });
    }
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Create Patient', href: route('patients.create') },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Create Patient" />
      <div className="w-full px-4 py-6 sm:px-6">
        {/* ── Top Notice ───────────────────────────────────────────────────── */}
        <div className="mb-6 rounded-lg border border-amber-300 bg-amber-50 p-4 text-amber-900 dark:border-amber-700 dark:bg-amber-900/20 dark:text-amber-200">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <p className="font-semibold">Before you add a patient</p>
              <p className="mt-1 text-sm">
                Please confirm the <b>company the patient belongs to</b>. If the company is not available, create it first, then come back and add the patient.
              </p>
            </div>
            <div className="flex flex-wrap gap-2">
              <Button variant="destructive" onClick={handleResetDraft}>
                Reset Form (clear draft)
              </Button>

              <Button asChild variant="secondary">
                <Link href={route('companies.create')}>Create Company</Link>
              </Button>

              <Button asChild>
                <a href="#patient-form">Add Patient</a>
              </Button>
            </div>
          </div>
        </div>
        {/* ─────────────────────────────────────────────────────────────────── */}

        {/* ── Duplicate panel ─────────────────────────────────────────── */}
        {duplicateCandidates.length > 0 && (
          <div className="mb-6 rounded-lg border border-red-300 bg-red-50 p-4 text-red-900 dark:border-red-700 dark:bg-red-900/20 dark:text-red-100">
            <h3 className="mb-2 text-base font-semibold">Possible duplicate(s) found</h3>
            <p className="mb-3 text-sm">
              A patient with the same National ID / Employee Number or matching Name + DOB/Phone already exists. Review and open the existing record instead of creating a new one.
            </p>

            <div className="space-y-2">
              {duplicateCandidates.map((p: any) => (
                <div key={p.id} className="flex flex-wrap items-center justify-between gap-3 rounded-md border border-red-200 bg-white/70 p-3 dark:border-red-800 dark:bg-neutral-900">
                  <div className="text-sm">
                    <div className="font-medium">{p.full_name}</div>
                    <div className="text-muted-foreground">
                      {[
                        p.date_of_birth ? `DOB: ${p.date_of_birth}` : null,
                        p.id_number ? `ID: ${p.id_number}` : null,
                        p.employee_number ? `Emp No: ${p.employee_number}` : null,
                        p.phone ? `Phone: ${p.phone}` : null,
                      ].filter(Boolean).join(' · ')}
                    </div>
                  </div>
                  <div className="flex gap-2">
                    <Button asChild variant="secondary">
                      <Link href={route('patients.show', p.id)}>View</Link>
                    </Button>
                    <Button asChild>
                      <Link href={route('triages.create', { patient_id: p.id })}>Triage</Link>
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}
        {/* ─────────────────────────────────────────────────────────────────── */}

        <form id="patient-form" onSubmit={handleSubmit} className="space-y-8">
          <div className="grid gap-6 sm:grid-cols-12">
            {/* Main form content */}
            <div className="col-span-12 space-y-6 sm:col-span-8">
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                {/* Header with redirect choice */}
                <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-start sm:justify-between">
                  <h2 className="text-xl font-semibold">Personal Information</h2>

                  <label className="flex select-none items-center gap-2 text-sm">
                    <input
                      type="checkbox"
                      className="h-5 w-5 rounded border-gray-300 text-blue-600 focus:ring-blue-500 dark:border-gray-700"
                      checked={!!data.redirect_to_dependent}
                      onChange={(e) => setData('redirect_to_dependent', e.target.checked)}
                    />
                    <span className="font-medium">
                      Redirect to <b>Dependent</b> after saving
                    </span>
                  </label>
                </div>

                <div className="mb-4 rounded-lg border border-blue-300 bg-blue-50 p-2 text-sm dark:border-blue-800 dark:bg-blue-900/20">
                  <div className="flex items-center gap-2">
                    <span className="font-medium">After saving, go to:</span>
                    <div className="inline-flex overflow-hidden rounded-lg border border-blue-300 dark:border-blue-800">
                      <button
                        type="button"
                        aria-pressed={!data.redirect_to_dependent}
                        onClick={() => setData('redirect_to_dependent', false)}
                        className={`px-3 py-1.5 text-sm ${!data.redirect_to_dependent ? 'bg-blue-600 text-white' : 'bg-transparent text-blue-700 dark:text-blue-200'}`}
                        style={{ cursor: 'pointer' }}
                      >
                        Create Triage
                      </button>
                      <button
                        type="button"
                        aria-pressed={data.redirect_to_dependent}
                        onClick={() => setData('redirect_to_dependent', true)}
                        className={`px-3 py-1.5 text-sm ${data.redirect_to_dependent ? 'bg-blue-600 text-white' : 'bg-transparent text-blue-700 dark:text-blue-200'}`}
                        style={{ cursor: 'pointer' }}
                      >
                        Create Dependent
                      </button>
                    </div>
                  </div>
                  <p className="mt-2 text-xs text-blue-800 dark:text-blue-200">
                    NB: This selection only controls the next screen. It does <i>not</i> change any patient data.
                  </p>
                </div>

                <PersonalInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Employment Information</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="employee_number">
                      Employee Number <span className="text-red-600">*</span>
                    </Label>
                    <Input
                      id="employee_number"
                      ref={employeeNumberRef}
                      value={data.employee_number}
                      onChange={(e) => setData('employee_number', e.target.value)}
                      required
                    />
                    {errors.employee_number && (
                      <p className="text-sm text-red-500">{errors.employee_number}</p>
                    )}
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="company_id">
                      Company <span className="text-red-600">*</span>
                    </Label>
                    <Select
                      options={companyOptions}
                      value={selectedCompany}
                      onChange={handleCompanyChange}
                      placeholder="Select company..."
                      styles={selectStyles}
                      className="react-select-container"
                      classNamePrefix="react-select"
                      isSearchable
                    />
                    {errors.company_id && <p className="text-sm text-red-500">{errors.company_id}</p>}
                  </div>
                </div>
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Address Information</h2>
                <AddressInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Contact Information</h2>
                <ContactInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-2 text-xl font-semibold">Medical Information</h2>
                <MedicalInfoSection data={data} errors={errors} onFieldChange={setData} />

                {/* NEW: Chronic patient checkbox (optional) */}
                <div className="mt-4 rounded-md border p-3">
                  <label className="flex items-center gap-2 text-sm">
                    <input
                      type="checkbox"
                      className="h-5 w-5 rounded border-gray-300 text-blue-600 focus:ring-blue-500 dark:border-gray-700"
                      checked={!!data.is_chronic_patient}
                      onChange={(e) => setData('is_chronic_patient', e.target.checked)}
                    />
                    <span className="font-medium">Mark as <b>Chronic Patient</b> (optional)</span>
                  </label>
                  <p className="mt-1 text-xs text-muted-foreground">
                    Use this flag to track patients on chronic care programs. You can change it anytime.
                  </p>
                </div>
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Emergency Contact</h2>
                <EmergencyContactSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="flex flex-col justify-end gap-4 sm:flex-row">
                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                  Cancel
                </Button>
                <Button type="submit" disabled={processing}>
                  {processing ? 'Saving...' : 'Save'}
                </Button>
              </div>
            </div>

            {/* Optional sidebar area */}
            <div className="col-span-12 space-y-6 sm:col-span-4">
              <div className="space-y-4">
                <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px' }}>
                  <h2 className="mb-4 text-xl font-semibold">Actions</h2>

                  {data.redirect_to_dependent ? (
                    <p className="mb-3 text-sm text-amber-700 dark:text-amber-300">
                      On save, you’ll go straight to <b>Create Dependent</b> with this patient set as the parent.
                    </p>
                  ) : (
                    <p className="mb-3 text-sm text-gray-600 dark:text-gray-300">
                      On save, you’ll proceed to <b>Create Triage</b> for this patient.
                    </p>
                  )}

                  <Button asChild className="mb-3 w-full">
                    <Link href={route('dependents.create')}>Create New Dependent</Link>
                  </Button>

                  <div className="rounded-md border border-amber-300 bg-amber-50 p-3 text-xs text-amber-900 dark:border-amber-700 dark:bg-amber-900/20 dark:text-amber-200">
                    <strong>NB:</strong> The selection above only affects where you’re redirected after saving this patient.
                  </div>
                </div>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
