import {
  AddressInfoSection,
  ContactInfoSection,
  EmergencyContactSection,
  MedicalInfoSection,
  PersonalInfoSection,
} from '@/components/PatientForm';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, Link, usePage } from '@inertiajs/react';
import React from 'react';
import Swal from 'sweetalert2';

type PageProps = {
  patient: any;
  companies: any[];
  // Provided by controller/middleware:
  missing_fields?: string[];
  flash?: {
    warning?: string | null;
  };
};

export default function PatientEdit() {
  const { patient, companies, missing_fields = [], flash = {} as any } = usePage<PageProps>().props;

  const { data, setData, put, errors, processing } = useForm({
    ...patient,
    company_id: patient.company?.id || '',
    // Ensure boolean for checkbox
    is_chronic_patient: Boolean(patient.is_chronic_patient),
  });

  // If you reached this page via middleware redirection, you may have ?return=<url> in the query.
  // We'll preserve it so "Save & Triage" can use that, otherwise we’ll construct a triage URL.
  const currentUrl = (usePage() as any).url as string;
  const query = new URLSearchParams(currentUrl.split('?')[1] || '');
  const returnFromQuery = query.get('return'); // might be /triages/create?patient_id=...

  const updateUrlBase = route('patients.update', patient.id);
  const patientsIndexUrl = route('patients.index');
  const triageUrl = returnFromQuery || route('triages.create', { patient_id: patient.id });

  /** Helpers */
  const onlyDigits = (val: any) => String(val ?? '').replace(/\D+/g, '');
  const isPhoneComplete = (val: any) => {
    const digits = onlyDigits(val);
    // Block legacy placeholder "263" and require a reasonable length (>= 9 local digits).
    if (!digits || digits === '263') return false;
    return digits.length >= 9;
  };

  const emergencyMissing = () => {
    const nameOk = String(data.emergency_contact_name || '').trim().length > 0;
    const relOk = String(data.emergency_contact_relation || '').trim().length > 0;
    const phoneOk = isPhoneComplete(data.emergency_contact_phone || '');
    return !(nameOk && relOk && phoneOk);
  };

  const patientPhoneMissing = () => !isPhoneComplete(data.phone || '');

  const showBlockingSwal = async (title: string, text: string) => {
    await Swal.fire({
      icon: 'warning',
      title,
      text,
      confirmButtonText: 'OK',
    });
  };

  const submitWithIntent = async (intent: 'index' | 'triage') => {
    // Front-end blocking validation before PUT
    if (patientPhoneMissing() && emergencyMissing()) {
      return showBlockingSwal(
        'Incomplete Contact Details',
        'Please enter a valid patient phone number and complete all emergency contact fields (name, relationship, and phone) before saving.'
      );
    }
    if (patientPhoneMissing()) {
      return showBlockingSwal('Invalid Patient Phone', 'Please enter a complete patient phone number before saving.');
    }
    if (emergencyMissing()) {
      return showBlockingSwal(
        'Emergency Contact Required',
        'Please provide the emergency contact name, relationship, and a valid emergency phone number before saving.'
      );
    }

    const target = intent === 'triage' ? triageUrl : patientsIndexUrl;
    // Append `return` so your PatientController@update can redirect server-side.
    const updateUrl = `${updateUrlBase}?return=${encodeURIComponent(target)}`;

    put(updateUrl, {
      preserveScroll: true,
      onSuccess: () => {
        // Fallback: if backend didn’t honor `return`, still send user to the chosen place.
        window.location.href = target;
      },
    });
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Edit Patient', href: route('patients.edit', patient.id) },
  ];

  const warning = flash?.warning || (missing_fields.length ? 'Some required information is missing.' : null);

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`Edit Patient`} />
      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 flex items-center justify-between">
          <h1 className="text-2xl font-bold">Edit Patient</h1>
        </div>

        {/* Missing fields banner (if any passed from middleware/controller) */}
        {warning && (
          <div className="mb-6 rounded-md border border-yellow-300 bg-yellow-50 p-4 text-sm text-yellow-900 dark:border-yellow-900/40 dark:bg-yellow-900/20 dark:text-yellow-100">
            <div className="font-medium">{warning}</div>
            {Array.isArray(missing_fields) && missing_fields.length > 0 && (
              <ul className="mt-2 list-disc pl-5">
                {missing_fields.map((item: string, idx: number) => (
                  <li key={idx}>{item}</li>
                ))}
              </ul>
            )}
          </div>
        )}

        <form onSubmit={(e) => e.preventDefault()} className="space-y-8">
          <div className="grid grid-cols-12 gap-8">
            <div className="col-span-12 lg:col-span-8 space-y-6">
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Personal Information</h2>
                <PersonalInfoSection
                  data={data}
                  errors={errors}
                  onFieldChange={(field, value) => setData(field, value)}
                />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Employment Information</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="employee_number">Employee Number</Label>
                    <Input
                      id="employee_number"
                      value={data.employee_number || ''}
                      onChange={(e) => setData('employee_number', e.target.value)}
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="company_id">Company</Label>
                    <select
                      id="company_id"
                      value={data.company_id || ''}
                      onChange={(e) => setData('company_id', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-black dark:text-white dark:placeholder-gray-400 dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select company</option>
                      {companies.map((company) => (
                        <option key={company.id} value={company.id}>
                          {company.name}
                        </option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Address Information</h2>
                <AddressInfoSection
                  data={data}
                  errors={errors}
                  onFieldChange={(field, value) => setData(field, value)}
                />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Contact Information</h2>
                {/* The inner section is unchanged; we enforce validation at save-time via Swal */}
                <ContactInfoSection
                  data={data}
                  errors={errors}
                  onFieldChange={(field, value) => setData(field, value)}
                />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-2 text-xl font-semibold">Medical Information</h2>
                <MedicalInfoSection
                  data={data}
                  errors={errors}
                  onFieldChange={(field, value) => setData(field, value)}
                />

                {/* NEW: Chronic patient checkbox (optional) */}
                <div className="mt-4 rounded-md border p-3">
                  <label className="flex items-center gap-2 text-sm">
                    <input
                      type="checkbox"
                      className="h-5 w-5 rounded border-gray-300 text-blue-600 focus:ring-blue-500 dark:border-gray-700"
                      checked={!!data.is_chronic_patient}
                      onChange={(e) => setData('is_chronic_patient', e.target.checked)}
                    />
                    <span className="font-medium">
                      Mark as <b>Chronic Patient</b> (optional)
                    </span>
                  </label>
                  <p className="mt-1 text-xs text-muted-foreground">
                    Toggle this if the patient is enrolled in a chronic care program.
                  </p>
                </div>
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Emergency Contact</h2>
                <EmergencyContactSection
                  data={data}
                  errors={errors}
                  onFieldChange={(field, value) => setData(field, value)}
                />
              </div>

              <div className="flex flex-col-reverse items-stretch gap-2 sm:flex-row sm:justify-end sm:gap-3">
                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                  Cancel
                </Button>

                {/* Save & go to Patients index */}
                <Button
                  type="button"
                  variant="secondary"
                  disabled={processing}
                  onClick={() => submitWithIntent('index')}
                >
                  {processing ? 'Saving…' : 'Save & Patients'}
                </Button>

                {/* Save & go to Triage create (for this patient) */}
                <Button
                  type="button"
                  disabled={processing}
                  onClick={() => submitWithIntent('triage')}
                >
                  {processing ? 'Saving…' : 'Save & Triage'}
                </Button>
              </div>
            </div>

            <div className="col-span-12 lg:col-span-4 space-y-6">
              <div className="space-y-4">
                <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px' }}>
                  <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                  <Button asChild className="mb-3 w-full">
                    <Link href={route('patients.create')}>Create New Patient</Link>
                  </Button>

                  <Button asChild variant="outline" className="mb-3 w-full">
                    <Link href={route('dependents.create')}>Create New Dependent</Link>
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
