import { useEffect, useMemo, useRef, useState } from 'react';
import { Head, Link, router, usePage } from '@inertiajs/react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';

import AppLayout from '@/layouts/app-layout';
import { API } from '@/config';

import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue
} from '@/components/ui/select';

import { Eye, Pencil, Plus, Search as SearchIcon, X, Trash2, Stethoscope, Info, Users, User } from 'lucide-react';
import moment from 'moment';

// Small debounce hook (no lodash)
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return debounced;
}

type PageProps = {
  patients: any;
  filters: any;
  companies: any[];
  stats: any;
  auth?: { user?: any };
  flash?: any;
};

export default function PatientIndex() {
  const pageProps = usePage<PageProps>().props as any;
  const { patients, filters, companies, stats } = pageProps;

  // Role checks
  const user = pageProps?.auth?.user;
  const isSuperAdmin =
    user?.role === 'superadmin' ||
    (Array.isArray(user?.roles) && user.roles.includes('superadmin'));

  const isNurse =
    user?.role === 'nurse' ||
    (Array.isArray(user?.roles) && user.roles.includes('nurse'));

  const [search, setSearch] = useState<string>(filters.search || '');
  const [gender, setGender] = useState<string>(filters.gender || '');
  const [companyId, setCompanyId] = useState<string>(
    filters.company_id ? String(filters.company_id) : ''
  );
  const [maritalStatus, setMaritalStatus] = useState<string>(filters.marital_status || '');
  const [ageMin, setAgeMin] = useState<string>(filters.age_min || '');
  const [ageMax, setAgeMax] = useState<string>(filters.age_max || '');

  const debouncedSearch = useDebouncedValue(search, 450);

  // ===== Bulk selection state (superadmin only) =====
  const [selectedIds, setSelectedIds] = useState<number[]>([]);
  const allIdsOnPage = useMemo<number[]>(
    () => patients?.data?.map((p: any) => p.id) ?? [],
    [patients?.data]
  );
  const allSelectedOnPage = useMemo(
    () => allIdsOnPage.length > 0 && allIdsOnPage.every((id) => selectedIds.includes(id)),
    [allIdsOnPage, selectedIds]
  );

  const toggleOne = (id: number) => {
    setSelectedIds((prev) =>
      prev.includes(id) ? prev.filter((x) => x !== id) : [...prev, id]
    );
  };

  const toggleAllOnPage = () => {
    setSelectedIds((prev) => {
      if (allSelectedOnPage) {
        // unselect all on page
        return prev.filter((id) => !allIdsOnPage.includes(id));
      }
      // add all on page
      const set = new Set(prev);
      allIdsOnPage.forEach((id) => set.add(id));
      return Array.from(set);
    });
  };

  // ===== Filters =====
  const applyFilters = (page = 1) => {
    router.get(
      `${API}/patients`,
      {
        page,
        search,
        gender,
        company_id: companyId,
        marital_status: maritalStatus,
        age_min: ageMin,
        age_max: ageMax,
      },
      {
        preserveState: true,
        replace: true,
      },
    );
  };

  const handlePageChange = (selected: { selected: number }) => {
    applyFilters(selected.selected + 1);
  };

  const resetFilters = () => {
    setSearch('');
    setGender('');
    setCompanyId('');
    setMaritalStatus('');
    setAgeMin('');
    setAgeMax('');
    router.get(`${API}/patients`, {}, { preserveState: true, replace: true });
  };

  // ===== Single delete (kept from your code) =====
  const handleDelete = async (patient: any) => {
    const fullName = [patient.first_name, patient.middle_name, patient.surname]
      .filter(Boolean)
      .join(' ');

    const { isConfirmed } = await Swal.fire({
      title: `Delete ${fullName}?`,
      icon: 'warning',
      html: `
        <div style="text-align:left">
          <p>This will permanently remove:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>Patient profile</b></li>
            <li><b>All dependents</b> (if any)</li>
            <li><b>All consultations</b> for this patient and dependents</li>
            <li><b>All triages</b> for this patient and dependents</li>
          </ul>
          <p style="margin-top:.5rem;"><b>This action cannot be undone.</b></p>
        </div>
      `,
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      focusCancel: true,
    });

    if (!isConfirmed) return;

    Swal.fire({
      title: 'Deleting...',
      text: 'Please wait',
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      didOpen: () => Swal.showLoading(),
    });

    router.visit(route('patients.destroy', patient.id), {
      method: 'delete',
      preserveScroll: true,
      onFinish: () => Swal.close(),
      onSuccess: () => {
        Swal.fire('Deleted', 'Patient and related records were deleted successfully.', 'success');
        // refresh current page
        applyFilters(patients.current_page ?? 1);
        // also clear from selection
        setSelectedIds((prev) => prev.filter((x) => x !== patient.id));
      },
      onError: () => Swal.fire('Error', 'Failed to delete the patient.', 'error'),
    });
  };

  // ===== Nurse: Start consultation =====
  const CONSULTATION_DRAFT_KEY = 'consultation_form_draft_v1';
  const handleStartConsultation = async (patient: any) => {
    const fullName = [patient.first_name, patient.middle_name, patient.surname]
      .filter(Boolean)
      .join(' ');

    const { isConfirmed } = await Swal.fire({
      title: 'Start Consultation?',
      html: `Do you want to start a consultation for <b>${fullName}</b>?`,
      icon: 'question',
      showCancelButton: true,
      confirmButtonText: 'Yes, start',
      cancelButtonText: 'Cancel',
    });
    localStorage.removeItem(CONSULTATION_DRAFT_KEY);
    if (!isConfirmed) return;

    router.visit(route('consultations.create', { patient_id: patient.id }));
  };

  // Debounced search
  const firstRun = useRef(true);
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    router.get(
      `${API}/patients`,
      {
        page: 1,
        search: debouncedSearch,
        gender,
        company_id: companyId,
        marital_status: maritalStatus,
        age_min: ageMin,
        age_max: ageMax,
      },
      { preserveState: true, replace: true },
    );
  }, [debouncedSearch]);

  // Helpers to display DOB and Age
  const renderDobAge = (dob?: string | null) => {
    if (!dob) return '-';
    const m = moment(dob, moment.ISO_8601, true);
    if (!m.isValid()) return '-';
    const age = moment().diff(m, 'years');
    return `${m.format('YYYY-MM-DD')} (${age} yrs)`;
  };

  // Stats with fallbacks
  const employeesTotal = stats?.employees_total ?? 0;
  const dependentsTotal = stats?.dependents_total ?? 0;
  const maleCount = stats?.gender?.male ?? 0;
  const femaleCount = stats?.gender?.female ?? 0;
  const unknownCount = stats?.gender?.unknown ?? 0;

  // ===== Bulk delete handler (superadmin only) =====
  const handleBulkDelete = async () => {
    if (selectedIds.length === 0) {
      Swal.fire('No selection', 'Please select at least one patient.', 'info');
      return;
    }

    const { isConfirmed } = await Swal.fire({
      title: `Delete ${selectedIds.length} selected record(s)?`,
      icon: 'warning',
      html: `
        <div style="text-align:left">
          <p>This will permanently remove the selected patients <b>and all their dependents, consultations, and triages</b>.</p>
          <p style="margin-top:.5rem;"><b>This action cannot be undone.</b></p>
        </div>
      `,
      showCancelButton: true,
      confirmButtonText: 'Yes, delete selected',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      focusCancel: true,
    });

    if (!isConfirmed) return;

    Swal.fire({
      title: 'Deleting...',
      text: 'Please wait',
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      didOpen: () => Swal.showLoading(),
    });

    router.post(
      route('patients.bulk-destroy'),
      { ids: selectedIds },
      {
        preserveScroll: true,
        onFinish: () => Swal.close(),
        onSuccess: () => {
          // We’ll show a nice summary using flash below in useEffect
          // Clear current selection on success
          setSelectedIds([]);
          // Re-fetch current page
          applyFilters(patients.current_page ?? 1);
        },
        onError: (errors: any) => {
          const firstError = (errors && Object.values(errors)[0]) || 'Bulk delete failed.';
          Swal.fire('Error', String(firstError), 'error');
        },
      }
    );
  };

  // Show flash summary after redirect
  useEffect(() => {
    const successText = pageProps?.flash?.success as string | undefined;
    const result = pageProps?.flash?.bulk_delete_result as
      | { affected: number; company_counts: Record<string, number> }
      | undefined;

    if (successText && result) {
      const lines = Object.entries(result.company_counts || {}).map(
        ([name, count]) => `<li>${name}: <b>${count}</b></li>`
      );
      Swal.fire({
        icon: 'success',
        title: successText,
        html: `
          <div style="text-align:left">
            <p><b>Deleted:</b> ${result.affected}</p>
            <p><b>By Company:</b></p>
            <ul style="margin-left:1rem">${lines.join('')}</ul>
          </div>
        `,
      });
    } else if (pageProps?.flash?.error) {
      Swal.fire('Error', String(pageProps.flash.error), 'error');
    } else if (pageProps?.flash?.info) {
      Swal.fire('Info', String(pageProps.flash.info), 'info');
    }
  }, [pageProps?.flash]);

  // Enhanced search actions
  const handleClearSearch = () => setSearch('');
  const handleEnterSearch = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter') applyFilters(1);
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Patients', href: `${API}/patients` }]} >
      <Head title="Patients" />

      <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">
        <div className="mb-4 space-y-3">
          <div className="flex items-center justify-between">
            <h1 className="text-2xl font-bold">Database</h1>
            <div className="flex gap-2">
              {isSuperAdmin && selectedIds.length > 0 && (
                <Button variant="destructive" onClick={handleBulkDelete}>
                  <Trash2 className="mr-2 h-4 w-4" />
                  Delete Selected ({selectedIds.length})
                </Button>
              )}

              <Button asChild>
                <Link href={route('patients.create')}>
                  <Plus className="mr-2 h-4 w-4" />
                  Add Patient
                </Link>
              </Button>
              <Button variant="secondary" asChild>
                <Link href={route('dependents.create')}>
                  <Plus className="mr-2 h-4 w-4" />
                  Add Dependent
                </Link>
              </Button>
            </div>
          </div>

          {/* Explanatory note */}
          <div className="rounded-md border border-blue-200 bg-blue-50 p-3 text-sm text-blue-800 dark:border-blue-900/40 dark:bg-blue-950/40 dark:text-blue-200">
            <p className="flex items-start gap-2">
              <Info className="mt-0.5 h-4 w-4 shrink-0" />
              <span>
                This database lists <strong>all employees</strong> (and their dependents) in the system—whether they have been
                <strong> consulted</strong> or <strong>not</strong>. Use the actions above to add new records.
              </span>
            </p>
          </div>

          {/* Summary cards */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-2 lg:grid-cols-5">
            <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
              <div className="text-sm text-muted-foreground">Employees (total)</div>
              <div className="mt-1 flex items-center gap-2">
                <Users className="h-4 w-4" />
                <div className="text-2xl font-semibold">{employeesTotal}</div>
              </div>
            </div>
            <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
              <div className="text-sm text-muted-foreground">Dependents (total)</div>
              <div className="mt-1 flex items-center gap-2">
                <User className="h-4 w-4" />
                <div className="text-2xl font-semibold">{dependentsTotal}</div>
              </div>
            </div>
            <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
              <div className="text-sm text-muted-foreground">Male (employees)</div>
              <div className="mt-1 text-2xl font-semibold">{maleCount}</div>
            </div>
            <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
              <div className="text-sm text-muted-foreground">Female (employees)</div>
              <div className="mt-1 text-2xl font-semibold">{femaleCount}</div>
            </div>
            <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
              <div className="text-sm text-muted-foreground">Unknown/Other (employees)</div>
              <div className="mt-1 text-2xl font-semibold">{unknownCount}</div>
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="mb-6 grid grid-cols-1 gap-3 md:grid-cols-2 lg:grid-cols-4 xl:grid-cols-5 2xl:grid-cols-6">
          {/* BIG SEARCH BAR */}
          <div className="relative col-span-full rounded-xl border bg-white/70 p-2 shadow-sm ring-1 ring-border dark:bg-neutral-900/60">
            <div className="relative">
              <SearchIcon className="text-muted-foreground absolute left-4 top-1/2 h-5 w-5 -translate-y-1/2" />
              <Input
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                onKeyDown={handleEnterSearch}
                placeholder="Search patients by name, ID number, or employee number…"
                className="h-12 w-full rounded-xl pl-12 pr-[8.5rem] text-base"
              />

              {/* Clear button inside input */}
              {search && (
                <Button
                  type="button"
                  variant="ghost"
                  className="absolute right-28 top-1/2 -translate-y-1/2 h-8 w-8 rounded-full"
                  onClick={handleClearSearch}
                  title="Clear"
                >
                  <X className="h-4 w-4" />
                </Button>
              )}

              {/* Inline search button */}
              <Button
                type="button"
                className="absolute right-2 top-1/2 -translate-y-1/2 h-10 rounded-lg px-4"
                onClick={() => applyFilters(1)}
                title="Search"
              >
                <SearchIcon className="mr-2 h-4 w-4" />
                Search
              </Button>
            </div>
            <div className="mt-1 px-1 text-xs text-muted-foreground">
              Tip: Press <kbd className="rounded border px-1">Enter</kbd> to search • Hyphens/spaces in IDs are ignored.
            </div>
          </div>

          <Select value={gender || undefined} onValueChange={setGender}>
            <SelectTrigger className="h-11">
              <SelectValue placeholder="All Genders" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="Male">Male</SelectItem>
              <SelectItem value="Female">Female</SelectItem>
            </SelectContent>
          </Select>

          <Select value={maritalStatus || undefined} onValueChange={setMaritalStatus}>
            <SelectTrigger className="h-11">
              <SelectValue placeholder="All Marital Statuses" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="Single">Single</SelectItem>
              <SelectItem value="Married">Married</SelectItem>
              <SelectItem value="Divorced">Divorced</SelectItem>
              <SelectItem value="Widowed">Widowed</SelectItem>
            </SelectContent>
          </Select>

          <Select value={companyId || undefined} onValueChange={setCompanyId}>
            <SelectTrigger className="h-11">
              <SelectValue placeholder="All Companies" />
            </SelectTrigger>
            <SelectContent>
              {companies.map((c: any) => (
                <SelectItem key={c.id} value={String(c.id)}>
                  {c.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>

          <Input
            type="number"
            placeholder="Min Age"
            value={ageMin}
            onChange={(e) => setAgeMin(e.target.value)}
            className="h-11"
          />
          <Input
            type="number"
            placeholder="Max Age"
            value={ageMax}
            onChange={(e) => setAgeMax(e.target.value)}
            className="h-11"
          />

          <div className="col-span-full flex flex-wrap gap-2 pt-1">
            <Button onClick={() => applyFilters()}>Apply Filters</Button>
            <Button variant="outline" onClick={resetFilters}>Reset</Button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted">
              <tr>
                {isSuperAdmin && (
                  <th className="px-4 py-2">
                    <input
                      type="checkbox"
                      checked={allSelectedOnPage}
                      onChange={toggleAllOnPage}
                    />
                  </th>
                )}
                <th className="px-4 py-2 text-left">Employee Number</th>
                <th className="px-4 py-2 text-left">Name</th>
                <th className="px-4 py-2 text-left">Gender</th>
                <th className="px-4 py-2 text-left">Date of Birth (Age)</th>
                <th className="px-4 py-2 text-left">Occupation</th>
                <th className="px-4 py-2 text-left">Phone</th>
                <th className="px-4 py-2 text-left">Company</th>
                <th className="px-4 py-2 text-left">Dependents</th>
                <th className="px-4 py-2 text-left">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              {patients.data.map((patient: any) => {
                const fullName = [patient.first_name, patient.middle_name, patient.surname]
                  .filter(Boolean)
                  .join(' ');
                const dependentsCount = patient.dependents_count ?? 0;
                const checked = selectedIds.includes(patient.id);

                return (
                  <tr key={patient.id}>
                    {isSuperAdmin && (
                      <td className="px-4 py-2">
                        <input
                          type="checkbox"
                          checked={checked}
                          onChange={() => toggleOne(patient.id)}
                        />
                      </td>
                    )}
                    <td className="px-4 py-2">{patient.employee_number ?? '-'}</td>
                    <td className="px-4 py-2">{fullName}</td>
                    <td className="px-4 py-2">{patient.gender ?? '-'}</td>
                    <td className="px-4 py-2">{renderDobAge(patient.date_of_birth)}</td>
                    <td className="px-4 py-2">
                      {patient.occupation ? (
                        <Badge className="whitespace-nowrap bg-green-100 text-green-800 hover:bg-green-100 dark:bg-green-900/20 dark:text-green-300">
                          {patient.occupation}
                        </Badge>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="px-4 py-2">{patient.phone ?? '-'}</td>
                    <td className="px-4 py-2">
                      {patient.company ? (
                        <Badge variant="secondary">{patient.company.name}</Badge>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="px-4 py-2">
                      <Badge variant="outline" className="text-xs">
                        {dependentsCount}
                      </Badge>
                    </td>
                    <td className="px-4 py-2">
                      <div className="flex flex-wrap items-center gap-2">
                        <Button asChild size="icon" title="View Patient">
                          <Link href={route('patients.show', patient.id)}>
                            <Eye className="h-4 w-4" />
                          </Link>
                        </Button>

                        <Button asChild variant="secondary" size="icon" title="Edit Patient">
                          <Link href={route('patients.edit', patient.id)}>
                            <Pencil className="h-4 w-4" />
                          </Link>
                        </Button>

                        {patient.latest_consultation_id ? (
                          <Button
                            asChild
                            variant='outline'
                            size="sm"
                            title="View Last Consultation"
                            className="inline-flex items-center gap-2"
                          >
                            <Link href={route('consultations.show', patient.latest_consultation_id)}>
                              <Stethoscope className="h-4 w-4" />
                              <span>View Last Consultation</span>
                            </Link>
                          </Button>
                        ) : (
                          <Badge variant="outline" className="text-xs">
                            No Consultation
                          </Badge>
                        )}

                        {isNurse && (
                          <Button
                            size="sm"
                            title="Start Consultation"
                            onClick={() => handleStartConsultation(patient)}
                            style={{ cursor: 'pointer' }}
                          >
                            Start Consultation
                          </Button>
                        )}

                        {isSuperAdmin && (
                          <Button
                            variant="destructive"
                            size="icon"
                            title="Delete Patient"
                            onClick={() => handleDelete(patient)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        )}
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={patients.last_page}
            forcePage={patients.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 border rounded hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 border rounded hover:bg-muted"
            nextClassName="px-3 py-1 border rounded hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
