import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import axios from 'axios';

import { ClipboardList, Eye, Pencil, Plus } from 'lucide-react';
import moment from 'moment';
import { useState, useEffect, useMemo } from 'react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCaption,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Separator } from '@/components/ui/separator';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

interface PageProps extends InertiaPageProps {
  triages: Array<{
    id: number;
    triage_level: string;
    created_at: string;
    patient: { id: number; first_name: string; surname: string };
    nurse: { id: number; name: string } | null;
    consultor: { id: number; name: string } | null;
  }>;
}

const DEFAULT_FUND_TAG = 'Providence Fund';

type PatientLite = {
  id: number;
  first_name: string;
  middle_name?: string | null;
  surname: string;
  employee_number?: string | null;
  company_id?: number | null;
  company?: { id: number; name: string } | null;
  parent_patient_id?: number | null;
  id_number?: string | null;
  gender?: string | null;
  date_of_birth?: string | null;
  phone?: string | null;
  dependents?: any[];
};

type FundMemberLite = {
  id: number;
  employee_number: string | null;
  first_name: string | null;
  middle_name: string | null;
  surname: string | null;
  company_id: number | null;
  company?: { id: number; name: string } | null;
  tag: string;
  created_at?: string;
};

type FundSummary =
  | { state: 'both'; member: FundMemberLite }
  | { state: 'fund_only'; member: FundMemberLite }
  | { state: 'not_on_fund' };

export default function SearchPage() {
  const { triages } = usePage<PageProps>().props;

  const [query, setQuery] = useState('');
  const [results, setResults] = useState<PatientLite[]>([]);
  const [selectedPatient, setSelectedPatient] = useState<PatientLite | null>(null);

  // Fund search results (from medical_fund_members)
  const [fundResults, setFundResults] = useState<FundMemberLite[]>([]);
  const [loading, setLoading] = useState(false);

  // Trigger both searches while typing (debounced here)
  useEffect(() => {
    const id = setTimeout(async () => {
      if (!query) {
        setResults([]);
        setFundResults([]);
        return;
      }

      setLoading(true);
      try {
        const [patientsRes, fundRes] = await Promise.all([
          axios.get(route('patient.find'), { params: { query } }), // existing endpoint
          axios.get(route('fund_members.search'), { params: { q: query, tag: DEFAULT_FUND_TAG } }),
        ]);

        setResults(patientsRes.data ?? []);
        setFundResults(fundRes.data ?? []);
      } catch {
        setResults([]);
        setFundResults([]);
      } finally {
        setLoading(false);
      }
    }, 300);

    return () => clearTimeout(id);
  }, [query]);

  const handleSelectPatient = (patient: any) => {
    setSelectedPatient(patient);
  };

  const isAccountHolder = !!selectedPatient && selectedPatient.parent_patient_id === null;

  // Decide what to show in the right-side Fund card
  const fundSummary: FundSummary = useMemo(() => {
    if (!query) return { state: 'not_on_fund' };

    if (fundResults.length === 0) return { state: 'not_on_fund' };

    // Build quick lookups for Patients for matching
    const byEmpNo = new Map<string, PatientLite[]>();
    for (const p of results) {
      const emp = (p.employee_number || '').trim();
      if (!emp) continue;
      const key = emp.toLowerCase();
      if (!byEmpNo.has(key)) byEmpNo.set(key, []);
      byEmpNo.get(key)!.push(p);
    }

    // Prefer exact employee_number matches
    for (const fm of fundResults) {
      const emp = (fm.employee_number || '').trim().toLowerCase();
      if (emp && byEmpNo.has(emp)) {
        return { state: 'both', member: fm };
      }
    }

    // Next, try name + (optional) company match
    const norm = (s: string | null | undefined) => (s || '').trim().toUpperCase();
    for (const fm of fundResults) {
      const fmFirst = norm(fm.first_name);
      const fmLast = norm(fm.surname);
      const fmCoId = fm.company_id ?? null;

      const inPatients = results.some((p) => {
        const pFirst = norm(p.first_name);
        const pLast = norm(p.surname);
        const pCoId = p.company_id ?? null;
        const nameMatch = pFirst === fmFirst && pLast === fmLast;
        const companyOk = fmCoId && pCoId ? fmCoId === pCoId : true; // ignore if either missing
        return nameMatch && companyOk;
      });

      if (inPatients) {
        return { state: 'both', member: fm };
      }
    }

    // If we have any fund member but no patient matched => "fund only"
    return { state: 'fund_only', member: fundResults[0] };
  }, [query, results, fundResults]);

  // Helpers to display on the fund card
  const fmFullName = (fm: FundMemberLite) =>
    [fm.first_name, fm.middle_name, fm.surname].filter(Boolean).join(' ') || 'Member';

  const fmEmployeeNo = (fm: FundMemberLite) => fm.employee_number || '—';
  const fmCompanyName = (fm: FundMemberLite) => fm.company?.name || '—';

  return (
    <AppLayout breadcrumbs={[{ title: 'Patients', href: `${API}/patients` }]}>
      <Head title="Patients" />

      <div className="grid gap-6 sm:grid-cols-12">
        {/* Left */}
        <div className="col-span-12 space-y-6 sm:col-span-8">
          <div className="p-6">
            {/* Header actions */}
            <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
              <h1 className="text-xl font-bold">Search Account Holder / Dependent</h1>

              <div className="flex flex-wrap gap-3">
                <Button
                  onClick={() => router.visit(route('bulk.import.form'))}
                  className="gap-2"
                  style={{ cursor: 'pointer' }}
                >
                  <Plus className="h-4 w-4" />
                  Bulk Upload Holder / Dependent
                </Button>
                <Button
                  onClick={() => router.visit(route('triages.create'))}
                  className="gap-2"
                  style={{ cursor: 'pointer' }}
                >
                  <Plus className="h-4 w-4" />
                  Check In Holder / Dependent
                </Button>

                <Button
                  onClick={() => router.visit(route('patients.create'))}
                  className="gap-2"
                  variant="secondary"
                  style={{ cursor: 'pointer' }}
                >
                  <Plus className="h-4 w-4" />
                  Add Account Holder / Member
                </Button>

                <Button
                  onClick={() => router.visit(route('dependents.create'))}
                  className="gap-2"
                  variant="outline"
                  style={{ cursor: 'pointer' }}
                >
                  <Plus className="h-4 w-4" />
                  Add Dependent
                </Button>
              </div>
            </div>

            {/* Search box */}
            <Card className="mb-6">
              <CardHeader>
                <CardTitle>Patient Lookup</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-4">
                  <Input
                    className="sm:col-span-3"
                    placeholder="Search by name, surname, employee number..."
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                  />
                  <Button onClick={() => { /* explicit search button, optional */ }}>
                    Search
                  </Button>
                </div>
                {loading && (
                  <div className="mt-2 text-xs text-muted-foreground">
                    Searching patients and medical fund…
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Selected patient card (kept as-is; selection from table below) */}
            {selectedPatient && (
              <Card className="mb-4 border-blue-200 dark:border-blue-900/40">
                <CardHeader>
                  <div className="flex items-center justify-between gap-2">
                    <CardTitle className="text-base">
                      Selected: {selectedPatient.first_name} {selectedPatient.surname}{' '}
                      {isAccountHolder ? (
                        <Badge variant="secondary" className="ml-2">
                          Account Holder
                        </Badge>
                      ) : (
                        <Badge variant="outline" className="ml-2">
                          Dependent
                        </Badge>
                      )}
                    </CardTitle>

                    <div className="flex gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => router.visit(route('patients.show', selectedPatient.id))}
                        className="gap-2"
                      >
                        <Eye className="h-4 w-4" />
                        View
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => router.visit(route('patients.edit', selectedPatient.id))}
                        className="gap-2"
                      >
                        <Pencil className="h-4 w-4" />
                        Edit
                      </Button>
                      <Button
                        size="sm"
                        onClick={() => router.visit(route('triages.create', { patient_id: selectedPatient.id }))}
                        className="gap-2"
                      >
                        Checkin
                      </Button>
                    </div>
                  </div>
                </CardHeader>

                <CardContent>
                  <Tabs defaultValue="details">
                    <TabsList>
                      <TabsTrigger value="details">Details</TabsTrigger>
                      {isAccountHolder && <TabsTrigger value="dependents">Dependents</TabsTrigger>}
                    </TabsList>

                    <TabsContent value="details" className="mt-4">
                      <div className="grid grid-cols-1 gap-2 text-sm sm:grid-cols-3">
                        <div>
                          <span className="text-muted-foreground">Company:</span>{' '}
                          {selectedPatient.company?.name ?? '-'}
                        </div>
                        <div>
                          <span className="text-muted-foreground">Phone:</span>{' '}
                          {selectedPatient.phone ?? '-'}
                        </div>
                        <div>
                          <span className="text-muted-foreground">Gender:</span>{' '}
                          {selectedPatient.gender ?? '-'}
                        </div>
                        <div>
                          <span className="text-muted-foreground">ID:</span>{' '}
                          {selectedPatient.id_number ?? '-'}
                        </div>
                        <div>
                          <span className="text-muted-foreground">DOB:</span>{' '}
                          {selectedPatient.date_of_birth
                            ? moment(selectedPatient.date_of_birth).format('MMM D, YYYY')
                            : '-'}
                        </div>
                      </div>
                    </TabsContent>

                    {isAccountHolder && (
                      <TabsContent value="dependents" className="mt-4">
                        <div className="text-sm text-muted-foreground">Dependents list…</div>
                      </TabsContent>
                    )}
                  </Tabs>
                </CardContent>
              </Card>
            )}

            {/* Results table (patients) */}
            {results.length > 0 && (
              <div className="rounded-lg border">
                <Table>
                  <TableCaption>Click a row to preview details above.</TableCaption>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-[28%]">Name</TableHead>
                      <TableHead>Employee #</TableHead>
                      <TableHead>Company</TableHead>
                      <TableHead>Type</TableHead>
                      <TableHead className="text-right">Action</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {results.map((patient) => {
                      const isSelected = selectedPatient?.id === patient.id;
                      const fullName = [patient.first_name, patient.middle_name, patient.surname]
                        .filter(Boolean)
                        .join(' ');
                      return (
                        <TableRow
                          key={patient.id}
                          onClick={() => handleSelectPatient(patient)}
                          className={`cursor-pointer ${isSelected ? 'bg-muted/50' : ''}`}
                        >
                          <TableCell className="font-medium">{fullName}</TableCell>
                          <TableCell>{patient.employee_number ?? '-'}</TableCell>
                          <TableCell>{patient.company?.name ?? '-'}</TableCell>
                          <TableCell>
                            {patient.parent_patient_id ? (
                              <Badge variant="outline">Dependent</Badge>
                            ) : (
                              <Badge variant="secondary">Account Holder</Badge>
                            )}
                          </TableCell>
                          <TableCell className="text-right">
                            <Button
                              size="sm"
                              onClick={(e) => {
                                e.stopPropagation();
                                router.visit(route('triages.create', { patient_id: patient.id }));
                              }}
                              className="gap-2"
                            >
                              Checkin
                            </Button>
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              </div>
            )}
          </div>
        </div>

        {/* Right: Fund status + Today's Triages */}
        <div className="col-span-12 space-y-6 sm:col-span-4">
          {/* Fund status card reacts to typing (NOT selection) */}
          <div
            className="rounded-lg border bg-white p-6 shadow-sm dark:bg-neutral-900"
            style={{ marginTop: '40px', marginLeft: '10px', marginRight: '10px' }}
          >
            <h2 className="mb-4 text-xl font-semibold">Medical Fund</h2>

            {!query && (
              <div className="rounded-md border border-muted-foreground/20 bg-muted p-4 text-sm text-muted-foreground">
                Start typing to search…
              </div>
            )}

            {query && loading && (
              <div className="rounded-md border border-muted-foreground/20 bg-muted p-4 text-sm text-muted-foreground">
                Checking medical fund…
              </div>
            )}

            {query && !loading && fundSummary.state === 'both' && (
              <div className="rounded-md border border-emerald-200 bg-emerald-50 p-4 text-sm text-emerald-800 dark:border-emerald-900/40 dark:bg-emerald-900/20 dark:text-emerald-200">
                <div className="mb-1">
                  <Badge className="bg-emerald-600 hover:bg-emerald-600">
                    {fundSummary.member.tag || DEFAULT_FUND_TAG}
                  </Badge>
                </div>
                <div className="font-medium">{fmFullName(fundSummary.member)}</div>
                {/* NEW: Employee # + Company */}
                <div className="mt-2 grid grid-cols-1 gap-1 text-xs sm:grid-cols-2">
                  <div>
                    <span className="text-muted-foreground">Employee #:</span>{' '}
                    {fmEmployeeNo(fundSummary.member)}
                  </div>
                  <div>
                    <span className="text-muted-foreground">Company:</span>{' '}
                    {fmCompanyName(fundSummary.member)}
                  </div>
                </div>
                <div className="mt-1 text-xs opacity-80">On medical fund (also exists in Patients)</div>
              </div>
            )}

            {query && !loading && fundSummary.state === 'fund_only' && (
              <div className="rounded-md border border-amber-200 bg-amber-50 p-4 text-sm text-amber-800 dark:border-amber-900/40 dark:bg-amber-900/20 dark:text-amber-200">
                <div className="mb-1">
                  <Badge className="bg-amber-600 hover:bg-amber-600">
                    {fundSummary.member.tag || DEFAULT_FUND_TAG}
                  </Badge>
                </div>
                <div className="font-medium">{fmFullName(fundSummary.member)}</div>
                {/* NEW: Employee # + Company */}
                <div className="mt-2 grid grid-cols-1 gap-1 text-xs sm:grid-cols-2">
                  <div>
                    <span className="text-muted-foreground">Employee #:</span>{' '}
                    {fmEmployeeNo(fundSummary.member)}
                  </div>
                  <div>
                    <span className="text-muted-foreground">Company:</span>{' '}
                    {fmCompanyName(fundSummary.member)}
                  </div>
                </div>
                <div className="mt-2 flex gap-2">
                  <Button
                    size="sm"
                    onClick={() => {
                      router.visit(route('patients.create'));
                    }}
                    className="gap-2"
                  >
                    <Plus className="h-4 w-4" />
                    Add as Account Holder
                  </Button>
                </div>
                <div className="mt-1 text-xs opacity-80">Found only in Medical Fund members</div>
              </div>
            )}

            {query && !loading && fundSummary.state === 'not_on_fund' && (
              <div className="rounded-md border border-rose-200 bg-rose-50 p-4 text-sm text-rose-800 dark:border-rose-900/40 dark:bg-rose-900/20 dark:text-rose-200">
                <div className="font-medium">Not on medical fund</div>
              </div>
            )}
          </div>

          {/* Today's Triages */}
          <div
            className="rounded-lg border bg-white p-6 shadow-sm dark:bg-neutral-900"
            style={{ marginLeft: '10px', marginRight: '10px' }}
          >
            <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
              <ClipboardList className="h-5 w-5" />
              Today's Triages
            </h2>

            {triages && triages.length > 0 ? (
              <div className="space-y-4">
                {triages.map((triage) => (
                  <div
                    key={triage.id}
                    className="cursor-pointer rounded border p-4 transition hover:bg-gray-50 dark:border-gray-700 dark:hover:bg-gray-800"
                    onClick={() => router.visit(route('patients.show', triage.patient.id))}
                  >
                    <div className="flex items-center justify-between">
                      <h3 className="font-medium">
                        {triage.patient.first_name} {triage.patient.surname}
                      </h3>
                      <span
                        className={
                          triage.triage_level === 'Emergency'
                            ? 'inline-flex items-center rounded-full bg-red-600 px-2 py-0.5 text-xs font-medium text-white'
                            : triage.triage_level === 'Urgent'
                              ? 'inline-flex items-center rounded-full bg-yellow-500 px-2 py-0.5 text-xs font-medium text-white'
                              : 'inline-flex items-center rounded-full bg-gray-200 px-2 py-0.5 text-xs font-medium text-gray-800'
                        }
                      >
                        {triage.triage_level}
                      </span>
                    </div>
                    <div className="mt-2 flex items-center justify-between text-sm text-gray-600 dark:text-gray-400">
                      <span>{moment(triage.created_at).format('h:mm A')}</span>
                      {triage.consultor?.name ? (
                        <span className="text-xs">Consulting Nurse/Dr: {triage.consultor.name}</span>
                      ) : null}
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-center text-gray-500 dark:text-gray-400">No triages recorded today</p>
            )}
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
