import { Button } from '@/components/ui/button'; // Adjust this if your Button component is elsewhere
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, router, usePage } from '@inertiajs/react';
import { ImageIcon, ShieldAlert } from 'lucide-react';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';
import React, { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';
import Swal from 'sweetalert2';
import { type SharedData } from '@/types';

const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';
const SignatureUrl = 'http://127.0.0.1:8001';

interface Company {
  id: number;
  name: string;
}

interface Consultation {
  id: string;
  patient_id: string;
  doctor_id: string;
  clinic_id: string;
  consultation_date: string;
  created_at?: string | null;          // added so we can use an accurate attendance timestamp
  injury_on_duty: boolean;
  systolic: string;
  diastolic: string;
  temperature: string;
  heart_rate: string;
  respiratory_rate: string;
  oxygen_saturation: string;
  weight: string;
  height: string;
  bmi: string;
  blood_sugar: string;
  urine_test_result: string;
  hiv_status: string;
  pregnancy_test_result: string;
  presenting_complaint: string;
  clinical_notes: string;
  diagnosis: string;
  treatment_plan: string;
  prescription: string;
  follow_up_required: boolean;

  // Leave / duty (shared fields)
  sick_leave: boolean;
  restricted_duty: boolean;
  reason_for_leave: string;
  sick_leave_notes: string;
  number_of_days: string;

  // Referrals
  refer_for_radiology: boolean;
  refer_for_lab: boolean;
  refer_to_specialist: boolean;
  specialist_name: string;
  refer_to_casualty: boolean;
  refer_out_patient: boolean;
  referred_hospital?: string;

  follow_up_date: string;
  referral_notes: string;

  patient: {
    id: number;
    full_name: string;
    first_name: string;
    surname: string;
    employee_number: string;
    home_address: string;
    date_of_birth: string;
    email: string;
    gender: string;
    id_number: string;
  };
  doctor: {
    name: string;
    email: string;
    signature: string;
    qualifications: string;
  };
  clinic: {
    name: string;
    address: string;
    contact_person: string;
    email_address: string;
    city: string;
  };

  // privacy flags (optional)
  can_view?: boolean;
  can_edit?: boolean;
}

interface Dependent {
  id: number;
  first_name: string;
  middle_name?: string;
  surname: string;
  date_of_birth?: string;
  id_number?: string;
  relationship?: string;
}

interface Parent {
  id: number;
  first_name: string;
  surname: string;
}

interface Patient {
  id: number;
  first_name: string;
  middle_name?: string;
  surname: string;
  employee_number?: string;
  company?: Company;
  parent?: Parent;
  parent_patient_id?: number;
  date_of_birth?: string;
  email?: string;
  gender?: string;
  id_number?: string;
  medical_aid_number?: string;
  medical_aid_provider?: string;
  relationship?: string;
  phone?: string;

  home_address?: string;
  work_area?: string;
  suburb?: string;

  emergency_contact_name?: string;
  emergency_contact_relation?: string;
  emergency_contact_phone?: string;
  allergies?: string;
  is_smoker?: boolean;
  occupation?: string;
  marital_status?: string;
  dependents?: Dependent[];
  latest_consultation?: Consultation;
  reserved?: string; // Reserved field added
  image_url?: string; // Optional image URL field
}

type UserRole =
  | 'superadmin'
  | 'receptionist'
  | 'nurse'
  | 'doctor'
  | 'lab_technician'
  | 'radiologist'
  | 'pharmacist'
  | 'nurse_aid'
  | 'admin'
  | 'specialist'
  | 'casualty'
  | 'user'
  | 'patient';

export default function PatientShow() {
  const { patient } = usePage<{ patient: Patient }>().props;

  const page = usePage<SharedData>();
  const { auth } = page.props;

  const userRole: UserRole | undefined = auth.user?.role as UserRole | undefined;

  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });
  const CONSULTATION_DRAFT_KEY = 'consultation_form_draft_v1';

  const {
    first_name,
    middle_name,
    surname,
    employee_number,
    id_number,
    date_of_birth,
    gender,
    email,
    phone,
    company,
    occupation,
    marital_status,
    is_smoker,
    medical_aid_number,
    medical_aid_provider,
    allergies,
    emergency_contact_name,
    emergency_contact_relation,
    emergency_contact_phone,

    home_address,
    work_area,
    suburb,

    dependents = [],
    parent_patient_id,
    relationship,
    parent,
    reserved,
    image_url,
  } = patient;

  const fullName = [first_name, middle_name, surname].filter(Boolean).join(' ');

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: fullName, href: route('patients.show', patient.id) },
  ];

  const Info = ({ label, value }: { label: string; value?: string | null }) => (
    <div>
      <div className="text-muted-foreground text-sm">{label}</div>
      <div className="text-foreground font-medium">{value || '-'}</div>
    </div>
  );

  const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
    <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 md:grid-cols-3">
      {items.map((item) => (
        <Info key={item.label} {...item} />
      ))}
    </div>
  );

  const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
    <section className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
      <h2 className="text-foreground mb-4 text-xl font-semibold">{title}</h2>
      {children}
    </section>
  );

  // Start Consultation handler
  const handleStartConsultation = async () => {
    const res = await Swal.fire({
      title: 'Start a new consultation?',
      text: 'This will clear any saved consultation draft on this device.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, start',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#2563eb',
      reverseButtons: true,
      focusCancel: true,
    });

    if (res.isConfirmed) {
      try {
        localStorage.removeItem(CONSULTATION_DRAFT_KEY);
      } catch { /* ignore storage errors */ }
      router.visit(route('consultations.create', { patient_id: patient.id }));
    }
  };

  const handleWritePrescription = () => {
    if (!patient?.latest_consultation) {
      Swal.fire({
        title: 'No Consultation Found',
        text: 'Please create a consultation before writing a prescription.',
        icon: 'warning',
        confirmButtonText: 'OK',
      });
    } else {
      router.visit(route('consultations.prescriptions.create', patient.latest_consultation.id));
    }
  };

  // Attendance timestamp (prefer created_at if present, fallback to consultation_date)
  const latest = patient.latest_consultation || null;
  const canViewLatest = latest?.can_view !== false; // default to true if backend didn't send it
  const attendanceTimestamp =
    latest?.created_at || latest?.consultation_date || undefined;
  const attendanceFull = attendanceTimestamp ? moment(attendanceTimestamp).format('MMMM Do YYYY, h:mm A') : '-';

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`Patient: ${fullName}`} />

      {/* Hidden printable attendance content (SHORT SENTENCE) */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="p-6 font-sans text-sm text-gray-800">
          <style>
            {`
              @page {
                size: auto;
                margin: 15mm 10mm;
              }
              @media print {
                body {
                  -webkit-print-color-adjust: exact;
                  print-color-adjust: exact;
                }
                .section-break { page-break-after: always; }
                .avoid-break { page-break-inside: avoid; }
              }
            `}
          </style>

          {/* Header */}
          <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
            <div className="flex items-center">
              <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
            </div>
            <div className="text-center text-sm text-gray-800">
              <p className="text-lg font-bold">Clinic Attendance</p>
              <p>{attendanceFull}</p>
            </div>
            <div className="flex items-center">
              <QRCodeSVG value={`Attendance For Patient:${patient.id}`} size={60} />
            </div>
          </div>

          {/* Clinic */}
          <div className="mt-8 flex justify-between">
            <div className="flex-1"></div>
            <div className="flex-1 text-right">
              <h2 className="text-lg font-bold">Clinic Information</h2>
              <p>{latest?.clinic?.name || '-'}</p>
              <p>Clinic Address: {latest?.clinic?.address || '-'}</p>
              <p>Contact Person: {latest?.clinic?.contact_person || '-'}</p>
              <p>Email Address: {latest?.clinic?.email_address || '-'}</p>
            </div>
          </div>

          {/* Patient */}
          <div className="mt-4">
            <h2 className="text-lg font-bold">Patient Information</h2>
            <p>Patient Name: {first_name} {surname}</p>
            <p>Patient ID: {patient.id}</p>
            <p>
              ID Number:{' '}
              <span style={{ fontWeight: 'bold', color: '#000080' }}>
                {id_number || '-'}
              </span>
            </p>
          </div>

          {/* SHORT BODY TEXT */}
          <div className="mt-4">
            <p>
              <span style={{ fontWeight: 'bold', color: '#000080' }}>
                {first_name} {surname}
              </span>{' '}
              (National ID:{' '}
              <span style={{ fontWeight: 'bold', color: '#000080' }}>
                {id_number || '-'}
              </span>
              ) attended our health facility,{' '}
              <span style={{ fontWeight: 'bold', color: '#000080' }}>
                {latest?.clinic?.name || '-'}
              </span>
              , on{' '}
              <span style={{ fontWeight: 'bold', color: '#000080' }}>
                {attendanceFull}
              </span>
              .
            </p>
          </div>

          {/* Doctor */}
          <div className="mt-4" style={{ marginTop: '6rem' }}>
            <h2 className="text-lg font-bold">Consultor's Information</h2>
            <p>Doctor's / RGN's Name: {latest?.doctor?.name || '-'}</p>
            <p>Doctor's Signature</p>
            <img
              src={
                latest?.doctor?.signature
                  ? `${SignatureUrl}/storage/${latest?.doctor?.signature}`
                  : '/placeholder-signature.png'
              }
              alt="Doctor Signature"
              style={{ height: '4rem' }}
            />
          </div>

          {/* Footer */}
          <div className="mt-auto pt-8" style={{ marginTop: '7rem' }}>
            <div className="flex justify-between">
              <div>
                <p>Date: {moment().format('MMMM Do YYYY, h:mm A')}</p>
              </div>
              <div className="h-24 w-34 border-2 border-dashed border-gray-300"></div>
            </div>
          </div>
        </div>
      </div>

      <div className="w-full space-y-8 px-4 py-10 sm:px-6">
        {/* Header: Image + Name + Buttons */}
        <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
          <div className="flex items-center gap-4">
            {image_url ? (
              <img src={image_url} alt="Patient" className="h-24 w-24 rounded-full border object-cover" />
            ) : (
              <div className="bg-muted text-muted-foreground flex h-24 w-24 items-center justify-center rounded-full border">
                <ImageIcon className="h-8 w-8" />
              </div>
            )}
            <div>
              <h1 className="text-foreground text-2xl font-bold">{fullName}</h1>
              {parent_patient_id === null ? (
                <div className="mt-1 inline-block rounded-full bg-green-100 px-3 py-1 text-sm font-medium text-green-800">
                  Account Holder
                </div>
              ) : (
                <div className="mt-1 inline-block rounded-full bg-red-100 px-3 py-1 text-sm font-medium text-red-800">
                  Dependent
                </div>
              )}
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex flex-wrap gap-2">
            <Button
              onClick={() => (window.location.href = route('dependents.create', { parent_id: patient.id }))}
              variant="outline"
              style={{ cursor: 'pointer' }}
            >
              Add Dependent
            </Button>

            {userRole && ['superadmin', 'nurse', 'doctor'].includes(userRole) && (
              <>
                <div className="flex flex-wrap gap-2">
                  <Button
                    variant="secondary"
                    onClick={handleStartConsultation}
                    style={{ cursor: 'pointer' }}
                  >
                    Start Consultation
                  </Button>
                </div>
                <Button onClick={handleWritePrescription} style={{ cursor: 'pointer' }}>
                  Write Prescription
                </Button>
              </>
            )}

            <Button onClick={reactToPrintFn} variant="outline" style={{ cursor: 'pointer' }}>
              Print Attendance Slip
            </Button>
            <Button
              onClick={() => alert('Export clicked')}
              variant="destructive"
              style={{ cursor: 'pointer' }}
            >
              Export Record
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-12 gap-8">
          <div className="col-span-8 space-y-6">
            <Section title="Personal Information">
              <InfoGrid
                items={[
                  { label: 'Full Name', value: fullName },
                  { label: 'ID Number', value: id_number },
                  { label: 'Date of Birth', value: date_of_birth ? moment(date_of_birth).format('DD MMM YYYY') : '-' },
                  { label: 'Gender', value: gender },
                  { label: 'Email', value: email },
                  { label: 'Phone', value: phone },
                  { label: 'Marital Status', value: marital_status },
                ]}
              />
            </Section>

            <Section title="Home Address & Work">
              <InfoGrid
                items={[
                  { label: 'Home Address', value: home_address },
                  { label: 'Suburb', value: suburb },
                  { label: 'Work Place', value: work_area },
                ]}
              />
            </Section>

            <Section title="Employment">
              <InfoGrid
                items={[
                  { label: 'Employee Number', value: employee_number },
                  { label: 'Company', value: company?.name },
                  { label: 'Occupation', value: occupation },
                ]}
              />
            </Section>

            <Section title="Medical & Lifestyle">
              <InfoGrid
                items={[
                  { label: 'Medical Aid Number', value: medical_aid_number },
                  { label: 'Medical Aid Provider', value: medical_aid_provider },
                  { label: 'Allergies', value: allergies },
                  { label: 'Smoker', value: is_smoker ? 'Yes' : 'No' },
                ]}
              />
            </Section>

            <Section title="Emergency Contact">
              <InfoGrid
                items={[
                  { label: 'Name', value: emergency_contact_name },
                  { label: 'Relation', value: emergency_contact_relation },
                  { label: 'Phone', value: emergency_contact_phone },
                ]}
              />
            </Section>
          </div>

          <div className="col-span-4">
            {parent_patient_id ? (
              <>
                <div className="space-y-4" style={{ marginBottom: '10px' }}>
                  <div className="bg-card rounded-lg border p-6 shadow-sm">
                    <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                    {userRole && ['superadmin', 'nurse', 'doctor'].includes(userRole) && (
                      <Button asChild className="w-full">
                        <a href={route('patients.consultations', patient.id)}>Consultation History</a>
                      </Button>
                    )}
                  </div>
                </div>

                {/* Latest consultation card ONLY if accessible */}
                {latest && canViewLatest && (
                  <Section title="Latest Consultation">
                    <div className="space-y-2 text-sm">
                      <div>
                        <span className="text-muted-foreground">Date • Time: </span>
                        <span className="font-medium">
                          {attendanceFull}
                        </span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Clinic: </span>
                        <span className="font-medium">{latest.clinic?.name || '-'}</span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Consulted By: </span>
                        <span className="font-medium">{latest.doctor?.name || '-'}</span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Injury On Duty: </span>
                        <span className="font-medium">{latest.injury_on_duty ? 'Yes' : 'No'}</span>
                      </div>
                      <div className="mt-2">
                        <Button asChild variant="link" className="h-auto p-0">
                          <a href={route('consultations.show', latest.id)}>View Consultation</a>
                        </Button>
                      </div>
                    </div>
                  </Section>
                )}

                {/* Dependents */}
                <Section title="Dependents">
                  {dependents.length > 0 ? (
                    <ul className="divide-border divide-y">
                      {dependents.map((dep) => (
                        <li key={dep.id} className="py-3">
                          <div className="text-foreground font-medium">
                            {[dep.first_name, dep.middle_name, dep.surname].filter(Boolean).join(' ')}
                          </div>
                          <div className="text-muted-foreground text-sm">
                            DOB: {dep.date_of_birth ? moment(dep.date_of_birth).format('DD MMM YYYY') : '-'} | ID: {dep.id_number || '-'}
                          </div>
                          <div className="text-muted-foreground text-sm">Relationship: {dep.relationship || '-'}</div>
                          <div className="text-foreground font-medium">
                            <Button
                              variant="link"
                              className="text-primary ml-2 h-auto p-0"
                              onClick={() => (window.location.href = route('dependents.show', dep.id))}
                            >
                              View
                            </Button>
                          </div>
                        </li>
                      ))}
                    </ul>
                  ) : (
                    <p className="text-muted-foreground">No dependents registered.</p>
                  )}
                </Section>
              </>
            ) : (
              <>
                <div className="space-y-4" style={{ marginBottom: '10px' }}>
                  <div className="bg-card rounded-lg border p-6 shadow-sm">
                    <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                    {userRole && ['superadmin', 'nurse', 'doctor'].includes(userRole) && (
                      <Button asChild className="mb-2 w-full">
                        <a href={route('patients.consultations', patient.id)}>Consultation History</a>
                      </Button>
                    )}
                    <Button asChild variant="outline" className="w-full">
                      <a href={route('patients.edit', patient.id)}>Edit Patient</a>
                    </Button>
                  </div>
                </div>

                {/* Latest consultation card ONLY if accessible */}
                {latest && canViewLatest && (
                  <Section title="Latest Consultation">
                    <div className="space-y-2 text-sm">
                      <div>
                        <span className="text-muted-foreground">Date • Time: </span>
                        <span className="font-medium">
                          {attendanceFull}
                        </span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Clinic: </span>
                        <span className="font-medium">{latest.clinic?.name || '-'}</span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Consulted By: </span>
                        <span className="font-medium">{latest.doctor?.name || '-'}</span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Injury On Duty: </span>
                        <span className="font-medium">{latest.injury_on_duty ? 'Yes' : 'No'}</span>
                      </div>
                      <div className="mt-2">
                        <Button asChild variant="link" className="h-auto p-0">
                          <a href={route('consultations.show', latest.id)}>View Consultation</a>
                        </Button>
                      </div>
                    </div>
                  </Section>
                )}

                {/* Dependents */}
                <Section title="Dependents">
                  {dependents.length > 0 ? (
                    <ul className="divide-border divide-y">
                      {dependents.map((dep) => (
                        <li key={dep.id} className="py-3">
                          <div className="text-foreground font-medium">
                            {[dep.first_name, dep.middle_name, dep.surname].filter(Boolean).join(' ')}
                          </div>
                          <div className="text-muted-foreground text-sm">
                            DOB: {dep.date_of_birth ? moment(dep.date_of_birth).format('DD MMM YYYY') : '-'} | ID: {dep.id_number || '-'}
                          </div>
                          <div className="text-muted-foreground text-sm">Relationship: {dep.relationship || '-'}</div>
                          <div className="text-foreground font-medium">
                            <Button
                              variant="link"
                              className="text-primary ml-2 h-auto p-0"
                              onClick={() => (window.location.href = route('dependents.show', dep.id))}
                            >
                              View
                            </Button>
                          </div>
                        </li>
                      ))}
                    </ul>
                  ) : (
                    <p className="text-muted-foreground">No dependents registered.</p>
                  )}
                </Section>
              </>
            )}
          </div>
        </div>

        {/* Reserved Section (kept for future use) */}
      </div>
    </AppLayout>
  );
}
