import AppLayout from '@/layouts/app-layout';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import { Loader2, Pill, Plus, Trash, XCircle } from 'lucide-react';
import { useEffect, useState } from 'react';
import Creatable from 'react-select/creatable';
import Swal from 'sweetalert2';
import moment from 'moment';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import PrintSickLeaveNote from '../Consultations/components/PrintSickLeaveNote';

interface Props extends PageProps {
    consultation: {
        id: string;
        patient_id: string;
        doctor_id: string;
        clinic_id: string;
        consultation_date: string;
        injury_on_duty: boolean;
        systolic: string;
        diastolic: string;
        temperature: string;
        heart_rate: string;
        respiratory_rate: string;
        oxygen_saturation: string;
        weight: string;
        height: string;
        bmi: string;
        blood_sugar: string;
        urine_test_result: string;
        hiv_status: string;
        pregnancy_test_result: string;
        presenting_complaint: string;
        clinical_notes: string;
        diagnosis: string;
        treatment_plan: string;
        prescription: string;
        follow_up_required: boolean;

        //New Fields
        sick_leave: boolean;
        reason_for_leave: string,
        sick_leave_notes: string,
        number_of_days: string;

        refer_for_radiology: boolean;
        refer_for_lab: boolean;
        refer_to_specialist: boolean;
        specialist_name: string;
        refer_to_casualty: boolean;
        refer_out_patient: boolean;
        //End of new fields

        follow_up_date: string;
        referral_notes: string;
        patient: {
            id: number;
            full_name: string;
            first_name: string;
            surname: string;
            employee_number: string;
            home_address: string;
            date_of_birth: string;
            email: string;
            gender: string;
            id_number: string;
        };
        doctor: {
            name: string;
            email: string;
            signature: string;
            qualifications: string;
        };
        clinic: {
            name: string;
            address: string;
            contact_person: string;
            email_address: string;
            city: string;
        };
    };
    medications: {
        id: number;
        name: string;
        dosage: string;
        form: string;
    }[];
    routes: string[];
    foodRelations: string[];
}

interface SelectedMedication {
    id: string;
    medication_id: string;
    name: string;
    dosage: string;
    form: string;
    frequency: string;
    duration: string;
    quantity: string;
    route: string;
    food_relation: string;
    notes: string;
}

interface PrescriptionFormData {
    consultation_id: number;
    patient_id: number;
    medications: Array<{
        medication_id: string;
        dosage: string;
        frequency: string;
        duration: string;
        quantity: string;
        route: string;
        food_relation: string;
        notes?: string;
    }>;
}

const PrescriptionCreate: React.FC = () => {
    const { consultation, medications, routes, foodRelations } = usePage<Props>().props;
    const [isDarkMode, setIsDarkMode] = useState(false);
    const [isLoading, setIsLoading] = useState(false);

    const [selectedMedications, setSelectedMedications] = useState<SelectedMedication[]>([]);
    const [currentMedication, setCurrentMedication] = useState<SelectedMedication>({
        id: '',
        medication_id: '',
        name: '',
        dosage: '',
        form: '',
        frequency: '',
        duration: '7',
        quantity: '',
        route: '',
        food_relation: '',
        notes: '',
    });

    const [selectedMedicationOption, setSelectedMedicationOption] = useState<any>(null); // Track the selected option

    const [autoCalculate, setAutoCalculate] = useState(true);



    // Add this inside PrescriptionCreate component (above return)
    const handleCloseFile = async (consultationId: number) => {
        const result = await Swal.fire({
            title: 'Close Consultation?',
            text: 'Are you sure you want to close this consultation?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, close it',
            cancelButtonText: 'Cancel',
        });
        if (!result.isConfirmed) return;

        Swal.fire({
            title: 'Closing Consultation...',
            text: 'Please wait',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading(),
        });

        router.post(route('consultations.close', consultationId), {}, {
            onSuccess: () => Swal.fire('Closed!', 'Consultation closed successfully.', 'success'),
            onError: () => Swal.fire('Error', 'Failed to close consultation.', 'error'),
        });
    };


    useEffect(() => {
        const checkDarkMode = () => {
            setIsDarkMode(document.documentElement.classList.contains('dark'));
        };
        checkDarkMode();

        const observer = new MutationObserver(checkDarkMode);
        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class'],
        });

        return () => observer.disconnect();
    }, []);

    const medicationOptions = medications.map((medication) => ({
        value: medication.id.toString(),
        label: `${medication.name} (${medication.dosage} ${medication.form})`,
        medication,
    }));

    const handleAddMedication = () => {
        if (
            !currentMedication.name ||
            !currentMedication.dosage ||
            !currentMedication.frequency ||
            !currentMedication.duration ||
            !currentMedication.quantity ||
            !currentMedication.route ||
            !currentMedication.food_relation
        ) {
            Swal.fire({
                title: 'Incomplete Information',
                text: 'Please fill all required fields',
                icon: 'warning',
                confirmButtonColor: isDarkMode ? '#4b5563' : '#3b82f6',
            });
            return;
        }

        setSelectedMedications([
            ...selectedMedications,
            {
                ...currentMedication,
                id: Date.now().toString(),
                medication_id: selectedMedicationOption?.value || '',
            },
        ]);

        // Reset current medication form
        setCurrentMedication({
            id: '',
            medication_id: '',
            name: '',
            dosage: '',
            form: '',
            frequency: '',
            duration: '3',
            quantity: '',
            route: '',
            food_relation: '',
            notes: '',
        });

        // Clear the selected medication option
        setSelectedMedicationOption(null);
    };

    const handleRemoveMedication = (id: string) => {
        setSelectedMedications(selectedMedications.filter((med) => med.id !== id));
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();

        if (selectedMedications.length === 0) {
            Swal.fire({
                title: 'No Medications',
                text: ' addPlease at least one medication',
                icon: 'warning',
                confirmButtonColor: isDarkMode ? '#4b5563' : '#3b82f6',
            });
            return;
        }

        setIsLoading(true);

        // Prepare the form data in the correct format for Inertia.js
        const formData: Record<string, any> = {
            consultation_id: consultation.id,
            patient_id: consultation.patient.id,
            medications: selectedMedications.map((med) => ({
                medication_id: med.medication_id,
                dosage: med.dosage,
                frequency: med.frequency,
                duration: med.duration,
                quantity: med.quantity,
                route: med.route,
                food_relation: med.food_relation,
                notes: med.notes,
            })),
        };

        router.post(route('prescriptions.store'), formData, {
            onSuccess: () => {
                setIsLoading(false);
                Swal.fire({
                    title: 'Success',
                    text: 'Prescription created successfully',
                    icon: 'success',
                    confirmButtonColor: isDarkMode ? '#4b5563' : '#3b82f6',
                });
                router.visit(route('consultations.show', consultation.id));
            },
            onError: (errors) => {
                setIsLoading(false);
                Swal.fire({
                    title: 'Error',
                    text: Object.values(errors).join('\n'),
                    icon: 'error',
                    confirmButtonColor: isDarkMode ? '#4b5563' : '#3b82f6',
                });
            },
        });
    };

    const handleMedicationChange = (selectedOption: any) => {
        setSelectedMedicationOption(selectedOption); // Update the selected option state
        const selectedMedication = selectedOption?.medication;

        if (selectedMedication) {
            setCurrentMedication({
                ...currentMedication,
                medication_id: selectedOption.value,
                name: selectedMedication.name,
                dosage: selectedMedication.dosage,
                form: selectedMedication.form,
            });
        } else if (selectedOption?.label) {
            // Handle new medication creation
            setCurrentMedication({
                ...currentMedication,
                medication_id: '',
                name: selectedOption.label,
                dosage: '',
                form: '',
            });
        }
    };

    const calculateQuantity = () => {
        if (!currentMedication.frequency || !currentMedication.duration || !currentMedication.dosage) return;

        const [dose] = currentMedication.dosage.split(' ');
        const doseValue = parseFloat(dose) || 0;
        const frequencyValue = parseFloat(currentMedication.frequency) || 0;
        const durationValue = parseFloat(currentMedication.duration) || 0;

        return Math.ceil(doseValue * frequencyValue * durationValue);
    };

    const selectStyles = {
        control: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            borderColor: isDarkMode ? '#374151' : '#d1d5db',
            color: isDarkMode ? '#fff' : '#000',
            '&:hover': {
                borderColor: isDarkMode ? '#4b5563' : '#9ca3af',
            },
        }),
        menu: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            color: isDarkMode ? '#fff' : '#000',
        }),
        option: (base: any, { isFocused }: any) => ({
            ...base,
            backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
            color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
            '&:active': {
                backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6',
            },
        }),
        singleValue: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
        input: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
        placeholder: (base: any) => ({
            ...base,
            color: isDarkMode ? '#9ca3af' : '#6b7280',
        }),
    };

    const Info = ({ label, value }: { label: string; value?: string | null }) => (
        <div>
            <div className="text-muted-foreground text-sm">{label}</div>
            <div className="text-foreground font-medium">{value || '-'}</div>
        </div>
    );

    const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 md:grid-cols-3">
            {items.map((item) => (
                <Info key={item.label} {...item} />
            ))}
        </div>
    );

    const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
        <section className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
            <h2 className="text-foreground mb-4 text-xl font-semibold">{title}</h2>
            {children}
        </section>
    );


    const breadcrumbs = [
        { title: 'Consultations', href: route('consultations.index') },
        {
            title: `Consultation #${consultation.id}`,
            href: route('consultations.show', consultation.id),
        },
        { title: 'Create Prescription', href: '#' },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Create Prescription" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">
                <h1 className="text-2xl font-bold">
                    Prescribe Medication for{' '}
                    <span className="capitalize">
                        {consultation.patient.first_name} {consultation.patient.surname}
                    </span>
                </h1>

                <div className="grid grid-cols-1 gap-8 lg:grid-cols-3">
                    <div className="lg:col-span-2">
                        <div className="bg-card rounded-lg border p-6 shadow-sm">
                            <form onSubmit={handleSubmit} className="space-y-6">
                                <div className="space-y-4">
                                    <h3 className="text-lg font-semibold">
                                        {selectedMedications.length > 0 ? ' AnotherAdd Medication' : 'Add Medication'}
                                    </h3>
                                    <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                                        <div className="sm:col-span-2">
                                            <label className="mb-1 block text-sm font-medium">Medication *</label>
                                            <Creatable
                                                options={medicationOptions}
                                                value={selectedMedicationOption}
                                                onChange={handleMedicationChange}
                                                placeholder="Search or type new medication..."
                                                styles={selectStyles}
                                                className="react-select-container"
                                                classNamePrefix="react-select"
                                                isSearchable
                                            />
                                        </div>

                                        <div>
                                            <label className="mb-1 block text-sm font-medium">Dosage *</label>
                                            <input
                                                type="text"
                                                value={currentMedication.dosage}
                                                onChange={(e) =>
                                                    setCurrentMedication({
                                                        ...currentMedication,
                                                        dosage: e.target.value,
                                                    })
                                                }
                                                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                            />
                                        </div>

                                        <div>
                                            <label className="mb-1 block text-sm font-medium">Frequency *</label>
                                            <select
                                                value={currentMedication.frequency}
                                                onChange={(e) => {
                                                    const frequency = e.target.value;
                                                    setCurrentMedication({
                                                        ...currentMedication,
                                                        frequency,
                                                    });
                                                    if (autoCalculate) {
                                                        const qty = calculateQuantity();
                                                        if (qty)
                                                            setCurrentMedication((prev) => ({
                                                                ...prev,
                                                                quantity: qty.toString(),
                                                            }));
                                                    }
                                                }}
                                                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                            >
                                                <option value="">Select frequency</option>
                                                <option value="1">Once daily</option>
                                                <option value="2">Twice daily</option>
                                                <option value="3">Three times daily</option>
                                                <option value="0.5">Every other day</option>
                                                <option value="1/7">Once weekly</option>
                                            </select>
                                        </div>

                                        <div>
                                            <label className="mb-1 block text-sm font-medium">Duration (days) *</label>
                                            <input
                                                type="number"
                                                min="1"
                                                value={currentMedication.duration}
                                                onChange={(e) => {
                                                    const duration = e.target.value;
                                                    setCurrentMedication({
                                                        ...currentMedication,
                                                        duration,
                                                    });
                                                    if (autoCalculate) {
                                                        const qty = calculateQuantity();
                                                        if (qty)
                                                            setCurrentMedication((prev) => ({
                                                                ...prev,
                                                                quantity: qty.toString(),
                                                            }));
                                                    }
                                                }}
                                                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                            />
                                        </div>

                                        <div>
                                            <label className="mb-1 block text-sm font-medium">Quantity *</label>
                                            <div className="flex items-center gap-2">
                                                <input
                                                    type="number"
                                                    min="1"
                                                    value={currentMedication.quantity}
                                                    onChange={(e) =>
                                                        setCurrentMedication({
                                                            ...currentMedication,
                                                            quantity: e.target.value,
                                                        })
                                                    }
                                                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                                />
                                                <button
                                                    type="button"
                                                    onClick={() => {
                                                        const qty = calculateQuantity();
                                                        if (qty)
                                                            setCurrentMedication((prev) => ({
                                                                ...prev,
                                                                quantity: qty.toString(),
                                                            }));
                                                    }}
                                                    className="rounded-md bg-blue-100 px-3 py-2 text-xs whitespace-nowrap text-blue-800 hover:bg-blue-200 dark:bg-blue-900/30 dark:text-blue-300 dark:hover:bg-blue-900/50"
                                                >
                                                    Calculate
                                                </button>
                                            </div>
                                            <div className="mt-1 flex items-center">
                                                <input
                                                    type="checkbox"
                                                    id="autoCalculate"
                                                    checked={autoCalculate}
                                                    onChange={(e) => setAutoCalculate(e.target.checked)}
                                                    className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500 dark:border-gray-600 dark:bg-gray-800"
                                                />
                                                <label htmlFor="autoCalculate" className="ml-2 text-xs text-gray-600 dark:text-gray-400">
                                                    Auto-calculate quantity
                                                </label>
                                            </div>
                                        </div>

                                        <div>
                                            <label className="mb-1 block text-sm font-medium">Route *</label>
                                            <select
                                                value={currentMedication.route}
                                                onChange={(e) =>
                                                    setCurrentMedication({
                                                        ...currentMedication,
                                                        route: e.target.value,
                                                    })
                                                }
                                                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                            >
                                                <option value="">Select route</option>
                                                {routes.map((route) => (
                                                    <option key={route} value={route}>
                                                        {route}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <label className="mb-1 block text-sm font-medium">Food Relation *</label>
                                            <select
                                                value={currentMedication.food_relation}
                                                onChange={(e) =>
                                                    setCurrentMedication({
                                                        ...currentMedication,
                                                        food_relation: e.target.value,
                                                    })
                                                }
                                                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                            >
                                                <option value="">Select relation</option>
                                                {foodRelations.map((relation) => (
                                                    <option key={relation} value={relation}>
                                                        {relation}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>

                                        <div className="sm:col-span-2">
                                            <label className="mb-1 block text-sm font-medium">Notes</label>
                                            <textarea
                                                value={currentMedication.notes}
                                                onChange={(e) =>
                                                    setCurrentMedication({
                                                        ...currentMedication,
                                                        notes: e.target.value,
                                                    })
                                                }
                                                rows={2}
                                                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                            />
                                        </div>
                                    </div>

                                    <div className="flex justify-end">
                                        <button
                                            type="button"
                                            onClick={handleAddMedication}
                                            className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-4 py-2 text-sm font-medium text-white hover:bg-green-700"
                                        >
                                            <Plus className="w4- h-4" />
                                            Add Medication
                                        </button>
                                    </div>
                                </div>

                                {selectedMedications.length > 0 && (
                                    <div className="flex justify-end gap-3 pt-4">
                                        <button
                                            type="button"
                                            onClick={() => router.visit(route('consultations.show', consultation.id))}
                                            className="rounded-lg border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50 dark:border-gray-600 dark:bg-gray-800 dark:text-gray-300 dark:hover:bg-gray-700"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            type="submit"
                                            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                                        >
                                            {isLoading ? <Loader2 className="h-4 w-4 animate-spin" /> : 'Create Prescription'} (
                                            {selectedMedications.length})
                                        </button>
                                    </div>
                                )}
                            </form>
                        </div>
                    </div>

                    <div className="lg:col-span-1">
                        <div className="bg-card rounded-lg border p-6 shadow-sm">

                            <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px', marginBottom: '10px' }}>
                                <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                                <Button onClick={() => handleCloseFile(Number(consultation.id))} className="mb-3 w-full" style={{ cursor: 'pointer' }}>
                                    Close Consultation
                                </Button>

                            </div>

                            <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                                <Pill className="text-muted-foreground h-5 w-5" />
                                Patient Information
                            </h2>


                            <div className="space-y-3">
                                <div>
                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">Patient Name</h3>
                                    <p className="capitalize">
                                        {consultation.patient.first_name} {consultation.patient.surname}
                                    </p>
                                </div>
                                <div>
                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">Consultation</h3>
                                    <p>#{consultation.id}</p>
                                </div>
                                {selectedMedications.length > 0 && (
                                    <div>
                                        <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">Medications</h3>
                                        <p>{selectedMedications.length} selected</p>
                                    </div>
                                )}
                            </div>

                        </div>

                        <div
                            className="bg-card rounded-lg border p-6 shadow-sm"
                            style={{
                                marginTop: '10px',
                            }}
                        >
                            {selectedMedications.length > 0 && (
                                <div className="space-y-4">
                                    <h3 className="text-lg font-semibold">Selected Medications</h3>
                                    <div className="space-y-3">
                                        {selectedMedications.map((med) => (
                                            <div key={med.id} className="flex items-center justify-between rounded-lg border p-3">
                                                <div>
                                                    <p className="font-medium">{med.name}</p>
                                                    <p className="text-sm text-gray-600 dark:text-gray-400">
                                                        {med.dosage} • {med.frequency} • {med.duration} days
                                                    </p>
                                                </div>
                                                <button
                                                    type="button"
                                                    onClick={() => handleRemoveMedication(med.id)}
                                                    className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300"
                                                >
                                                    <Trash className="h-5 w-5" />
                                                </button>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}
                        </div>

                        <div className="col-span-12 sm:col-span-4 mt-6">
                            <Section title="Consultation Information">
                                {consultation ? (
                                    <>
                                        <InfoGrid
                                            items={[
                                                {
                                                    label: 'Date',
                                                    value: moment(consultation.consultation_date).format('DD MMM YYYY, h:mm a'),
                                                },
                                                {
                                                    label: 'Doctor',
                                                    value: consultation.doctor?.name || '-',
                                                },
                                                {
                                                    label: 'Clinic',
                                                    value: consultation.clinic?.name || '-',
                                                },
                                                {
                                                    label: 'Height',
                                                    value: consultation.height ? `${consultation.height} cm` : '-',
                                                },
                                                {
                                                    label: 'Weight',
                                                    value: consultation.weight ? `${consultation.weight} kg` : '-',
                                                },
                                                {
                                                    label: 'BMI',
                                                    value: consultation.bmi ? consultation.bmi : '-',
                                                },
                                                {
                                                    label: 'Injury on Duty',
                                                    value: consultation.injury_on_duty ? 'Yes' : 'No',
                                                },
                                                {
                                                    label: 'Systolic BP',
                                                    value: consultation.systolic ? consultation.systolic : '-',
                                                },
                                                {
                                                    label: 'Diastolic BP',
                                                    value: consultation.diastolic ? consultation.diastolic : '-',
                                                },
                                            ]}
                                        />

                                        {consultation.presenting_complaint && (
                                            <div className="mt-4">
                                                <div className="text-muted-foreground text-sm">Presenting Complaint</div>
                                                <div className="text-foreground rounded border p-3 font-medium">
                                                    {consultation.presenting_complaint}
                                                </div>
                                            </div>
                                        )}

                                        {consultation.diagnosis && (
                                            <div className="my-4">
                                                <div className="text-muted-foreground text-sm">Diagnosis</div>
                                                <div className="text-foreground rounded border p-3 font-medium">
                                                    {consultation.diagnosis}
                                                </div>
                                            </div>
                                        )}

                                        {consultation.clinical_notes && (
                                            <div className="my-4">
                                                <div className="text-muted-foreground text-sm">Clinical Notes</div>
                                                <div className="text-foreground rounded border p-3 font-medium">
                                                    {consultation.clinical_notes}
                                                </div>
                                            </div>
                                        )}

                                        {/* Sick leave details */}
                                        {consultation.sick_leave ? (
                                            <div className="my-4">
                                                <div className="text-muted-foreground text-sm">Sick Leave Notes</div>
                                                <div className="text-foreground rounded border p-3 font-medium">
                                                    {consultation.sick_leave_notes || 'No notes provided'}
                                                </div>

                                                <div className="mt-2 flex space-x-4">
                                                    <div>
                                                        <div className="text-muted-foreground text-sm">Number of Days</div>
                                                        <div className="text-foreground font-medium">{consultation.number_of_days}</div>
                                                    </div>
                                                    <div>
                                                        <div className="text-muted-foreground text-sm">Reason for Leave</div>
                                                        <div className="text-foreground font-medium">{consultation.reason_for_leave}</div>
                                                    </div>
                                                </div>
                                            </div>
                                        ) : (
                                            <p className="text-muted-foreground mt-4">No sick leave note available</p>
                                        )}
                                        <div style={{
                                            marginTop: '10px',

                                        }}></div>
                                        {
                                            consultation.sick_leave ? (
                                                <PrintSickLeaveNote
                                                    consultation={consultation}
                                                />
                                            ) : (
                                                <p className="text-muted-foreground">No sick leave note available</p>
                                            )
                                        }

                                        <Button
                                            variant="outline"
                                            className="w-full mt-6"
                                            onClick={() => router.visit(route('consultations.show', consultation.id))}
                                            style={{ cursor: 'pointer' }}
                                        >
                                            View Full Consultation
                                        </Button>
                                    </>
                                ) : (
                                    <div className="text-muted-foreground">No consultation found</div>
                                )}
                            </Section>
                        </div>




                    </div>
                </div>
            </div>
        </AppLayout>
    );
};

export default PrescriptionCreate;
