import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Printer, Search } from 'lucide-react';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';
import { useRef, useState } from 'react';
import ReactPaginate from 'react-paginate';
import { useReactToPrint } from 'react-to-print';
import Swal from 'sweetalert2';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Prescriptions', href: '/prescriptions' }];

interface Prescription {
    id: number;
    patient: {
        id: number;
        first_name: string;
        surname: string;
        employee_number: string;
        date_of_birth: string;
        phone: string;
        gender: string;
        id_number: string;
    };
    medication: {
        id: number;
        name: string;
    };
    prescriber: {
        id: number;
        name: string;
    };
    consultation: {
        id: number;
        clinic: {
            id: number;
            name: string;
            address?: string; // add more if needed
        };
    };
    consultation_id: number;
    prescribed_by: number;
    dosage: string;
    frequency: string;
    duration: string;
    quantity: number;
    route: string;
    food_relation: string;
    notes: string | null;
}

interface PagePropsWithPrescriptions extends PageProps {
    prescriptions: {
        data: Prescription[];
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    filters: {
        search?: string;
        prescriber?: string;
    };
    prescribers: Array<{
        id: number;
        name: string;
    }>;
}

export default function PrescriptionList() {
    const { prescriptions, filters, prescribers } = usePage<PagePropsWithPrescriptions>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');
    const [selectedPrescriber, setSelectedPrescriber] = useState(filters.prescriber || '');

    const contentRef = useRef<HTMLDivElement>(null);
    const reactToPrintFn = useReactToPrint({ contentRef });

    const handleSearch = () => {
        router.get(`${API}/prescriptions`, { search: searchQuery, prescriber: selectedPrescriber }, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(`${API}/prescriptions`, { page: selected.selected + 1, search: searchQuery, prescriber: selectedPrescriber }, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This prescription will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('prescriptions.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The prescription has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    // Group prescriptions by consultation_id
    const groupedPrescriptions = prescriptions.data.reduce(
        (acc, prescription) => {
            if (!acc[prescription.consultation_id]) {
                acc[prescription.consultation_id] = [];
            }
            acc[prescription.consultation_id].push(prescription);
            return acc;
        },
        {} as { [key: number]: Prescription[] },
    );

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Prescriptions" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Prescriptions</h1>
                </div>

                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            placeholder="Search prescriptions..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                    <button onClick={handleSearch} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                        Search
                    </button>
                </div>

                <div className="mb-4">
                    <select
                        value={selectedPrescriber}
                        onChange={(e) => setSelectedPrescriber(e.target.value)}
                        className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-3 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                    >
                        <option value="">Filter by Prescriber</option>
                        {prescribers.map((prescriber) => (
                            <option key={prescriber.id} value={prescriber.id.toString()}>
                                {prescriber.name}
                            </option>
                        ))}
                    </select>
                </div>

                {Object.keys(groupedPrescriptions).map((consultationId) => {
                    const consultationPrescriptions = groupedPrescriptions[parseInt(consultationId)];
                    const patient = consultationPrescriptions[0].patient;
                    const prescriberName = consultationPrescriptions[0].prescriber.name;
                    const consultation = consultationPrescriptions[0].consultation;
                    const clinic = consultation?.clinic;

                    return (
                        <div key={consultationId}>
                            {/* Print Prescription Section */}
                            <div style={{ display: 'none' }}>
                                <div ref={contentRef} className="p-6 font-sans text-sm text-gray-800">
                                    <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
                                        <div className="flex items-center">
                                            <img
                                                src="https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png"
                                                alt="PHC Logo"
                                                style={{
                                                    height: '4rem',
                                                }}
                                            />
                                        </div>
                                        <div className="text-center text-sm text-gray-800">
                                            <p className="font-semibold">Clinic: {clinic?.name}</p>
                                            {clinic?.address && <p>{clinic.address}</p>}
                                        </div>
                                        <div className="flex items-center">
                                            <QRCodeSVG value={`Consultation ID: ${consultationId}`} size={60} />
                                        </div>
                                    </div>

                                    <div className="mt-4">
                                        <p className="mb-2 text-lg font-bold">Patient Information</p>
                                        <div className="grid grid-cols-2 gap-y-2">
                                            <p className="font-semibold">Name:</p>
                                            <p>
                                                {patient.first_name} {patient.surname}
                                            </p>

                                            <p className="font-semibold">Employee Number:</p>
                                            <p>{patient.employee_number}</p>

                                            <p className="font-semibold">Date of Birth:</p>
                                            <p>{moment(patient.date_of_birth).format('DD MMMM YYYY')}</p>

                                            <p className="font-semibold">Phone:</p>
                                            <p>{patient.phone}</p>

                                            <p className="font-semibold">Gender:</p>
                                            <p>{patient.gender}</p>

                                            <p className="font-semibold">ID Number:</p>
                                            <p>{patient.id_number}</p>
                                        </div>
                                    </div>

                                    <div className="mt-6">
                                        <table className="w-full border-collapse text-xs">
                                            <thead>
                                                <tr>
                                                    <th className="border p-2 text-left">Medication</th>
                                                    <th className="border p-2 text-left">Dosage</th>
                                                    <th className="border p-2 text-left">Frequency</th>
                                                    <th className="border p-2 text-left">Duration</th>
                                                    <th className="border p-2 text-left">Quantity</th>
                                                    <th className="border p-2 text-left">Route</th>
                                                    <th className="border p-2 text-left">Food Relation</th>
                                                    <th className="border p-2 text-left">Notes</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                {consultationPrescriptions.map((prescription) => (
                                                    <tr key={prescription.id}>
                                                        <td className="border p-2">{prescription.medication.name}</td>
                                                        <td className="border p-2">{prescription.dosage}</td>
                                                        <td className="border p-2">{prescription.frequency}</td>
                                                        <td className="border p-2">{prescription.duration}</td>
                                                        <td className="border p-2">{prescription.quantity}</td>
                                                        <td className="border p-2">{prescription.route}</td>
                                                        <td className="border p-2">{prescription.food_relation}</td>
                                                        <td className="border p-2">{prescription.notes || 'N/A'}</td>
                                                    </tr>
                                                ))}
                                            </tbody>
                                        </table>
                                    </div>

                                    <div className="mt-6">
                                        <p className="text-lg font-bold">Prescriber Information</p>
                                        <p className="text-base">Name: {prescriberName}</p>
                                    </div>

                                    <div className="mt-4">
                                        <p className="text-sm font-semibold">Doctor's Signature</p>
                                        <div className="mt-1 h-4 w-64 border border-dashed border-gray-300"></div>
                                    </div>
                                </div>
                            </div>

                            {/* End of Print Prescription Section */}
                            <div className="mb-4 flex justify-between">
                                <h2 className="text-xl font-semibold">
                                    Consultation Number:
                                    <span
                                        style={{
                                            fontWeight: 'bold',
                                            paddingLeft: '20px',
                                        }}
                                    >
                                        #{consultationId}
                                    </span>
                                </h2>
                                <button
                                    className="inline-flex items-center gap-2 rounded-2xl bg-green-600 px-4 py-2 text-sm font-semibold text-white shadow-md transition-colors hover:bg-green-700 focus:ring-2 focus:ring-green-500 focus:ring-offset-2 focus:outline-none"
                                    onClick={reactToPrintFn}
                                    style={{
                                        cursor: 'pointer',
                                    }}
                                >
                                    <Printer className="h-4 w-4" />
                                    Print
                                </button>
                            </div>
                            <div className="mb-4">
                                <p className="text-lg font-semibold">
                                    Patient: {patient.first_name} {patient.surname}
                                </p>
                                <p className="text-lg font-semibold">Prescriber: {prescriberName}</p>
                            </div>
                            <div className="overflow-x-auto rounded-xl border">
                                <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                                    <thead className="bg-muted">
                                        <tr>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Medication
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Dosage
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Frequency
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Duration
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Quantity
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Route
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Food Relation
                                            </th>
                                            <th className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">
                                                Notes
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y">
                                        {consultationPrescriptions.map((prescription) => (
                                            <tr key={prescription.id} className="hover:bg-muted/50 transition">
                                                <td className="px-4 py-3">{prescription.medication.name}</td>
                                                <td className="px-4 py-3">{prescription.dosage}</td>
                                                <td className="px-4 py-3">{prescription.frequency}</td>
                                                <td className="px-4 py-3">{prescription.duration}</td>
                                                <td className="px-4 py-3">{prescription.quantity}</td>
                                                <td className="px-4 py-3">{prescription.route}</td>
                                                <td className="px-4 py-3">{prescription.food_relation}</td>
                                                <td className="px-4 py-3">{prescription.notes || 'N/A'}</td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                            <div
                                className="flex justify-end space-x-2"
                                style={{
                                    marginTop: '10px',
                                }}
                            >
                                <Link
                                    href={route('prescriptions.show', consultationPrescriptions[0].id)}
                                    className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white shadow hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 focus:outline-none"
                                    title="View Prescription"
                                >
                                    <Eye className="h-4 w-4" />
                                </Link>
                                <Link
                                    href={route('prescriptions.edit', consultationPrescriptions[0].id)}
                                    className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white shadow hover:bg-yellow-600 focus:ring-2 focus:ring-yellow-400 focus:ring-offset-2 focus:outline-none"
                                    title="Edit Prescription"
                                >
                                    <Pencil className="h-4 w-4" />
                                </Link>
                            </div>
                        </div>
                    );
                })}

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={prescriptions.last_page}
                        forcePage={prescriptions.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}
