import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, useForm, usePage } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Checkbox } from '@/components/ui/checkbox';
import { Separator } from '@/components/ui/separator';
import Swal from 'sweetalert2';
import moment from 'moment';

type Company = { id: number; name: string };

type Patient = {
  id: number;
  first_name: string;
  surname: string;
  gender?: string | null;
  date_of_birth?: string | null;
  employee_number?: string | null;
  company?: Company | null;
};

type Consultation = {
  id: number;
  consultation_date?: string | null;

  systolic?: string | number | null;
  diastolic?: string | number | null;
  temperature?: string | number | null;
  heart_rate?: string | number | null;
  respiratory_rate?: string | number | null;
  oxygen_saturation?: string | number | null;
  weight?: string | number | null;
  height?: string | number | null;
  bmi?: string | number | null;

  blood_sugar?: string | number | null;
  blood_sugar_unit?: string | null;
  blood_sugar_context?: string | null;
  blood_sugar_measured_at?: string | null;

  presenting_complaint?: string | null;
  clinical_notes?: string | null;
  diagnosis?: string | null;
  treatment_plan?: string | null;

  follow_up_required?: boolean;
  follow_up_date?: string | null;

  injury_on_duty?: boolean;
  refer_for_lab?: boolean;
  refer_for_radiology?: boolean;
};

type LabLatest = {
  id: number;
  provider?: string | null;
  lab_ref_no?: string | null;
  requested_at?: string | null;
  status: 'draft' | 'sent' | 'completed' | 'cancelled';
  tests_requested?: string | null;
  attachment_path?: string | null;
} | null;

type Props = {
  consultation: Consultation;
  patient: Patient;
  defaults: { provider: string; requested_at: string };
  lab_latest: LabLatest;
};

const EXAMS = [
  { key: 'ultrasound', label: 'Ultrasound' },
  { key: 'x_rays', label: 'X-Rays' },
  { key: 'ct_scan', label: 'CT Scan' },
  { key: 'mri_scanning', label: 'MRI Scanning' },
] as const;

// Format helpers
const fmt = (d?: string | null, format = 'D MMM YYYY') =>
  d && moment(d).isValid() ? moment(d).format(format) : (d ?? '-');

const fmtLocal = (d?: string | null) =>
  d && moment(d).isValid() ? moment(d).format('YYYY-MM-DDTHH:mm') : '';

const fmtDateOnly = (d?: string | null) =>
  d && moment(d).isValid() ? moment(d).format('YYYY-MM-DD') : '';

function PatientConsultationCard({ p, c }: { p: Patient; c: Consultation }) {
  return (
    <Card className="border bg-muted/30">
      <CardHeader className="pb-3">
        <CardTitle className="text-sm">Patient</CardTitle>
        <CardDescription>Recipient & consultation</CardDescription>
      </CardHeader>
      <CardContent className="grid gap-2 text-sm sm:grid-cols-2">
        <div><span className="text-muted-foreground">Name:</span> {p.first_name} {p.surname}</div>
        <div><span className="text-muted-foreground">Gender:</span> {p.gender ?? '-'}</div>
        <div><span className="text-muted-foreground">DOB:</span> {fmt(p.date_of_birth)}</div>
        <div><span className="text-muted-foreground">Employee #:</span> {p.employee_number ?? '-'}</div>
        <div className="sm:col-span-2">
          <span className="text-muted-foreground">Company:</span>{' '}
          {p.company?.name ?? '-'}
        </div>
        <div className="sm:col-span-2">
          <span className="text-muted-foreground">Consultation:</span> #{c.id}{' '}
          {c.consultation_date ? `(${fmt(c.consultation_date)})` : ''}
        </div>
      </CardContent>
    </Card>
  );
}

function ConsultationSummary({ c }: { c: Consultation }) {
  return (
    <Card>
      <CardHeader className="pb-3">
        <CardTitle className="text-sm">Consultation Summary</CardTitle>
        <CardDescription>Key clinical details</CardDescription>
      </CardHeader>
      <CardContent className="space-y-3 text-sm">
        {/* Vitals */}
        <div className="grid gap-2 sm:grid-cols-2">
          <div><span className="text-muted-foreground">BP:</span>{' '}
            {c.systolic || c.diastolic ? `${c.systolic ?? '-'} / ${c.diastolic ?? '-'}` : '-'}</div>
          <div><span className="text-muted-foreground">Temperature:</span> {c.temperature ?? '-'}</div>
          <div><span className="text-muted-foreground">Heart Rate:</span> {c.heart_rate ?? '-'}</div>
          <div><span className="text-muted-foreground">Resp. Rate:</span> {c.respiratory_rate ?? '-'}</div>
          <div><span className="text-muted-foreground">O₂ Sat:</span> {c.oxygen_saturation ?? '-'}</div>
          <div><span className="text-muted-foreground">Weight (kg):</span> {c.weight ?? '-'}</div>
          <div><span className="text-muted-foreground">Height (cm):</span> {c.height ?? '-'}</div>
          <div><span className="text-muted-foreground">BMI:</span> {c.bmi ?? '-'}</div>
        </div>

        {/* Glucose */}
        <div className="grid gap-2 sm:grid-cols-2">
          <div>
            <span className="text-muted-foreground">Blood Sugar:</span>{' '}
            {c.blood_sugar ? `${c.blood_sugar} ${c.blood_sugar_unit ?? 'mmol/L'}` : '-'}
          </div>
          <div>
            <span className="text-muted-foreground">Measured At:</span>{' '}
            {fmt(c.blood_sugar_measured_at, 'D MMM YYYY, HH:mm')}
          </div>
        </div>

        {/* Clinical notes */}
        <div>
          <span className="text-muted-foreground">Presenting Complaint:</span>
          <div className="whitespace-pre-wrap">{c.presenting_complaint || '-'}</div>
        </div>
        <div>
          <span className="text-muted-foreground">Diagnosis:</span>
          <div className="whitespace-pre-wrap">{c.diagnosis || '-'}</div>
        </div>
        <div>
          <span className="text-muted-foreground">Treatment Plan:</span>
          <div className="whitespace-pre-wrap">{c.treatment_plan || '-'}</div>
        </div>

        {/* Follow-up / flags */}
        <div className="grid gap-2 sm:grid-cols-2">
          <div>
            <span className="text-muted-foreground">Follow-up Required:</span>{' '}
            {c.follow_up_required ? 'Yes' : 'No'}
          </div>
          <div>
            <span className="text-muted-foreground">Follow-up Date:</span>{' '}
            {fmt(c.follow_up_date, 'D MMM YYYY')}
          </div>
        </div>
        <div className="grid gap-2 sm:grid-cols-2">
          <div>
            <span className="text-muted-foreground">Injury on Duty:</span>{' '}
            {c.injury_on_duty ? 'Yes' : 'No'}
          </div>
          <div>
            <span className="text-muted-foreground">Referrals flagged:</span>{' '}
            {[
              c.refer_for_lab ? 'Lab' : null,
              c.refer_for_radiology ? 'Imaging' : null,
            ].filter(Boolean).join(', ') || '-'}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}

function LabLatestCard({
  lab,
  consultationId,
  patientId,
}: {
  lab: NonNullable<LabLatest>;
  consultationId: number;
  patientId: number;
}) {
  return (
    <Card>
      <CardHeader className="pb-3">
        <CardTitle className="text-sm">Lab (latest)</CardTitle>
        <CardDescription>Most recent lab referral for this consultation</CardDescription>
      </CardHeader>
      <CardContent className="space-y-2 text-sm">
        <div className="grid gap-2 sm:grid-cols-2">
          <div><span className="text-muted-foreground">Provider:</span> {lab.provider ?? '-'}</div>
          <div><span className="text-muted-foreground">Ref #:</span> {lab.lab_ref_no ?? '-'}</div>
          <div><span className="text-muted-foreground">Requested:</span> {fmt(lab.requested_at, 'D MMM YYYY, HH:mm')}</div>
          <div><span className="text-muted-foreground">Status:</span> {lab.status}</div>
        </div>
        <div>
          <span className="text-muted-foreground">Tests/Notes:</span>{' '}
          {lab.tests_requested ? <span className="whitespace-pre-wrap">{lab.tests_requested}</span> : '-'}
        </div>

        <div className="pt-2 flex flex-wrap gap-2">
          <Button asChild size="sm" variant="secondary">
            <Link href={route('referrals.lab.edit', lab.id)}>
              Edit Lab
            </Link>
          </Button>
          <Button
            size="sm"
            variant="outline"
            onClick={() =>
              router.visit(
                route('referrals.lab.create', { consultation: consultationId, patient: patientId })
              )
            }
          >
            New Lab
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}

export default function CreateImaging({ consultation, patient, defaults, lab_latest }: Props) {
  const { auth } = usePage<{ auth?: { user?: { role?: string; roles?: string[] } } }>().props;
  const isDoctor = !!(auth?.user?.role === 'doctor' || auth?.user?.roles?.includes?.('doctor'));

  const form = useForm({
    consultation_id: consultation.id,
    patient_id: patient.id,
    provider: defaults.provider,
    requested_at: defaults.requested_at,
    exams: [] as string[],
    region_of_interest: '',
    history: '',
    lmp: '',
    is_pregnant: false,
    referring_doctor_name: '',
    notes: '',
    attachment: null as File | null,
    status: 'sent' as 'draft' | 'sent' | 'completed' | 'cancelled',

    // ✅ bring back the post-save chooser
    next: '' as '' | 'dispense' | 'lab_edit' | 'notes',
  });

  const { data, setData, post, processing, errors, progress, transform } = form;

  // ✅ SweetAlert chooser on save
  const submit = async (e: React.FormEvent) => {
    e.preventDefault();

    const inputOptions: Record<string, string> = {
      dispense: 'Go to Dispensing',
      lab_edit: 'Edit Lab Referral (or create one if none)',
    };
    if (isDoctor) inputOptions['notes'] = 'Doctor Notes';

    const { value: nextChoice, isConfirmed } = await Swal.fire({
      title: 'Save referral — next step',
      html: `
        <div style="text-align:left">
          <p>The referral will be saved. Choose where to go next:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>Dispensing</b> — proceed to medication dispensing.</li>
            <li><b>Edit Lab Referral</b> — jump to latest lab referral for this consult, or create one.</li>
            ${isDoctor ? '<li><b>Doctor Notes</b> — add clinical notes for this consultation.</li>' : ''}
          </ul>
        </div>
      `,
      input: 'radio',
      inputOptions,
      inputValue: 'dispense',
      inputValidator: (v) => (!v ? 'Please choose one option.' : null),
      showCancelButton: true,
      confirmButtonText: 'Save & go',
      cancelButtonText: 'Cancel',
      reverseButtons: true,
      width: '60rem',
      focusConfirm: false,
    });

    if (!isConfirmed || !nextChoice) return;

    // Ensure `next` is included only for this request
    transform((d) => ({ ...d, next: nextChoice as any }));
    post(route('referrals.imaging.store'), {
      forceFormData: true,
      onFinish: () => {
        // reset transform
        transform((d) => d);
      },
    });
  };

  // Top quick actions (unchanged)
  const goToDispensing = () => {
    router.visit(route('consultations.dispense', { consultation: consultation.id }));
  };
  const goToLab = () => {
    if (lab_latest) {
      router.visit(route('referrals.lab.edit', lab_latest.id));
    } else {
      router.visit(route('referrals.lab.create', { consultation: consultation.id, patient: patient.id }));
    }
  };
  const goToDoctorNotes = () => {
    if (!isDoctor) return;
    router.visit(route('consultations.doctor-notes.create', { consultation: consultation.id }));
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Imaging Referrals', href: route('referrals.imaging.index') },
        { title: 'Create', href: '#' },
      ]}
    >
      <Head title="Create Imaging Referral" />

      <div className="w-full px-4 py-6 sm:px-6 lg:px-8">
        {/* Header / quick actions */}
        <div className="mb-6 flex flex-wrap items-center justify-between gap-3">
          <div>
            <h1 className="text-2xl font-bold">Create Imaging Referral</h1>
            <p className="text-muted-foreground mt-1 text-sm">
              Complete the details below to send the imaging request.
            </p>
          </div>

          <div className="flex flex-wrap items-center gap-2">
            <Button variant="outline" type="button" onClick={goToDispensing}>
              Go To Dispensing Page
            </Button>
            <Button variant="secondary" type="button" onClick={goToLab}>
              Go To Lab
            </Button>
            {isDoctor && (
              <Button type="button" onClick={goToDoctorNotes}>
                Add Doctors Notes
              </Button>
            )}
          </div>
        </div>

        <form onSubmit={submit} className="space-y-6">
          <div className="grid grid-cols-1 gap-6 lg:grid-cols-12">
            {/* Left column: Patient + Consultation + (optional Lab latest) */}
            <div className="lg:col-span-4">
              <div className="lg:sticky lg:top-6 lg:space-y-6">
                <PatientConsultationCard p={patient} c={consultation} />
                <ConsultationSummary c={consultation} />
                {lab_latest && (
                  <LabLatestCard
                    lab={lab_latest}
                    consultationId={consultation.id}
                    patientId={patient.id}
                  />
                )}
              </div>
            </div>

            {/* Right column: Imaging form */}
            <div className="lg:col-span-8">
              <Card>
                <CardHeader>
                  <CardTitle>Imaging Details</CardTitle>
                  <CardDescription>Fill in the request specifics</CardDescription>
                </CardHeader>

                <CardContent className="space-y-6">
                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Provider</Label>
                      <Input
                        value={data.provider}
                        onChange={(e) => setData('provider', e.target.value)}
                      />
                      {errors.provider && (
                        <p className="mt-1 text-sm text-destructive">{errors.provider}</p>
                      )}
                    </div>
                    <div>
                      <Label>Requested At</Label>
                      <Input
                        type="datetime-local"
                        value={fmtLocal(data.requested_at)}
                        onChange={(e) => setData('requested_at', e.target.value)}
                      />
                      {errors.requested_at && (
                        <p className="mt-1 text-sm text-destructive">{errors.requested_at}</p>
                      )}
                    </div>
                  </div>

                  <Separator />

                  <div>
                    <Label className="mb-2 block">Examinations Requested</Label>
                    <div className="grid gap-3 sm:grid-cols-2 md:grid-cols-4">
                      {EXAMS.map((ex) => {
                        const checked = data.exams.includes(ex.key);
                        return (
                          <label key={ex.key} className="flex items-center gap-2">
                            <Checkbox
                              checked={checked}
                              onCheckedChange={(v) => {
                                const on = v === true;
                                setData(
                                  'exams',
                                  on
                                    ? [...data.exams, ex.key]
                                    : data.exams.filter((k) => k !== ex.key),
                                );
                              }}
                            />
                            <span>{ex.label}</span>
                          </label>
                        );
                      })}
                    </div>
                    {errors.exams && (
                      <p className="mt-1 text-sm text-destructive">{errors.exams}</p>
                    )}
                  </div>

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Region of Interest</Label>
                      <Input
                        value={data.region_of_interest}
                        onChange={(e) => setData('region_of_interest', e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>LMP</Label>
                      <Input
                        type="date"
                        value={fmtDateOnly(data.lmp)}
                        onChange={(e) => setData('lmp', e.target.value)}
                      />
                    </div>
                  </div>

                  <div className="flex items-center gap-2">
                    <Checkbox
                      checked={data.is_pregnant}
                      onCheckedChange={(v) => setData('is_pregnant', v === true)}
                    />
                    <Label>Is the patient pregnant?</Label>
                  </div>

                  <div>
                    <Label>History</Label>
                    <Textarea
                      rows={4}
                      value={data.history}
                      onChange={(e) => setData('history', e.target.value)}
                    />
                  </div>

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Referring Clinician</Label>
                      <Input
                        value={data.referring_doctor_name}
                        onChange={(e) => setData('referring_doctor_name', e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>Status</Label>
                      <select
                        className="w-full rounded-md border border-input bg-background p-2 text-sm"
                        value={data.status}
                        onChange={(e) => setData('status', e.target.value as any)}
                      >
                        <option value="draft">Draft</option>
                        <option value="sent">Sent</option>
                        <option value="completed">Completed</option>
                        <option value="cancelled">Cancelled</option>
                      </select>
                    </div>
                  </div>

                  <div>
                    <Label>Notes</Label>
                    <Textarea
                      rows={4}
                      value={data.notes}
                      onChange={(e) => setData('notes', e.target.value)}
                    />
                  </div>

                  <div>
                    <Label>Attach scanned form (optional)</Label>
                    <Input
                      type="file"
                      onChange={(e) => setData('attachment', e.target.files?.[0] ?? null)}
                    />
                    {progress && (
                      <p className="mt-1 text-sm text-muted-foreground">
                        Uploading… {progress.percentage}%
                      </p>
                    )}
                  </div>

                  <div className="flex items-center justify-end gap-2">
                    <Link href={route('referrals.imaging.index')}>
                      <Button variant="outline" type="button">
                        Cancel
                      </Button>
                    </Link>
                    <Button type="submit" disabled={processing}>
                      {processing ? 'Saving…' : 'Save Referral'}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
