import AppLayout from '@/layouts/app-layout';
import { Head, Link, useForm } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Checkbox } from '@/components/ui/checkbox';
import { Separator } from '@/components/ui/separator';
import moment from 'moment';

type Patient = {
  id: number;
  first_name: string;
  surname: string;
  gender?: string | null;
  date_of_birth?: string | null;
  employee_number?: string | null;
};

type Consultation = { id: number; consultation_date?: string | null };

type Referral = {
  id: number;
  consultation_id: number;
  patient_id: number;
  provider?: string | null;
  requested_at?: string | null;
  exams?: string[] | null;
  region_of_interest?: string | null;
  history?: string | null;
  lmp?: string | null;
  is_pregnant?: boolean | null;
  referring_doctor_name?: string | null;
  notes?: string | null;
  status: 'draft' | 'sent' | 'completed' | 'cancelled';
  attachment_path?: string | null;
};

type Props = {
  referral: Referral;
  consultation?: Consultation | null;
  patient?: Patient | null;
  defaults?: {
    requested_at?: string | null;
  };
};

const EXAMS = [
  { key: 'ultrasound', label: 'Ultrasound' },
  { key: 'x_rays', label: 'X-Rays' },
  { key: 'ct_scan', label: 'CT Scan' },
  { key: 'mri_scanning', label: 'MRI Scanning' },
] as const;

// Format helpers
const fmt = (d?: string | null, format = 'D MMM YYYY') =>
  d && moment(d).isValid() ? moment(d).format(format) : (d ?? '-');

const fmtLocal = (d?: string | null) =>
  d && moment(d).isValid() ? moment(d).format('YYYY-MM-DDTHH:mm') : '';

const fmtDateOnly = (d?: string | null) =>
  d && moment(d).isValid() ? moment(d).format('YYYY-MM-DD') : '';

function PatientInfo({ p, c }: { p?: Patient | null; c?: Consultation | null }) {
  if (!p || !c) return null;
  return (
    <Card className="border bg-muted/30">
      <CardHeader className="pb-3">
        <CardTitle className="text-sm">Patient</CardTitle>
        <CardDescription>Referral recipient</CardDescription>
      </CardHeader>
      <CardContent className="grid gap-2 text-sm sm:grid-cols-2">
        <div><span className="text-muted-foreground">Name:</span> {p.first_name} {p.surname}</div>
        <div><span className="text-muted-foreground">Gender:</span> {p.gender ?? '-'}</div>
        <div><span className="text-muted-foreground">DOB:</span> {fmt(p.date_of_birth)}</div>
        <div><span className="text-muted-foreground">Employee #:</span> {p.employee_number ?? '-'}</div>
        <div className="sm:col-span-2"><span className="text-muted-foreground">Consultation:</span> #{c.id} {c.consultation_date ? `(${fmt(c.consultation_date)})` : ''}</div>
      </CardContent>
    </Card>
  );
}

export default function EditImaging({ referral, consultation, patient }: Props) {
  const { data, setData, post, processing, errors, progress } = useForm({
    _method: 'put' as const,
    provider: referral.provider ?? '',
    requested_at: referral.requested_at ?? '',
    exams: (referral.exams ?? []) as string[],
    region_of_interest: referral.region_of_interest ?? '',
    history: referral.history ?? '',
    lmp: referral.lmp ?? '',
    is_pregnant: Boolean(referral.is_pregnant),
    referring_doctor_name: referral.referring_doctor_name ?? '',
    notes: referral.notes ?? '',
    status: referral.status ?? 'sent',
    attachment: null as File | null,
  });

  const submit = (e: React.FormEvent) => {
    e.preventDefault();
    post(route('referrals.imaging.update', referral.id), { forceFormData: true });
  };

  const publicAttachmentUrl =
    referral.attachment_path ? `/storage/${referral.attachment_path}` : null;

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Imaging Referrals', href: route('referrals.imaging.index') },
        { title: `Edit #${referral.id}`, href: '#' },
      ]}
    >
      <Head title={`Edit Imaging Referral #${referral.id}`} />

      {/* Full-width content area */}
      <div className="w-full px-4 py-6 sm:px-6 lg:px-8">
        {/* Page header */}
        <div className="mb-6 flex flex-wrap items-center justify-between gap-3">
          <div>
            <h1 className="text-2xl font-bold">Edit Imaging Referral</h1>
            <p className="text-muted-foreground mt-1 text-sm">
              Update the details below and save your changes.
            </p>
          </div>

          <div className="flex items-center gap-2">
            <Link href={route('referrals.imaging.show', referral.id)}>
              <Button variant="outline" type="button">Back to referral</Button>
            </Link>
            <Button onClick={submit} disabled={processing}>
              {processing ? 'Saving…' : 'Save Changes'}
            </Button>
          </div>
        </div>

        <form onSubmit={submit} className="space-y-6">
          {/* Responsive full-page grid */}
          <div className="grid grid-cols-1 gap-6 lg:grid-cols-12">
            {/* Left column: Patient info (sticky on large) */}
            <div className="lg:col-span-4">
              <div className="lg:sticky lg:top-6 lg:space-y-6">
                <PatientInfo p={patient} c={consultation} />

                {publicAttachmentUrl && (
                  <Card className="border">
                    <CardHeader className="pb-2">
                      <CardTitle className="text-sm">Existing Attachment</CardTitle>
                      <CardDescription>Previously uploaded file</CardDescription>
                    </CardHeader>
                    <CardContent className="text-sm">
                      <a
                        href={publicAttachmentUrl}
                        target="_blank"
                        rel="noreferrer"
                        className="text-primary underline"
                      >
                        View current attachment
                      </a>
                    </CardContent>
                  </Card>
                )}
              </div>
            </div>

            {/* Right column: Form */}
            <div className="lg:col-span-8">
              <Card>
                <CardHeader>
                  <CardTitle>Imaging Details</CardTitle>
                  <CardDescription>Modify the request specifics</CardDescription>
                </CardHeader>

                <CardContent className="space-y-6">
                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Provider</Label>
                      <Input
                        value={data.provider}
                        onChange={(e) => setData('provider', e.target.value)}
                      />
                      {errors.provider && <p className="mt-1 text-sm text-destructive">{errors.provider}</p>}
                    </div>
                    <div>
                      <Label>Requested At</Label>
                      <Input
                        type="datetime-local"
                        value={fmtLocal(data.requested_at)}
                        onChange={(e) => setData('requested_at', e.target.value)}
                      />
                      {errors.requested_at && <p className="mt-1 text-sm text-destructive">{errors.requested_at}</p>}
                    </div>
                  </div>

                  <Separator />

                  <div>
                    <Label className="mb-2 block">Examinations Requested</Label>
                    <div className="grid gap-3 sm:grid-cols-2 md:grid-cols-4">
                      {EXAMS.map((ex) => {
                        const checked = data.exams.includes(ex.key);
                        return (
                          <label key={ex.key} className="flex items-center gap-2">
                            <Checkbox
                              checked={checked}
                              onCheckedChange={(v) => {
                                const on = v === true;
                                setData('exams', on ? [...data.exams, ex.key] : data.exams.filter((k) => k !== ex.key));
                              }}
                            />
                            <span>{ex.label}</span>
                          </label>
                        );
                      })}
                    </div>
                    {errors.exams && <p className="mt-1 text-sm text-destructive">{errors.exams}</p>}
                  </div>

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Region of Interest</Label>
                      <Input
                        value={data.region_of_interest}
                        onChange={(e) => setData('region_of_interest', e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>LMP</Label>
                      <Input
                        type="date"
                        value={fmtDateOnly(data.lmp)}
                        onChange={(e) => setData('lmp', e.target.value)}
                      />
                    </div>
                  </div>

                  <div className="flex items-center gap-2">
                    <Checkbox
                      checked={data.is_pregnant}
                      onCheckedChange={(v) => setData('is_pregnant', v === true)}
                    />
                    <Label>Is the patient pregnant?</Label>
                  </div>

                  <div>
                    <Label>History</Label>
                    <Textarea
                      rows={4}
                      value={data.history}
                      onChange={(e) => setData('history', e.target.value)}
                    />
                  </div>

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Referring Clinician</Label>
                      <Input
                        value={data.referring_doctor_name}
                        onChange={(e) => setData('referring_doctor_name', e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>Status</Label>
                      <select
                        className="w-full rounded-md border border-input bg-background p-2 text-sm"
                        value={data.status}
                        onChange={(e) => setData('status', e.target.value as any)}
                      >
                        <option value="draft">Draft</option>
                        <option value="sent">Sent</option>
                        <option value="completed">Completed</option>
                        <option value="cancelled">Cancelled</option>
                      </select>
                    </div>
                  </div>

                  <div>
                    <Label>Notes</Label>
                    <Textarea
                      rows={4}
                      value={data.notes}
                      onChange={(e) => setData('notes', e.target.value)}
                    />
                  </div>

                  <div>
                    <Label>Replace / attach file (optional)</Label>
                    <Input
                      type="file"
                      onChange={(e) => setData('attachment', e.target.files?.[0] ?? null)}
                    />
                    {progress && (
                      <p className="mt-1 text-sm text-muted-foreground">
                        Uploading… {progress.percentage}%
                      </p>
                    )}
                  </div>

                  <div className="flex items-center justify-end gap-2">
                    <Link href={route('referrals.imaging.show', referral.id)}>
                      <Button variant="outline" type="button">Cancel</Button>
                    </Link>
                    <Button type="submit" disabled={processing}>
                      {processing ? 'Saving…' : 'Save Changes'}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
