import React, { useEffect, useMemo, useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Separator } from '@/components/ui/separator';
import { Eye, Pencil, Trash2, Plus, Search } from 'lucide-react';
import moment from 'moment';
import Swal from 'sweetalert2';
import ReactPaginate from 'react-paginate';

// shadcn/ui table
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';

type Patient = {
  id: number;
  first_name: string;
  surname: string;
  gender?: string | null;
  date_of_birth?: string | null;
  employee_number?: string | null;
};

type Consultation = {
  id: number;
  consultation_date?: string | null;
};

type Referrer = {
  id: number;
  name: string;
};

type ImagingReferralListItem = {
  id: number;
  provider?: string | null;
  requested_at?: string | null;
  exams?: string[] | null;
  status: 'draft' | 'sent' | 'completed' | 'cancelled';
  patient?: Patient;
  consultation?: Consultation;
  referrer?: Referrer;
};

type Paginator<T> = {
  data: T[];
  meta?: {
    current_page: number;
    last_page: number;
    from: number | null;
    to: number | null;
    total: number;
    per_page: number;
    links: Array<{ url: string | null; label: string; active: boolean }>;
  };
  links?: Array<{ url: string | null; label: string; active: boolean }>;
};

type Props = {
  filters: {
    q?: string | null;
    status?: string | null;
    perPage?: number | null;
  };
  items: Paginator<ImagingReferralListItem>;
};

const fmt = (d?: string | null, f = 'D MMM YYYY, HH:mm') =>
  d && moment(d).isValid() ? moment(d).format(f) : '-';

const statusColor = (s: string) => {
  switch (s) {
    case 'draft':
      return 'bg-gray-200 text-gray-800 dark:bg-gray-800 dark:text-gray-100';
    case 'sent':
      return 'bg-blue-200 text-blue-800 dark:bg-blue-800 dark:text-blue-100';
    case 'completed':
      return 'bg-green-200 text-green-800 dark:bg-green-800 dark:text-green-100';
    case 'cancelled':
      return 'bg-red-200 text-red-800 dark:bg-red-800 dark:text-red-100';
    default:
      return 'bg-secondary text-secondary-foreground';
  }
};

function DataTable({
  rows,
  onDestroy,
}: {
  rows: ImagingReferralListItem[];
  onDestroy: (id: number) => void;
}) {
  return (
    <div className="overflow-x-auto rounded-xl border">
      <Table className="min-w-full table-fixed">
        <colgroup>
          <col className="w-[13rem]" /> {/* Requested */}
          <col className="w-[16rem]" /> {/* Patient */}
          <col className="w-[16rem]" /> {/* Consultation */}
          <col className="w-[16rem]" /> {/* Provider */}
          <col />                       {/* Exams (flex) */}
          <col className="w-[14rem]" /> {/* Referrer */}
          <col className="w-[8rem]" />  {/* Status */}
          <col className="w-[15rem]" /> {/* Actions */}
        </colgroup>

        <TableHeader className="bg-muted">
          <TableRow>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Requested</TableHead>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Patient</TableHead>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Consultation</TableHead>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Provider</TableHead>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Exams</TableHead>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Referrer</TableHead>
            <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Status</TableHead>
            <TableHead className="text-right text-xs font-semibold uppercase tracking-wider">Actions</TableHead>
          </TableRow>
        </TableHeader>

        <TableBody>
          {rows.length === 0 ? (
            <TableRow>
              <TableCell colSpan={8} className="py-10 text-center text-sm text-muted-foreground">
                No referrals found.
              </TableCell>
            </TableRow>
          ) : (
            rows.map((r) => (
              <TableRow key={r.id} className="align-top">
                <TableCell className="whitespace-nowrap">{fmt(r.requested_at)}</TableCell>

                <TableCell title={r.patient ? `${r.patient.first_name} ${r.patient.surname}` : '-'}>
                  <div className="truncate">
                    {r.patient ? `${r.patient.first_name} ${r.patient.surname}` : '-'}
                  </div>
                </TableCell>

                <TableCell
                  title={
                    r.consultation
                      ? `#${r.consultation.id} ${
                          r.consultation.consultation_date
                            ? `(${fmt(r.consultation.consultation_date, 'D MMM YYYY')})`
                            : ''
                        }`
                      : '-'
                  }
                >
                  <div className="truncate">
                    {r.consultation ? (
                      <>
                        #{r.consultation.id}{' '}
                        {r.consultation.consultation_date
                          ? `(${fmt(r.consultation.consultation_date, 'D MMM YYYY')})`
                          : ''}
                      </>
                    ) : (
                      '-'
                    )}
                  </div>
                </TableCell>

                <TableCell title={r.provider ?? '-'}>
                  <div className="truncate">{r.provider ?? '-'}</div>
                </TableCell>

                <TableCell title={(r.exams ?? []).join(', ') || '-'}>
                  <div className="truncate">
                    {(r.exams ?? []).length ? (r.exams ?? []).join(', ') : '-'}
                  </div>
                </TableCell>

                <TableCell title={r.referrer?.name ?? '-'}>
                  <div className="truncate">{r.referrer?.name ?? '-'}</div>
                </TableCell>

                <TableCell>
                  <span className={`inline-flex items-center rounded px-2 py-0.5 text-xs ${statusColor(r.status)}`}>
                    {r.status}
                  </span>
                </TableCell>

                <TableCell className="text-right">
                  <div className="flex flex-wrap justify-end gap-2">
                    <Button asChild size="sm" className="gap-1">
                      <Link href={route('referrals.imaging.show', r.id)}>
                        <Eye className="h-4 w-4" />
                        View
                      </Link>
                    </Button>
                    <Button asChild size="sm" variant="secondary" className="gap-1">
                      <Link href={route('referrals.imaging.edit', r.id)}>
                        <Pencil className="h-4 w-4" />
                        Edit
                      </Link>
                    </Button>
                    <Button
                      size="sm"
                      variant="destructive"
                      className="gap-1"
                      onClick={() => onDestroy(r.id)}
                    >
                      <Trash2 className="h-4 w-4" />
                      Delete
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))
          )}
        </TableBody>
      </Table>
    </div>
  );
}

export default function ImagingIndex({ filters, items }: Props) {
  const [q, setQ] = useState<string>(String(filters.q ?? ''));
  const [status, setStatus] = useState<string>(String(filters.status ?? ''));
  const [perPage, setPerPage] = useState<number>(Number(filters.perPage ?? items.meta?.per_page ?? 20));

  // compute counts
  const from = items.meta?.from ?? (items.data.length > 0 ? 1 : 0);
  const to = items.meta?.to ?? items.data.length;
  const total = items.meta?.total ?? items.data.length;

  // apply filters / pagination
  const applyFilters = (page = items.meta?.current_page ?? 1) => {
    const params: Record<string, string | number> = {};
    if (q) params.q = q;
    if (status) params.status = status;
    if (perPage) params.per_page = perPage;
    params.page = page;

    router.get(route('referrals.imaging.index'), params, {
      preserveScroll: true,
      preserveState: true,
      replace: true,
    });
  };

  const clearFilters = () => {
    setQ('');
    setStatus('');
    setPerPage(20);
    router.get(route('referrals.imaging.index'), {}, { preserveScroll: true, replace: true });
  };

  const onPageChange = (sel: { selected: number }) => {
    applyFilters(sel.selected + 1);
  };

  const destroy = async (id: number) => {
    const res = await Swal.fire({
      title: 'Delete referral?',
      html:
        '<div style="text-align:left">' +
        '<p>This will permanently delete the imaging referral.</p>' +
        '<p class="mt-2"><b>This action cannot be undone.</b></p>' +
        '</div>',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      reverseButtons: true,
      width: '42rem',
    });
    if (!res.isConfirmed) return;

    router.delete(route('referrals.imaging.destroy', id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({ icon: 'success', title: 'Deleted', timer: 1200, showConfirmButton: false });
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Imaging Referrals', href: route('referrals.imaging.index') },
      ]}
    >
      <Head title="Imaging Referrals" />

      {/* Page shell (matches your other index pages) */}
      <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
        {/* Header */}
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">Imaging Referrals</h1>

          <Button asChild variant="outline">
            <Link href={route('referrals.imaging.index')}>
              <Plus className="mr-2 h-4 w-4" />
              How to create
            </Link>
          </Button>
        </div>

        {/* Filter Section (bordered, compact, non-overlapping) */}
        <div className="mb-2 rounded-lg border p-4">
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
            {/* Search */}
            <div>
              <Label htmlFor="q" className="mb-1 block text-sm font-medium">Search</Label>
              <div className="relative">
                <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
                <Input
                  id="q"
                  placeholder="Patient name or notes…"
                  value={q}
                  onChange={(e) => setQ(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>

            {/* Status */}
            <div>
              <Label htmlFor="status" className="mb-1 block text-sm font-medium">Status</Label>
              <select
                id="status"
                className="w-full rounded-lg border border-input bg-background p-2 text-sm"
                value={status}
                onChange={(e) => setStatus(e.target.value)}
              >
                <option value="">Any</option>
                <option value="draft">Draft</option>
                <option value="sent">Sent</option>
                <option value="completed">Completed</option>
                <option value="cancelled">Cancelled</option>
              </select>
            </div>

            {/* Per page */}
            <div>
              <Label htmlFor="perPage" className="mb-1 block text-sm font-medium">Per page</Label>
              <select
                id="perPage"
                className="w-full rounded-lg border border-input bg-background p-2 text-sm"
                value={perPage}
                onChange={(e) => setPerPage(Number(e.target.value))}
              >
                <option value="10">10</option>
                <option value="20">20</option>
                <option value="50">50</option>
                <option value="100">100</option>
              </select>
            </div>

            {/* spacer for grid symmetry */}
            <div className="hidden lg:block" />
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              onClick={clearFilters}
              className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
              type="button"
            >
              Reset Filters
            </button>
            <button
              onClick={() => applyFilters(1)}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
              type="button"
            >
              Apply Filters
            </button>
          </div>
        </div>

        {/* Results header */}
        <Card className="mb-2">
          <CardHeader className="pb-2">
            <CardTitle className="text-base">Results</CardTitle>
            <CardDescription>
              Showing {from}–{to} of {total}
            </CardDescription>
          </CardHeader>
          <CardContent>
            <DataTable rows={items.data} onDestroy={destroy} />
          </CardContent>
        </Card>

        {/* Pagination (ReactPaginate) */}
        {items.meta?.last_page && items.meta.last_page > 1 && (
          <>
            <Separator className="my-4" />
            <div className="mt-2 flex justify-center">
              <ReactPaginate
                pageCount={items.meta.last_page}
                forcePage={(items.meta.current_page ?? 1) - 1}
                onPageChange={onPageChange}
                marginPagesDisplayed={1}
                pageRangeDisplayed={3}
                previousLabel="← Prev"
                nextLabel="Next →"
                breakLabel="…"
                containerClassName="flex items-center gap-2 text-sm"
                pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                activeClassName="bg-blue-600 text-white"
                previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                breakClassName="px-2"
              />
            </div>
          </>
        )}
      </div>
    </AppLayout>
  );
}
