import AppLayout from '@/layouts/app-layout';
import { Head, Link, useForm, usePage } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Checkbox } from '@/components/ui/checkbox';
import { Separator } from '@/components/ui/separator';
import Swal from 'sweetalert2';
import moment from 'moment';

type Patient = {
  id: number;
  first_name: string;
  surname: string;
  gender?: string | null;
  date_of_birth?: string | null;
  employee_number?: string | null;
};
type Consultation = { id: number; consultation_date?: string | null };

type LabReferral = {
  id: number;
  consultation_id: number;
  patient_id: number;
  provider?: string | null;
  lab_ref_no?: string | null;
  requested_at?: string | null;
  clinical_data?: string | null;
  specimen_types?: string[] | null;
  tests_requested?: string | null;

  medical_aid_provider?: string | null;
  membership_no?: string | null;
  membership_suffix?: string | null;
  relation_to_member?: string | null;

  copies_to?: string | null;
  patient_bled_by?: string | null;
  tubes_edta?: number | null;
  tubes_sst?: number | null;
  tubes_plain?: number | null;
  tubes_fluoride?: number | null;
  tubes_citrate?: number | null;
  tubes_heparin?: number | null;

  collected_at?: string | null;
  received_by?: string | null;
  received_at?: string | null;

  referral_notes?: string | null;
  status: 'draft' | 'sent' | 'completed' | 'cancelled';
  attachment_path?: string | null;
};

type Props = {
  referral: LabReferral;
  consultation: Consultation | null;
  patient: Patient | null;
  defaults: {
    requested_at?: string | null;
    collected_at?: string | null;
    received_at?: string | null;
  };
};

const SPECIMENS = [
  { key: 'blood', label: 'Blood' },
  { key: 'urine', label: 'Urine' },
  { key: 'swab', label: 'Swab' },
  { key: 'fluid', label: 'Fluid' },
  { key: 'stool', label: 'Stool' },
  { key: 'serum', label: 'Serum' },
  { key: 'other', label: 'Other' },
] as const;

const fmt = (d?: string | null, format = 'D MMM YYYY') =>
  d && moment(d).isValid() ? moment(d).format(format) : (d ?? '-');

const fmtLocal = (d?: string | null) =>
  d && moment(d).isValid() ? moment(d).format('YYYY-MM-DDTHH:mm') : '';

function PatientInfo({ p, c }: { p: Patient; c: Consultation }) {
  return (
    <Card className="border bg-muted/30">
      <CardHeader className="pb-3">
        <CardTitle className="text-sm">Patient</CardTitle>
        <CardDescription>Referral recipient</CardDescription>
      </CardHeader>
      <CardContent className="grid gap-2 text-sm sm:grid-cols-2">
        <div><span className="text-muted-foreground">Name:</span> {p.first_name} {p.surname}</div>
        <div><span className="text-muted-foreground">Gender:</span> {p.gender ?? '-'}</div>
        <div><span className="text-muted-foreground">DOB:</span> {fmt(p.date_of_birth)}</div>
        <div><span className="text-muted-foreground">Employee #:</span> {p.employee_number ?? '-'}</div>
        <div className="sm:col-span-2"><span className="text-muted-foreground">Consultation:</span> #{c.id} {c.consultation_date ? `(${fmt(c.consultation_date)})` : ''}</div>
      </CardContent>
    </Card>
  );
}

export default function EditLab({ referral, consultation, patient, defaults }: Props) {
  const { auth } = usePage<{ auth?: { user?: { role?: string; roles?: string[] } } }>().props;
  const isDoctor = !!(auth?.user?.role === 'doctor' || auth?.user?.roles?.includes?.('doctor'));

  const form = useForm({
    _method: 'put' as const,
    provider: referral.provider ?? '',
    lab_ref_no: referral.lab_ref_no ?? '',
    requested_at: defaults.requested_at ?? referral.requested_at ?? '',
    clinical_data: referral.clinical_data ?? '',
    specimen_types: Array.isArray(referral.specimen_types) ? referral.specimen_types : [],
    tests_requested: referral.tests_requested ?? '',

    medical_aid_provider: referral.medical_aid_provider ?? '',
    membership_no: referral.membership_no ?? '',
    membership_suffix: referral.membership_suffix ?? '',
    relation_to_member: referral.relation_to_member ?? '',

    copies_to: referral.copies_to ?? '',
    patient_bled_by: referral.patient_bled_by ?? '',
    tubes_edta: referral.tubes_edta ?? 0,
    tubes_sst: referral.tubes_sst ?? 0,
    tubes_plain: referral.tubes_plain ?? 0,
    tubes_fluoride: referral.tubes_fluoride ?? 0,
    tubes_citrate: referral.tubes_citrate ?? 0,
    tubes_heparin: referral.tubes_heparin ?? 0,

    collected_at: defaults.collected_at ?? referral.collected_at ?? '',
    received_by: referral.received_by ?? '',
    received_at: defaults.received_at ?? referral.received_at ?? '',

    referral_notes: referral.referral_notes ?? '',
    status: referral.status ?? 'sent',
    attachment: null as File | null,

    next: '' as '' | 'dispense' | 'imaging' | 'notes',
  });

  const { data, setData, post, processing, errors, progress, transform } = form;

  const submit = async (e: React.FormEvent) => {
    e.preventDefault();

    const inputOptions: Record<string, string> = {
      dispense: 'Go to Dispensing',
      imaging: 'Create Imaging Referral',
    };
    if (isDoctor) inputOptions['notes'] = 'Doctor Notes';

    const { value: nextChoice, isConfirmed } = await Swal.fire({
      title: 'Save changes — next step',
      html: `
        <div style="text-align:left">
          <p>The referral will be updated. Choose where to go next:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>Dispensing</b> — proceed to medication dispensing.</li>
            <li><b>Imaging referral</b> — open the imaging referral form.</li>
            ${isDoctor ? '<li><b>Doctor Notes</b> — add clinical notes for this consultation.</li>' : ''}
          </ul>
        </div>
      `,
      input: 'radio',
      inputOptions,
      inputValue: 'dispense',
      inputValidator: (v) => (!v ? 'Please choose one option.' : null),
      showCancelButton: true,
      confirmButtonText: 'Save & go',
      cancelButtonText: 'Cancel',
      reverseButtons: true,
      width: '60rem',
      focusConfirm: false,
    });

    if (!isConfirmed || !nextChoice) return;

    transform((d) => ({ ...d, next: nextChoice as any }));
    post(route('referrals.lab.update', referral.id), {
      forceFormData: true,
      onFinish: () => {
        transform((d) => d);
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Lab Referrals', href: route('referrals.lab.index') },
        { title: `Edit #${referral.id}`, href: '#' },
      ]}
    >
      <Head title={`Edit Lab Referral #${referral.id}`} />

      <div className="w-full px-4 py-6 sm:px-6 lg:px-8">
        <div className="mb-6 flex flex-wrap items-center justify-between gap-3">
          <div>
            <h1 className="text-2xl font-bold">Edit Lab Referral</h1>
            <p className="text-muted-foreground mt-1 text-sm">
              Update details and save.
            </p>
          </div>

          <div className="flex items-center gap-2">
            <Link href={route('referrals.lab.show', referral.id)}>
              <Button variant="outline" type="button">Back to referral</Button>
            </Link>
            <Button onClick={submit} disabled={processing}>
              {processing ? 'Saving…' : 'Save Changes'}
            </Button>
          </div>
        </div>

        <form onSubmit={submit} className="space-y-6">
          <div className="grid grid-cols-1 gap-6 lg:grid-cols-12">
            <div className="lg:col-span-4">
              <div className="lg:sticky lg:top-6 lg:space-y-6">
                {patient && consultation ? (
                  <PatientInfo p={patient} c={consultation} />
                ) : (
                  <Card className="border bg-muted/30">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm">Patient</CardTitle>
                      <CardDescription>Referral recipient</CardDescription>
                    </CardHeader>
                    <CardContent className="text-sm text-muted-foreground">
                      Patient/Consultation details unavailable.
                    </CardContent>
                  </Card>
                )}
              </div>
            </div>

            <div className="lg:col-span-8">
              <Card>
                <CardHeader>
                  <CardTitle>Lab Details</CardTitle>
                  <CardDescription>Edit request specifics</CardDescription>
                </CardHeader>

                <CardContent className="space-y-6">
                  <div className="grid gap-4 sm:grid-cols-3">
                    <div>
                      <Label>Provider</Label>
                      <Input value={data.provider} onChange={(e) => setData('provider', e.target.value)} />
                      {errors.provider && <p className="mt-1 text-sm text-destructive">{errors.provider}</p>}
                    </div>
                    <div>
                      <Label>Lab Ref No</Label>
                      <Input value={data.lab_ref_no} onChange={(e) => setData('lab_ref_no', e.target.value)} />
                      {errors.lab_ref_no && <p className="mt-1 text-sm text-destructive">{errors.lab_ref_no}</p>}
                    </div>
                    <div>
                      <Label>Requested At</Label>
                      <Input type="datetime-local" value={fmtLocal(data.requested_at)} onChange={(e) => setData('requested_at', e.target.value)} />
                      {errors.requested_at && <p className="mt-1 text-sm text-destructive">{errors.requested_at}</p>}
                    </div>
                  </div>

                  <Separator />

                  <div>
                    <Label>Clinical Data</Label>
                    <Textarea rows={4} value={data.clinical_data} onChange={(e) => setData('clinical_data', e.target.value)} />
                  </div>

                  <div>
                    <Label className="mb-2 block">Specimen Types</Label>
                    <div className="grid gap-3 sm:grid-cols-3">
                      {SPECIMENS.map((sp) => {
                        const checked = data.specimen_types.includes(sp.key);
                        return (
                          <label key={sp.key} className="flex items-center gap-2">
                            <Checkbox
                              checked={checked}
                              onCheckedChange={(v) => {
                                const on = v === true;
                                setData('specimen_types', on ? [...data.specimen_types, sp.key] : data.specimen_types.filter((k) => k !== sp.key));
                              }}
                            />
                            <span>{sp.label}</span>
                          </label>
                        );
                      })}
                    </div>
                    {errors.specimen_types && <p className="mt-1 text-sm text-destructive">{errors.specimen_types}</p>}
                  </div>

                  <div>
                    <Label>Examination / Tests Requested</Label>
                    <Textarea rows={5} value={data.tests_requested} onChange={(e) => setData('tests_requested', e.target.value)} />
                  </div>

                  <Separator />

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Medical Aid Provider</Label>
                      <Input value={data.medical_aid_provider} onChange={(e) => setData('medical_aid_provider', e.target.value)} />
                    </div>
                    <div>
                      <Label>Membership No</Label>
                      <Input value={data.membership_no} onChange={(e) => setData('membership_no', e.target.value)} />
                    </div>
                    <div>
                      <Label>Membership Suffix</Label>
                      <Input value={data.membership_suffix} onChange={(e) => setData('membership_suffix', e.target.value)} />
                    </div>
                    <div>
                      <Label>Relation to Member</Label>
                      <Input value={data.relation_to_member} onChange={(e) => setData('relation_to_member', e.target.value)} />
                    </div>
                  </div>

                  <Separator />

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Copies To</Label>
                      <Input value={data.copies_to} onChange={(e) => setData('copies_to', e.target.value)} />
                    </div>
                    <div>
                      <Label>Patient Bled By</Label>
                      <Input value={data.patient_bled_by} onChange={(e) => setData('patient_bled_by', e.target.value)} />
                    </div>
                  </div>

                  <div className="grid gap-4 sm:grid-cols-3">
                    <div><Label>EDTA Tubes</Label><Input type="number" min={0} value={data.tubes_edta} onChange={(e) => setData('tubes_edta', e.target.value)} /></div>
                    <div><Label>SST Tubes</Label><Input type="number" min={0} value={data.tubes_sst} onChange={(e) => setData('tubes_sst', e.target.value)} /></div>
                    <div><Label>Plain Tubes</Label><Input type="number" min={0} value={data.tubes_plain} onChange={(e) => setData('tubes_plain', e.target.value)} /></div>
                    <div><Label>Fluoride Tubes</Label><Input type="number" min={0} value={data.tubes_fluoride} onChange={(e) => setData('tubes_fluoride', e.target.value)} /></div>
                    <div><Label>Citrate Tubes</Label><Input type="number" min={0} value={data.tubes_citrate} onChange={(e) => setData('tubes_citrate', e.target.value)} /></div>
                    <div><Label>Heparin Tubes</Label><Input type="number" min={0} value={data.tubes_heparin} onChange={(e) => setData('tubes_heparin', e.target.value)} /></div>
                  </div>

                  <div className="grid gap-4 sm:grid-cols-3">
                    <div><Label>Collected At</Label><Input type="datetime-local" value={fmtLocal(data.collected_at)} onChange={(e) => setData('collected_at', e.target.value)} /></div>
                    <div><Label>Received By</Label><Input value={data.received_by} onChange={(e) => setData('received_by', e.target.value)} /></div>
                    <div><Label>Received At</Label><Input type="datetime-local" value={fmtLocal(data.received_at)} onChange={(e) => setData('received_at', e.target.value)} /></div>
                  </div>

                  <div>
                    <Label>Referral Notes</Label>
                    <Textarea rows={4} value={data.referral_notes} onChange={(e) => setData('referral_notes', e.target.value)} />
                  </div>

                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label>Status</Label>
                      <select
                        className="w-full rounded-md border border-input bg-background p-2 text-sm"
                        value={data.status}
                        onChange={(e) => setData('status', e.target.value as any)}
                      >
                        <option value="draft">Draft</option>
                        <option value="sent">Sent</option>
                        <option value="completed">Completed</option>
                        <option value="cancelled">Cancelled</option>
                      </select>
                    </div>
                    <div>
                      <Label>Replace/attach scanned form (optional)</Label>
                      <Input type="file" onChange={(e) => setData('attachment', e.target.files?.[0] ?? null)} />
                      {progress && <p className="mt-1 text-sm text-muted-foreground">Uploading… {progress.percentage}%</p>}
                    </div>
                  </div>

                  <div className="flex items-center justify-end gap-2">
                    <Link href={route('referrals.lab.show', referral.id)}>
                      <Button variant="outline" type="button">Cancel</Button>
                    </Link>
                    <Button type="submit" disabled={processing}>{processing ? 'Saving…' : 'Save Changes'}</Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
