import React, { useEffect, useMemo, useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Eye, Pencil, Trash2, Paperclip, Search } from 'lucide-react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import moment from 'moment';

// shadcn/ui table
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';

type Patient = {
  id: number;
  first_name: string;
  surname: string;
  gender?: string | null;
  date_of_birth?: string | null;
  employee_number?: string | null;
};

type Consultation = {
  id: number;
  consultation_date?: string | null;
};

type Referrer = { id: number; name: string };

type LabReferralRow = {
  id: number;
  consultation_id: number;
  patient_id: number;
  provider?: string | null;
  lab_ref_no?: string | null;
  requested_at?: string | null;
  tests_requested?: string | null;
  status: 'draft' | 'sent' | 'completed' | 'cancelled';
  attachment_path?: string | null;

  patient?: Patient;
  consultation?: Consultation;
  referrer?: Referrer;
};

type Pagination<T> = {
  data: T[];
  current_page: number;
  last_page: number;
  total: number;
  per_page: number;
  from: number | null;
  to: number | null;
};

type PageProps = {
  filters: { q?: string | null; status?: string | null; perPage?: number | string | null };
  items: Pagination<LabReferralRow>;
};

const STATUS_OPTIONS = [
  { value: '', label: 'All statuses' },
  { value: 'draft', label: 'Draft' },
  { value: 'sent', label: 'Sent' },
  { value: 'completed', label: 'Completed' },
  { value: 'cancelled', label: 'Cancelled' },
] as const;

const PER_PAGE_OPTIONS = [10, 20, 50, 100];

const fmt = (d?: string | null, format = 'D MMM YYYY, HH:mm') =>
  d && moment(d).isValid() ? moment(d).format(format) : '-';

function StatusBadge({ status }: { status: LabReferralRow['status'] }) {
  const map:
    | Record<
        LabReferralRow['status'],
        { variant: 'default' | 'secondary' | 'outline' | 'destructive'; label: string; className?: string }
      > = {
    draft: { variant: 'secondary', label: 'Draft' },
    sent: { variant: 'default', label: 'Sent' },
    completed: { variant: 'outline', label: 'Completed', className: 'border-green-500 text-green-600' },
    cancelled: { variant: 'destructive', label: 'Cancelled' },
  };
  const item = map[status] ?? { variant: 'outline', label: status };
  return (
    <Badge variant={item.variant} className={item.className}>
      {item.label}
    </Badge>
  );
}

export default function LabIndex() {
  const { filters, items } = usePage<PageProps>().props;

  const [q, setQ] = useState<string>(String(filters.q ?? '').trim());
  const [status, setStatus] = useState<string>(String(filters.status ?? ''));
  const [perPage, setPerPage] = useState<number>(Number(filters.perPage ?? items.per_page ?? 20));

  // sync state when navigating back/forward
  useEffect(() => {
    setQ(String(filters.q ?? '').trim());
    setStatus(String(filters.status ?? ''));
    setPerPage(Number(filters.perPage ?? items.per_page ?? 20));
  }, [filters.q, filters.status, filters.perPage, items.per_page]);

  const applyFilters = (page = 1) => {
    const params: Record<string, string | number> = {};
    if (q) params.q = q;
    if (status) params.status = status;
    if (perPage) params.per_page = perPage;
    if (page) params.page = page;

    router.get(route('referrals.lab.index'), params, {
      preserveScroll: true,
      preserveState: true,
      replace: true,
    });
  };

  const handleDelete = async (id: number) => {
    const res = await Swal.fire({
      title: 'Delete referral?',
      html:
        '<div style="text-align:left">' +
        '<p>This will permanently delete the lab referral.</p>' +
        '<p class="mt-2"><b>This action cannot be undone.</b></p>' +
        '</div>',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      reverseButtons: true,
      width: '42rem',
    });
    if (!res.isConfirmed) return;

    router.delete(route('referrals.lab.destroy', id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({ icon: 'success', title: 'Deleted', timer: 1200, showConfirmButton: false });
      },
    });
  };

  const showing = useMemo(() => {
    const from = items.from ?? 0;
    const to = items.to ?? 0;
    return `Showing ${from}–${to} of ${items.total}`;
  }, [items.from, items.to, items.total]);

  const onReset = () => {
    setQ('');
    setStatus('');
    setPerPage(20);
    router.get(route('referrals.lab.index'), {}, { preserveScroll: true, replace: true });
  };

  const onPageChange = (sel: { selected: number }) => {
    applyFilters(sel.selected + 1);
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Referrals', href: '#' },
        { title: 'Lab Referrals', href: route('referrals.lab.index') },
      ]}
    >
      <Head title="Lab Referrals" />

      {/* Page shell (matches ConsultationIndex look/feel) */}
      <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
        {/* Header */}
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">Lab Referrals</h1>
          <div className="text-sm text-muted-foreground">{showing}</div>
        </div>

        {/* Filter Section (bordered, compact) */}
        <div className="mb-2 rounded-lg border p-4">
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
            {/* Search */}
            <div>
              <label className="mb-1 block text-sm font-medium">Search</label>
              <div className="relative">
                <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
                <input
                  type="text"
                  value={q}
                  onChange={(e) => setQ(e.target.value)}
                  placeholder="Patient or tests…"
                  className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:outline-none focus:ring-2 focus:ring-blue-600"
                />
              </div>
            </div>

            {/* Status */}
            <div>
              <label className="mb-1 block text-sm font-medium">Status</label>
              <select
                value={status}
                onChange={(e) => setStatus(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                {STATUS_OPTIONS.map((s) => (
                  <option key={s.value} value={s.value}>
                    {s.label}
                  </option>
                ))}
              </select>
            </div>

            {/* Per Page */}
            <div>
              <label className="mb-1 block text-sm font-medium">Per Page</label>
              <select
                value={perPage}
                onChange={(e) => setPerPage(Number(e.target.value))}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                {PER_PAGE_OPTIONS.map((n) => (
                  <option key={n} value={n}>
                    {n}
                  </option>
                ))}
              </select>
            </div>

            {/* Spacer for grid symmetry */}
            <div className="hidden lg:block" />
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              onClick={onReset}
              className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
              type="button"
            >
              Reset Filters
            </button>
            <button
              onClick={() => applyFilters(1)}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
              type="button"
            >
              Apply Filters
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-full">
            <colgroup>
              <col className="w-[16rem]" /> {/* Patient */}
              <col className="w-[14rem]" /> {/* Consultation */}
              <col className="w-[16rem]" /> {/* Provider */}
              <col className="w-[13rem]" /> {/* Requested */}
              <col />                       {/* Tests/Notes */}
              <col className="w-[8rem]" />  {/* Status */}
              <col className="w-[15rem]" /> {/* Actions */}
            </colgroup>

            <TableHeader className="bg-muted">
              <TableRow>
                <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Patient</TableHead>
                <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Consultation</TableHead>
                <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Provider</TableHead>
                <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Requested</TableHead>
                <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Tests / Notes</TableHead>
                <TableHead className="text-left text-xs font-semibold uppercase tracking-wider">Status</TableHead>
                <TableHead className="text-right text-xs font-semibold uppercase tracking-wider">Actions</TableHead>
              </TableRow>
            </TableHeader>

            <TableBody>
              {items.data.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={7} className="py-10 text-center text-sm text-muted-foreground">
                    Nothing to show. Try adjusting your filters.
                  </TableCell>
                </TableRow>
              ) : (
                items.data.map((row) => {
                  const patientName = row.patient
                    ? `${row.patient.first_name} ${row.patient.surname}`
                    : `#${row.patient_id}`;
                  const consultDate = row.consultation?.consultation_date
                    ? moment(row.consultation.consultation_date).format('D MMM YYYY')
                    : '-';
                  const tests = row.tests_requested ?? '';
                  const words = tests.trim() ? tests.split(/\s+/) : [];
                  const shortTests = words.slice(0, 14).join(' ') + (words.length > 14 ? '…' : '');

                  return (
                    <TableRow key={row.id} className="align-top">
                      {/* Patient */}
                      <TableCell title={patientName}>
                        <div className="truncate font-medium">{patientName}</div>
                        <div className="text-muted-foreground text-xs">#{row.patient_id}</div>
                      </TableCell>

                      {/* Consultation */}
                      <TableCell
                        title={`#${row.consultation_id}${consultDate !== '-' ? ` (${consultDate})` : ''}`}
                      >
                        <div className="truncate">#{row.consultation_id}</div>
                        <div className="text-muted-foreground text-xs">{consultDate}</div>
                      </TableCell>

                      {/* Provider + ref */}
                      <TableCell title={row.provider ?? '-'}>
                        <div className="truncate">{row.provider ?? '-'}</div>
                        {row.lab_ref_no ? (
                          <div className="text-muted-foreground text-xs">Ref: {row.lab_ref_no}</div>
                        ) : null}
                      </TableCell>

                      {/* Requested */}
                      <TableCell className="whitespace-nowrap">{fmt(row.requested_at)}</TableCell>

                      {/* Tests / Notes */}
                      <TableCell title={tests || '-'}>
                        <div className="truncate">{shortTests || '-'}</div>
                        {row.attachment_path && (
                          <div className="mt-1 inline-flex items-center gap-1 text-xs text-muted-foreground">
                            <Paperclip className="h-3.5 w-3.5" />
                            Scanned form attached
                          </div>
                        )}
                      </TableCell>

                      {/* Status */}
                      <TableCell>
                        <StatusBadge status={row.status} />
                      </TableCell>

                      {/* Actions */}
                      <TableCell className="text-right">
                        <div className="flex flex-wrap justify-end gap-2">
                          <Button asChild size="sm" variant="outline" className="gap-1">
                            <Link href={route('referrals.lab.show', row.id)}>
                              <Eye className="h-4 w-4" />
                              View
                            </Link>
                          </Button>
                          <Button asChild size="sm" variant="secondary" className="gap-1">
                            <Link href={route('referrals.lab.edit', row.id)}>
                              <Pencil className="h-4 w-4" />
                              Edit
                            </Link>
                          </Button>
                          <Button
                            size="sm"
                            variant="destructive"
                            className="gap-1"
                            onClick={() => handleDelete(row.id)}
                          >
                            <Trash2 className="h-4 w-4" />
                            Delete
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  );
                })
              )}
            </TableBody>
          </Table>
        </div>

        {/* Pagination (ReactPaginate, centered) */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={items.last_page}
            forcePage={(items.current_page ?? 1) - 1}
            onPageChange={onPageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="…"
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
