import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Search, Trash, Download, FileSpreadsheet } from 'lucide-react';
import moment from 'moment';
import { useEffect, useState } from 'react';
import ReactPaginate from 'react-paginate';
import * as XLSX from 'xlsx';
import { useDebouncedCallback } from 'use-debounce';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Reports', href: '/reports' }];

interface PagePropsWithConsultations extends PageProps {
  consultations: {
    data: any;
    total: number;
    per_page: number;
    current_page: number;
    last_page: number;
  };
  clinics: Array<{ id: number; name: string }>;
  doctors: Array<{ id: number; name: string }>;
  years: Array<number>;
  patientTypes: Array<{ value: string; label: string }>;
  periods: Array<{ value: string; label: string }>;
  filters: {
    search?: string;
    start_date?: string;
    end_date?: string;
    clinic_id?: number;
    doctor_id?: number;
    year?: number;
    month?: number;
    period?: string;
    patient_type?: string;
  };
}


type AuthUser = {
  id: number;
  name: string;
  role?: string;
  roles?: string[];
  can?: Record<string, boolean>;
};

export default function ReportsIndex() {
  const { consultations, clinics, doctors, years, patientTypes, periods, filters } =
    usePage<PagePropsWithConsultations>().props;

  const [searchQuery, setSearchQuery] = useState(filters.search || '');
  const [startDate, setStartDate] = useState(filters.start_date || '');
  const [endDate, setEndDate] = useState(filters.end_date || '');
  const [clinicId, setClinicId] = useState<any>(filters.clinic_id || '');
  const [doctorId, setDoctorId] = useState<any>(filters.doctor_id || '');
  const [year, setYear] = useState<any>(filters.year || '');
  const [month, setMonth] = useState<any>(filters.month || '');
  const [period, setPeriod] = useState(filters.period || '');
  const [patientType, setPatientType] = useState(filters.patient_type || '');


  const { auth } = usePage<{ auth: { user: AuthUser } }>().props;

  // visible if role is exactly "superadmin" (with a couple of safe fallbacks)
  const canModify =
    auth?.user?.role === 'superadmin' ||
    auth?.user?.roles?.includes?.('superadmin') ||
    auth?.user?.can?.delete_consultations === true;

  // Debounced search fetch (500ms)
  const debouncedSearch = useDebouncedCallback((value: string) => {
    router.get(
      `${API}/reports`,
      {
        search: value || undefined,
        start_date: startDate || undefined,
        end_date: endDate || undefined,
        clinic_id: clinicId || undefined,
        doctor_id: doctorId || undefined,
        year: year || undefined,
        month: month || undefined,
        period: period || undefined,
        patient_type: patientType || undefined,
      },
      {
        preserveState: true,
        preserveScroll: true,
        replace: true,
        only: ['consultations', 'filters'],
      }
    );
  }, 500);

  useEffect(() => {
    return () => debouncedSearch.cancel();
  }, [debouncedSearch]);

  const handleFilter = () => {
    router.get(
      `${API}/reports`,
      {
        search: searchQuery || undefined,
        start_date: startDate || undefined,
        end_date: endDate || undefined,
        clinic_id: clinicId || undefined,
        doctor_id: doctorId || undefined,
        year: year || undefined,
        month: month || undefined,
        period: period || undefined,
        patient_type: patientType || undefined,
      },
      { preserveState: true, preserveScroll: true, only: ['consultations', 'filters'] }
    );
  };

  const handleResetFilters = () => {
    setSearchQuery('');
    setStartDate('');
    setEndDate('');
    setClinicId('');
    setDoctorId('');
    setYear('');
    setMonth('');
    setPeriod('');
    setPatientType('');

    router.get(`${API}/reports`, {}, { preserveState: true, replace: true, only: ['consultations', 'filters'] });
  };

  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      `${API}/reports`,
      {
        page: selected.selected + 1,
        search: searchQuery || undefined,
        start_date: startDate || undefined,
        end_date: endDate || undefined,
        clinic_id: clinicId || undefined,
        doctor_id: doctorId || undefined,
        year: year || undefined,
        month: month || undefined,
        period: period || undefined,
        patient_type: patientType || undefined,
      },
      { preserveState: true, preserveScroll: true, only: ['consultations', 'filters'] }
    );
  };

  // Existing client-side export (quick table dump)
  const exportToExcel = () => {
    const data = consultations.data.map((consultation: any) => ({
      Date: moment(consultation.consultation_date).format('MMMM Do YYYY'),
      'Patient Name': consultation.patient?.first_name || '',
      'Last Name': consultation.patient?.surname || '',
      Gender: consultation.patient?.gender || '',
      'Date of Birth': consultation.patient?.date_of_birth
        ? moment(consultation.patient.date_of_birth).format('DD MMM YYYY')
        : '-',
      'National ID': consultation.patient?.id_number || '',
      Clinic: consultation.clinic?.name || '',
      Doctor: consultation.doctor?.name || '',
      Diagnosis: consultation.diagnosis || '',
    }));

    const worksheet = XLSX.utils.json_to_sheet(data);
    const keys = data.length > 0 ? Object.keys(data[0]) : [];
    const cols = keys.map((key) => {
      const maxLength = Math.max(key.length, ...data.map((row: any) => (row[key] ? row[key].toString().length : 0)));
      return { wch: maxLength + 2 };
    });
    (worksheet as any)['!cols'] = cols;

    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, 'Consultations');
    XLSX.writeFile(workbook, 'Patient_Report.xlsx', { compression: true });
  };

  // NEW: Server-generated monthly report (OpenSpout)
  const downloadReport = () => {
    if (!year || !month) return;

    const q: Record<string, string> = {
      year: String(year),
      month: String(month),
    };

    if (searchQuery) q.search = searchQuery;
    if (startDate) q.start_date = startDate;
    if (endDate) q.end_date = endDate;
    if (clinicId) q.clinic_id = String(clinicId);
    if (doctorId) q.doctor_id = String(doctorId);
    if (period) q.period = String(period);
    if (patientType) q.patient_type = String(patientType);

    const qs = new URLSearchParams(q).toString();

    // @ts-ignore - route may be declared globally by Ziggy
    const base = typeof route === 'function'
      // @ts-ignore
      ? route('reports.consultations.export')
      : `${API}/reports/consultations/export`;

    const url = `${base}?${qs}`;
    window.location.href = url;
  };

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Reports" />
      <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">Reports</h1>
          <div className="flex items-center gap-3">
            {/* Quick client-side export */}
            <button
              onClick={exportToExcel}
              className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-4 py-2 text-sm font-medium text-white hover:bg-green-700"
              style={{ cursor: 'pointer' }}
            >
              <Download className="h-4 w-4" />
              Export Excel
            </button>

            {/* Server-generated OpenSpout export */}
            {year && month ? (
              <button
                onClick={downloadReport}
                className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                title="Download the monthly consultations report (OpenSpout)"
              >
                <Download className="h-4 w-4" />
                Download Report
              </button>
            ) : null}
          </div>
        </div>

        {/* 🔹 Excel Reports Portal Card */}
        <Card className="border-dashed">
          <CardHeader className="pb-3">
            <div className="flex flex-wrap items-start justify-between gap-3">
              <div>
                <CardTitle className="text-lg">Excel Reports Portal</CardTitle>
                <CardDescription>
                  Generate all Excel reports in one place — bulk exports with advanced filters and presets.
                </CardDescription>
              </div>
              <Button asChild>
                <Link href={`${API}/reports/portal`} className="inline-flex items-center gap-2">
                  <FileSpreadsheet className="h-4 w-4" />
                  Open Portal
                </Link>
              </Button>
            </div>
          </CardHeader>
          <CardContent className="text-sm text-muted-foreground">
            Tip: Use the portal for large exports (multi-month, multi-clinic) or when you need preformatted spreadsheets.
          </CardContent>
        </Card>

        {/* Filter Section */}
        <div className="mb-6 rounded-lg border p-4">
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
            {/* Search (debounced) */}
            <div>
              <label className="mb-1 block text-sm font-medium">Search</label>
              <div className="relative">
                <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => {
                    setSearchQuery(e.target.value);
                    debouncedSearch(e.target.value);
                  }}
                  placeholder="Search..."
                  className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:outline-none focus:ring-2 focus:ring-blue-600"
                />
              </div>
            </div>

            {/* Date Range */}
            <div>
              <label className="mb-1 block text-sm font-medium">Date Range</label>
              <div className="grid grid-cols-2 gap-2">
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
              </div>
            </div>

            {/* Clinic */}
            <div>
              <label className="mb-1 block text-sm font-medium">Clinic</label>
              <select
                value={clinicId}
                onChange={(e) => setClinicId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Clinics</option>
                {clinics.map((clinic) => (
                  <option key={clinic.id} value={clinic.id}>
                    {clinic.name}
                  </option>
                ))}
              </select>
            </div>

            {/* Doctor */}
            <div>
              <label className="mb-1 block text-sm font-medium">Doctor</label>
              <select
                value={doctorId}
                onChange={(e) => setDoctorId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Doctors</option>
                {doctors.map((doctor) => (
                  <option key={doctor.id} value={doctor.id}>
                    {doctor.name}
                  </option>
                ))}
              </select>
            </div>

            {/* Year */}
            <div>
              <label className="mb-1 block text-sm font-medium">Year</label>
              <select
                value={year}
                onChange={(e) => setYear(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Years</option>
                {years.map((y) => (
                  <option key={y} value={y}>
                    {y}
                  </option>
                ))}
              </select>
            </div>

            {/* Month */}
            <div>
              <label className="mb-1 block text-sm font-medium">Month</label>
              <select
                value={month}
                onChange={(e) => setMonth(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Months</option>
                {Array.from({ length: 12 }, (_, i) => (
                  <option key={i + 1} value={i + 1}>
                    {moment().month(i).format('MMMM')}
                  </option>
                ))}
              </select>
            </div>

            {/* Period */}
            <div>
              <label className="mb-1 block text-sm font-medium">Period</label>
              <select
                value={period}
                onChange={(e) => setPeriod(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Periods</option>
                {periods.map((p) => (
                  <option key={p.value} value={p.value}>
                    {p.label}
                  </option>
                ))}
              </select>
            </div>

            {/* Patient Type */}
            <div>
              <label className="mb-1 block text-sm font-medium">Patient Type</label>
              <select
                value={patientType}
                onChange={(e) => setPatientType(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Types</option>
                {patientTypes.map((type) => (
                  <option key={type.value} value={type.value}>
                    {type.label}
                  </option>
                ))}
              </select>
            </div>
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              onClick={handleResetFilters}
              className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
            >
              Reset Filters
            </button>
            <button
              onClick={handleFilter}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
            >
              Apply Filters
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted">
              <tr>
                {['Date', 'Patient Name', 'Last Name', 'Clinic', 'Doctor', 'Diagnosis', 'Actions'].map((header) => (
                  <th
                    key={header}
                    className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider"
                  >
                    {header}
                  </th>
                ))}
              </tr>
            </thead>
            <tbody className="divide-y">
              {consultations.data.map((consultation: any) => (
                <tr key={consultation.id} className="transition hover:bg-muted/50">
                  <td className="px-4 py-3">{moment(consultation.consultation_date).format('MMMM Do YYYY')}</td>
                  <td className="px-4 py-3">{consultation.patient?.first_name}</td>
                  <td className="px-4 py-3">{consultation.patient?.surname}</td>
                  <td className="px-4 py-3">{consultation.clinic?.name}</td>
                  <td className="px-4 py-3">{consultation.doctor?.name}</td>
                  <td className="px-4 py-3">{consultation.diagnosis}</td>
                  <td className="flex items-center gap-2 px-4 py-3">
                    <Link
                      href={route('consultations.show', consultation.id)}
                      className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                      title="View"
                    >
                      <Eye className="h-4 w-4" />
                    </Link>

                    {
                      canModify && (
                        <>
                          <Link
                            href={route('consultations.edit', consultation.id)}
                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                            title="Edit"
                          >
                            <Pencil className="h-4 w-4" />
                          </Link>
                          <button
                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                            title="Delete"
                          >
                            <Trash className="h-4 w-4" />
                          </button>

                        </>
                      )

                    }

                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={consultations.last_page}
            forcePage={consultations.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
