import AppLayout from '@/layouts/app-layout';
import { Head, usePage } from '@inertiajs/react';
import { useMemo, useState } from 'react';
import Swal from 'sweetalert2';
import {
  Download,
  Briefcase,
  CalendarCheck2,
  Stethoscope,
  Pill,
  Hospital,
  Activity,
  Ambulance,
  Users2,
  FileBarChart,
  Info,
  Package // ✅ for ZIP icon
} from 'lucide-react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';

type Props = {
  defaults: {
    year: number;
    month: number;
  };
};

const months = [
  { value: 1, label: 'January' },
  { value: 2, label: 'February' },
  { value: 3, label: 'March' },
  { value: 4, label: 'April' },
  { value: 5, label: 'May' },
  { value: 6, label: 'June' },
  { value: 7, label: 'July' },
  { value: 8, label: 'August' },
  { value: 9, label: 'September' },
  { value: 10, label: 'October' },
  { value: 11, label: 'November' },
  { value: 12, label: 'December' },
];

// Helper: try getting a named route; return null if not available
function tryRoute(name: string): string | null {
  try {
    // @ts-ignore ziggy provided globally
    return route(name);
  } catch {
    return null;
  }
}

type ReportKey =
  | 'payroll'
  | 'payrollByCompany'      // multi-sheet dependents + employees at end
  | 'payrollZipByCompany'   // ✅ NEW: ZIP with one XLSX per company
  | 'dispensations'
  | 'attendance'
  | 'diagnosis'
  | 'sickLeave'
  | 'iod'
  | 'referrals'
  | 'clinicStock'
  | 'nurseStock'
  | 'systemData';

type ReportDef = {
  key: ReportKey;
  title: string;
  desc: string;
  icon: any;
  routeName: string;
  filenameBase: string;
  fileExt?: 'xlsx' | 'zip';   // ✅ default xlsx
  comingSoon: boolean;
  badge: 'Monthly' | 'Inventory' | 'Master';
};

export default function ReportsPortal({ defaults }: Props) {
  const [year, setYear] = useState<number>(defaults.year);
  const [month, setMonth] = useState<number>(defaults.month);
  const [downloadingKey, setDownloadingKey] = useState<ReportKey | null>(null);

  // Read role from Inertia shared props (no restrictions applied anymore)
  const { auth } = usePage<{ auth?: { user?: { role?: string; roles?: string[] } } }>().props;

  // All reports (complete catalog)
  const reports = useMemo<Readonly<ReportDef[]>>(
    () => [
      {
        key: 'payroll',
        title: 'Payroll Report',
        desc: 'Employees & Dependents grouped. Includes triage defaults if missing.',
        icon: Briefcase,
        routeName: 'reports.payroll.monthly',
        filenameBase: 'Payroll_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      // ✅ Dependents by Company (multi-sheet workbook + Employees at end)
      {
        key: 'payrollByCompany',
        title: 'Dependents by Company (Payroll)',
        desc: 'Dependents only, one sheet per parent company. Employees sheet appended at the end.',
        icon: Users2,
        routeName: 'reports.payroll.dependents-by-company',
        filenameBase: 'Payroll_Dependents_ByCompany',
        comingSoon: false,
        badge: 'Monthly',
      },
      // ✅ NEW: ZIP with one XLSX per company (dependents only)
      {
        key: 'payrollZipByCompany',
        title: 'Dependents ZIP (Per Company)',
        desc: 'Creates one XLSX per company for dependents and bundles everything into a ZIP.',
        icon: Package,
        routeName: 'reports.payroll.dependents-zip-by-company',
        filenameBase: 'Payroll_Dependents_Per_Company',
        fileExt: 'zip',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'dispensations',
        title: 'Medication Dispensations',
        desc: 'All dispensed meds with patient, diagnosis & sick-leave tick. Grouped by clinic & role.',
        icon: Pill,
        routeName: 'reports.dispensations.monthly',
        filenameBase: 'Dispensations_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'attendance',
        title: 'Attendance',
        desc: 'Daily attendance sheets for the month (employees vs dependents).',
        icon: CalendarCheck2,
        routeName: 'reports.attendance.monthly',
        filenameBase: 'Attendance_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'diagnosis',
        title: 'Diagnosis Summary',
        desc: 'Monthly totals, daily breakdown, and counts by nurse.',
        icon: Stethoscope,
        routeName: 'reports.diagnosis.monthly',
        filenameBase: 'Diagnosis_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'sickLeave',
        title: 'Sick Leave',
        desc: 'Consultations with sick leave issued; durations & notes.',
        icon: FileBarChart,
        routeName: 'reports.sickleaves.monthly',
        filenameBase: 'Sick_Leave_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'iod',
        title: 'Injury on Duty',
        desc: 'IOD consultations grouped by company & day.',
        icon: Ambulance,
        routeName: 'reports.iod.monthly',
        filenameBase: 'Injury_On_Duty_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'referrals',
        title: 'Referrals',
        desc: 'Radiology, Lab & Specialist referrals per day.',
        icon: Activity,
        routeName: 'reports.referrals.monthly',
        filenameBase: 'Referrals_Report',
        comingSoon: false,
        badge: 'Monthly',
      },
      {
        key: 'clinicStock',
        title: 'Clinic Stock',
        desc: 'Clinic cabinet stock levels by medication & batch.',
        icon: Hospital,
        routeName: 'reports.clinic.stock.all',
        filenameBase: 'Clinic_Stock_Report',
        comingSoon: false,
        badge: 'Inventory',
      },
      {
        key: 'nurseStock',
        title: 'Nurse Stock Take',
        desc: 'Authenticated nurse’s stock take (cabinet).',
        icon: Users2,
        routeName: 'reports.nurse.stock.all',
        filenameBase: 'Nurse_Stock_Take',
        comingSoon: false,
        badge: 'Inventory',
      },
      // System Data (no month/year filter)
      {
        key: 'systemData',
        title: 'System Data Export',
        desc: 'All patients (account holders), dependents, and grouped view. No date filter.',
        icon: Info,
        routeName: 'reports.systemdata.all',
        filenameBase: 'System_Data',
        comingSoon: false,
        badge: 'Master',
      },
    ],
    []
  );

  // ✅ No limitations: everyone (including receptionists) sees all reports
  const visibleReports = reports;

  const downloadReport = async (r: ReportDef) => {
    // Only enforce month/year for monthly reports
    const needsPeriod = r.badge === 'Monthly';

    if (needsPeriod && (!year || !month)) {
      Swal.fire({ icon: 'warning', title: 'Select period', text: 'Please choose a valid month and year.' });
      return;
    }

    const endpoint = tryRoute(r.routeName);
    if (!endpoint) {
      Swal.fire({
        icon: 'info',
        title: 'Not available (yet)',
        text: `The "${r.title}" endpoint (${r.routeName}) isn’t wired on the server yet.`,
      });
      return;
    }

    try {
      setDownloadingKey(r.key);
      Swal.fire({
        title: `Preparing ${r.title}…`,
        html: 'Please wait while we generate your report.',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      });

      let url = endpoint;
      if (needsPeriod) {
        const params = new URLSearchParams({ year: String(year), month: String(month) }).toString();
        url = `${endpoint}?${params}`;
      }

      const res = await fetch(url, {
        method: 'GET',
        headers: { 'X-Requested-With': 'XMLHttpRequest' },
      });

      if (!res.ok) {
        Swal.close();
        const text = await res.text();
        throw new Error(text || 'Failed to download report.');
      }

      const blob = await res.blob();
      const objUrl = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = objUrl;

      // Filename uses proper extension (xlsx default; zip for the new ZIP report)
      const ext = r.fileExt ?? 'xlsx';
      a.download = needsPeriod
        ? `${r.filenameBase}_${year}_${String(month).padStart(2, '0')}.${ext}`
        : `${r.filenameBase}.${ext}`;

      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(objUrl);

      Swal.close();
      Swal.fire({ icon: 'success', title: 'Downloaded!', timer: 1500, showConfirmButton: false });
    } catch (e: any) {
      Swal.close();
      Swal.fire({ icon: 'error', title: 'Error', text: e?.message || 'Could not generate the report.' });
    } finally {
      setDownloadingKey(null);
    }
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Reports Portal', href: route('reports.portal') }]}>
      <Head title="Reports Portal" />

      <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-6 rounded-xl p-6 shadow-sm">
        {/* Header */}
        <div className="flex flex-wrap items-end justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold">Reports Portal</h1>
            <p className="text-muted-foreground mt-1 text-sm">
              Pick a month and year (where required), then download any report below.
            </p>
          </div>

          {/* Period selectors */}
          <div className="grid grid-cols-1 gap-3 sm:grid-cols-3">
            <div className="min-w-[180px]">
              <label className="mb-1 block text-sm font-medium">Month</label>
              <select
                value={month}
                onChange={(e) => setMonth(Number(e.target.value))}
                className="w-full rounded-lg border border-border bg-background p-2 text-sm"
              >
                {months.map((m) => (
                  <option key={m.value} value={m.value}>
                    {m.label}
                  </option>
                ))}
              </select>
            </div>

            <div className="min-w-[150px]">
              <label className="mb-1 block text-sm font-medium">Year</label>
              <input
                type="number"
                value={year}
                onChange={(e) => setYear(Number(e.target.value))}
                className="w-full rounded-lg border border-border bg-background p-2 text-sm"
                min={2000}
                max={2100}
              />
            </div>

            <div className="flex items-end">
              <Button
                onClick={() =>
                  Swal.fire({
                    icon: 'info',
                    title: 'How it works',
                    html:
                      '<div style="text-align:left">' +
                      '<p><strong>Monthly</strong> reports respect the Month & Year above.</p>' +
                      '<p><strong>System Data Export</strong> includes all patients (no date filter):</p>' +
                      '<ul style="margin-left:1rem">' +
                      '<li>Sheet 1: Account Holders + Company</li>' +
                      '<li>Sheet 2: Dependents + Holder details</li>' +
                      '<li>Sheet 3: Grouped — holder then dependents</li>' +
                      '</ul>' +
                      '</div>',
                  })
                }
                variant="outline"
                className="w-full gap-2"
              >
                <Info className="h-4 w-4" />
                How it works
              </Button>
            </div>
          </div>
        </div>

        <Separator />

        {/* Report grid */}
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4">
          {visibleReports.map((r) => {
            const Icon = r.icon;
            const isDownloading = downloadingKey === r.key;

            return (
              <Card key={r.key} className="flex h-full flex-col">
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <span className="inline-flex h-8 w-8 items-center justify-center rounded-full bg-primary/10">
                        <Icon className="h-4 w-4 text-primary" />
                      </span>
                      <CardTitle className="text-base">{r.title}</CardTitle>
                    </div>
                    <Badge variant={r.comingSoon ? 'secondary' : 'default'}>{r.badge}</Badge>
                  </div>
                  <CardDescription className="mt-2">{r.desc}</CardDescription>
                </CardHeader>

                <CardContent className="grow" />

                <CardFooter className="flex items-center justify-between gap-2">
                  <Button
                    className="inline-flex items-center gap-2"
                    onClick={() => downloadReport(r)}
                    disabled={isDownloading}
                    style={{ cursor: 'pointer' }}
                  >
                    <Download className="h-4 w-4" />
                    {isDownloading ? 'Generating…' : 'Download'}
                  </Button>

                  {r.comingSoon && (
                    <Badge variant="outline" className="text-xs">
                      Not wired yet
                    </Badge>
                  )}
                </CardFooter>
              </Card>
            );
          })}
        </div>
      </div>
    </AppLayout>
  );
}
