import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { FormEvent } from 'react';
import * as Yup from 'yup';

interface SupplierData {
    name: string;
    email: string;
    phone: string;
    contact_person: string;
    address: string;
    city: string;
    province: string;
    country: string;
    notes: string;
    [key: string]: any;
}

export default function CreateSupplier() {
    const { data, setData, post, processing, errors, reset } = useForm<SupplierData>({
        name: '',
        email: '',
        phone: '',
        contact_person: '',
        address: '',
        city: '',
        province: '',
        country: '',
        notes: '',
    });

    const schema = Yup.object().shape({
        name: Yup.string().required('Supplier name is required'),
        email: Yup.string().email('Invalid email').nullable(),
        phone: Yup.string().nullable(),
        contact_person: Yup.string().nullable(),
        address: Yup.string().nullable(),
        city: Yup.string().nullable(),
        province: Yup.string().nullable(),
        country: Yup.string().nullable(),
        notes: Yup.string().nullable(),
    });

    const handleSubmit = async (e: FormEvent) => {
        e.preventDefault();
        try {
            await schema.validate(data, { abortEarly: false });
            post(route('suppliers.store'), {
                preserveScroll: true,
                onSuccess: () => reset(),
            });
        } catch (validationError: any) {
            const fieldErrors: Record<string, string> = {};
            validationError.inner.forEach((err: any) => {
                if (err.path) fieldErrors[err.path] = err.message;
            });
            // Optional: update `errors` state if managing manually
        }
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Suppliers', href: route('suppliers.index') },
        { title: 'Create Supplier', href: route('suppliers.create') },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Create Supplier" />

            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                <h1 className="text-foreground text-2xl font-bold">Create Supplier</h1>

                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                        {[
                            'name',
                            'email',
                            'phone',
                            'contact_person',
                            'address',
                            'city',
                            'province',
                            'country',
                            'notes',
                        ].map((field) => (
                            <div key={field} className="grid gap-2">
                                <Label htmlFor={field}>
                                    {field.replace(/_/g, ' ').replace(/\b\w/g, (l) => l.toUpperCase())}
                                    {field === 'name' && <span className="text-red-500">*</span>}
                                </Label>
                                <Input
                                    id={field}
                                    name={field}
                                    type={field === 'email' ? 'email' : 'text'}
                                    value={data[field]}
                                    onChange={(e) => setData(field, e.target.value)}
                                />
                                <InputError message={errors[field]} />
                            </div>
                        ))}
                    </div>

                    <div className="flex justify-end gap-3">
                        <Button type="button" variant="outline" onClick={() => window.history.back()}>
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Saving...' : 'Save Supplier'}
                        </Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
