// resources/js/Pages/SystemCorrections/EmployeesWithoutGender.tsx

import AppLayout from '@/layouts/app-layout';
import { Head, router, usePage } from '@inertiajs/react';
import React, { useMemo, useState } from 'react';
import { type SharedData } from '@/types';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { toast } from 'sonner';
import { Check, X, Users, Save, Search as SearchIcon } from 'lucide-react';

type EmployeeRow = {
    id: number;
    full_name: string;
    employee_number?: string | null;
    company?: string | null;
    gender?: string | null;
    id_number?: string | null;
};

type PageProps = SharedData & {
    employees: {
        data: EmployeeRow[];
        current_page: number;
        last_page: number;
        per_page: number;
        total: number;
        links: { url: string | null; label: string; active: boolean }[];
    };
    filters: {
        search?: string;
        per_page?: number;
    };
};

export default function EmployeesWithoutGender() {
    const { employees, filters } = usePage<PageProps>().props;

    // two independent selections
    const [maleSet, setMaleSet] = useState<Set<number>>(new Set());
    const [femaleSet, setFemaleSet] = useState<Set<number>>(new Set());
    const [search, setSearch] = useState(filters.search ?? '');
    const [perPage, setPerPage] = useState<number>(Number(filters.per_page ?? 50));

    // toggle helpers - ensure a patient cannot be in both sets
    const toggleMale = (id: number, checked: boolean) => {
        setMaleSet(prev => {
            const next = new Set(prev);
            if (checked) {
                next.add(id);
                // remove from female if present
                setFemaleSet(f => {
                    const n = new Set(f);
                    n.delete(id);
                    return n;
                });
            } else {
                next.delete(id);
            }
            return next;
        });
    };

    const toggleFemale = (id: number, checked: boolean) => {
        setFemaleSet(prev => {
            const next = new Set(prev);
            if (checked) {
                next.add(id);
                // remove from male if present
                setMaleSet(m => {
                    const n = new Set(m);
                    n.delete(id);
                    return n;
                });
            } else {
                next.delete(id);
            }
            return next;
        });
    };

    const submit = () => {
        const male_ids = Array.from(maleSet);
        const female_ids = Array.from(femaleSet);

        router.post(
            route('system-corrections.gender.bulk'),
            { male_ids, female_ids },
            {
                preserveScroll: true,
                onSuccess: () => {
                    toast.success('Gender updated successfully.');
                    setMaleSet(new Set());
                    setFemaleSet(new Set());
                    // refresh list
                    router.reload({ only: ['employees'] });
                },
                onError: (errs) => {
                    const msg = Object.values(errs)[0] as string ?? 'Failed to update.';
                    toast.error(msg);
                },
            }
        );
    };

    const doSearch = (e: React.FormEvent) => {
        e.preventDefault();
        router.get(
            route('system-corrections.employees-without-gender'),
            { search, per_page: perPage },
            { preserveState: true, preserveScroll: true }
        );
    };

    const clearSelections = () => {
        setMaleSet(new Set());
        setFemaleSet(new Set());
    };

    const allCurrentIds = useMemo(() => employees.data.map(r => r.id), [employees.data]);

    const selectAllMaleOnPage = () => {
        setMaleSet(prev => {
            const next = new Set(prev);
            allCurrentIds.forEach(id => {
                next.add(id);
            });
            setFemaleSet(new Set()); // clear female on this page to avoid overlap
            return next;
        });
    };

    const selectAllFemaleOnPage = () => {
        setFemaleSet(prev => {
            const next = new Set(prev);
            allCurrentIds.forEach(id => {
                next.add(id);
            });
            setMaleSet(new Set()); // clear male on this page to avoid overlap
            return next;
        });
    };

    return (
        <AppLayout>
            <Head title="Employees Without Gender" />

            <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">



                <div className="flex items-center justify-between mb-4">
                    <div className="space-y-1">
                        <h1 className="text-2xl font-semibold tracking-tight flex items-center gap-2">
                            <Users className="h-6 w-6" />
                            Employees Without Gender
                        </h1>
                        <p className="text-muted-foreground">Select which employees should be marked Male vs Female, then submit.</p>
                    </div>
                </div>

                <Card className="mb-4">
                    <CardHeader className="pb-2">
                        <CardTitle className="text-base">Filters</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={doSearch} className="flex flex-col md:flex-row items-stretch md:items-end gap-3">
                            <div className="flex-1">
                                <label className="text-sm text-muted-foreground">Search (name, employee #, ID #)</label>
                                <div className="relative">
                                    <Input
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        placeholder="e.g. Jane, 12345, 63-123456Z63"
                                        className="pr-10"
                                    />
                                    <SearchIcon className="absolute right-2 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                </div>
                            </div>

                            <div className="w-full md:w-40">
                                <label className="text-sm text-muted-foreground">Per page</label>
                                <Input
                                    type="number"
                                    min={10}
                                    step={10}
                                    value={perPage}
                                    onChange={(e) => setPerPage(parseInt(e.target.value || '50'))}
                                />
                            </div>

                            <div className="flex gap-2">
                                <Button type="submit">Apply</Button>
                                <Button type="button" variant="secondary" onClick={() => { setSearch(''); setPerPage(50); }}>
                                    Reset
                                </Button>
                            </div>
                        </form>
                    </CardContent>
                </Card>

                <Card>
                    <CardHeader className="pb-2">
                        <CardTitle className="text-base">Select Genders</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="flex flex-col gap-3 md:flex-row md:items-center md:justify-between">
                            <div className="flex gap-2">
                                <Button variant="outline" onClick={selectAllMaleOnPage}><Check className="mr-2 h-4 w-4" />All Male (this page)</Button>
                                <Button variant="outline" onClick={selectAllFemaleOnPage}><X className="mr-2 h-4 w-4" />All Female (this page)</Button>
                                <Button variant="ghost" onClick={clearSelections}>Clear selections</Button>
                            </div>

                            <div className="flex items-center gap-2 text-sm">
                                <Badge variant="default">Male: {maleSet.size}</Badge>
                                <Badge variant="secondary">Female: {femaleSet.size}</Badge>
                                <Separator orientation="vertical" className="h-5" />
                                <span className="text-muted-foreground">Total found: {employees.total}</span>
                            </div>
                        </div>

                        <div className="mt-4 overflow-x-auto">
                            <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead className="w-[60px] text-center">Male</TableHead>
                                        <TableHead className="w-[60px] text-center">Female</TableHead>
                                        <TableHead>Employee</TableHead>
                                        <TableHead>Employee #</TableHead>
                                        <TableHead>Company</TableHead>
                                        <TableHead>ID Number</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {employees.data.map((row) => {
                                        const isMale = maleSet.has(row.id);
                                        const isFemale = femaleSet.has(row.id);

                                        return (
                                            <TableRow key={row.id}>
                                                <TableCell className="text-center">
                                                    <Checkbox
                                                        checked={isMale}
                                                        onCheckedChange={(val) => toggleMale(row.id, Boolean(val))}
                                                        aria-label={`Mark ${row.full_name} as Male`}
                                                        style={{
                                                            cursor: 'pointer'
                                                        }}
                                                    />
                                                </TableCell>
                                                <TableCell className="text-center">
                                                    <Checkbox
                                                        checked={isFemale}
                                                        onCheckedChange={(val) => toggleFemale(row.id, Boolean(val))}
                                                        aria-label={`Mark ${row.full_name} as Female`}
                                                        style={{
                                                            cursor: 'pointer'
                                                        }}
                                                    />
                                                </TableCell>
                                                <TableCell>
                                                    <div className="font-medium">{row.full_name}</div>
                                                </TableCell>
                                                <TableCell>{row.employee_number ?? '-'}</TableCell>
                                                <TableCell>{row.company ?? '-'}</TableCell>
                                                <TableCell>{row.id_number ?? '-'}</TableCell>
                                            </TableRow>
                                        );
                                    })}
                                </TableBody>
                            </Table>
                        </div>

                        {/* Simple pagination via Inertia links, or use your ReactPaginate component */}
                        <div className="flex flex-wrap gap-2 mt-4">
                            {employees.links.map((lnk, idx) => (
                                <Button
                                    key={idx}
                                    variant={lnk.active ? 'default' : 'outline'}
                                    disabled={!lnk.url}
                                    onClick={() => lnk.url && router.visit(lnk.url, { preserveState: true, preserveScroll: true })}
                                    className="h-8"
                                >
                                    {/* labels come like "&laquo; Previous" etc. */}
                                    <span dangerouslySetInnerHTML={{ __html: lnk.label }} />
                                </Button>
                            ))}
                        </div>

                        <div className="flex justify-end mt-6">
                            <Button onClick={submit}>
                                <Save className="mr-2 h-4 w-4" />
                                Submit Changes
                            </Button>
                        </div>
                    </CardContent>
                </Card>

            </div>

        </AppLayout>
    );
}
