import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import { Head, useForm, usePage } from '@inertiajs/react';
import { AlertCircle, ClipboardList, CreditCard, FileText, RefreshCw, User, Info } from 'lucide-react';
import { useEffect } from 'react';
import Swal from 'sweetalert2';

interface TriageFormData {
  [key: string]: any;
  patient_id: string;
  clinic_id: string;
  nurse_id: string;
  consultor_id: string;
  triage_level: string;
  bill_type: string;
  bill_class: string;
  payment_point: string;
  payment_method: string;
  chief_complaint: string;
  notes: string;
}

interface PageProps extends InertiaPageProps {
  clinics: any[];
  nurses: any[];
  companies: Array<{ id: number; name: string }>;
  patients: any[];
  selectedPatient?: {
    id: string | number;
    company_id: string | number | null;
    parent_patient_id: string | number | null;
    parent_company_id?: string | number | null;
    parent_employee_number?: string | null;
    parent_employee_gender?: string | null;
  } | null;
  // NEW props from backend:
  accountHolder?: {
    id: number | string;
    full_name: string;
    employee_number?: string | null;
    company_id?: number | string | null;
    company_name?: string | null;
  } | null;
  accountHolderFund?: {
    on_fund: boolean;
    tag?: string;
    employee_number?: string | null;
    company_name?: string | null;
  } | null;

  triageLevels: string[];
  billTypes: string[];
  billClasses: string[];
  paymentMethods: string[];
  patient_id?: string | number;
  auth: {
    user: {
      id: string | number;
      clinic_id: string | number | null;
      role: string;
    };
  };
}

export default function TriageCreate() {
  const {
    nurses,
    companies,
    patients,
    triageLevels,
    billTypes,
    billClasses,
    paymentMethods,
    patient_id: qpPatientId,
    selectedPatient,
    accountHolder,
    accountHolderFund,
    auth,
  } = usePage<PageProps>().props;

  const initialPatientId: string =
    selectedPatient?.id !== undefined && selectedPatient?.id !== null
      ? String(selectedPatient.id)
      : qpPatientId
      ? String(qpPatientId)
      : '';

  const initialPaymentPoint: string = selectedPatient
    ? String(selectedPatient.parent_company_id ?? selectedPatient.company_id ?? '')
    : '';

  const { data, setData, post, processing, errors } = useForm<TriageFormData>({
    patient_id: initialPatientId,
    clinic_id: auth.user.clinic_id ? String(auth.user.clinic_id) : '',
    nurse_id: String(auth.user.id),
    consultor_id: '',
    triage_level: 'Routine',
    bill_type: '',
    bill_class: '',
    payment_point: initialPaymentPoint,
    payment_method: '',
    chief_complaint: '',
    notes: '',
  });

  const handleReset = () => {
    setData({
      patient_id: initialPatientId,
      clinic_id: auth.user.clinic_id ? String(auth.user.clinic_id) : '',
      nurse_id: String(auth.user.id),
      consultor_id: '',
      triage_level: 'Routine',
      bill_type: '',
      bill_class: '',
      payment_point: initialPaymentPoint,
      payment_method: '',
      chief_complaint: '',
      notes: '',
    });
  };

  // keep payment_point synced with selected patient
  useEffect(() => {
    if (data.patient_id && patients.length > 0) {
      const pid = Number(data.patient_id);
      const patient = patients.find((p) => p.id === pid);
      if (patient) {
        const ppId = patient.parent_patient_id ? Number(patient.parent_patient_id) : null;
        const parent = ppId ? patients.find((p) => p.id === ppId) : null;
        const computedPaymentPoint =
          (parent?.company_id != null ? String(parent.company_id) : '') ||
          (patient.company_id != null ? String(patient.company_id) : '');
        setData('payment_point', computedPaymentPoint || '');
      }
    }
  }, [data.patient_id, patients, setData]);

  // ---------- validation ----------
  const REQUIRED_LABELS: Record<
    Exclude<keyof TriageFormData, 'chief_complaint' | 'notes'>,
    string
  > = {
    patient_id: 'Patient',
    clinic_id: 'Clinic',
    nurse_id: 'Nurse',
    consultor_id: 'Consulting RGN / Doctor',
    triage_level: 'Triage Level',
    bill_type: 'Bill Type',
    bill_class: 'Bill Class',
    payment_point: 'Payment Point (Company)',
    payment_method: 'Payment Method',
  };

  const REQUIRED_KEYS = Object.keys(REQUIRED_LABELS) as Array<
    Exclude<keyof TriageFormData, 'chief_complaint' | 'notes'>
  >;

  const isEmpty = (v: any) =>
    v === null || v === undefined || (typeof v === 'string' && v.trim() === '');

  const validateRequired = (): string[] => {
    const missing: string[] = [];
    for (const k of REQUIRED_KEYS) {
      if (isEmpty(data[k])) {
        missing.push(REQUIRED_LABELS[k]);
      }
    }
    return missing;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    const missing = validateRequired();
    if (missing.length > 0) {
      const list = `<ul style="text-align:left; margin:.5rem 0 0 1rem; list-style:disc;">
        ${missing.map((m) => `<li><b>${m}</b></li>`).join('')}
      </ul>`;
      await Swal.fire({
        icon: 'error',
        title: 'Missing required fields',
        html: `<p>Please complete the following before saving:</p>${list}`,
        confirmButtonText: 'OK',
      });
      return;
    }

    post(route('triages.store'));
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Triage', href: '#' },
  ];

  // Derive account-holder info for the sidebar (fallbacks if backend not provided for some reason)
  const sidebarAccountHolderName =
    accountHolder?.full_name ??
    (() => {
      if (!data.patient_id) return '';
      const pid = Number(data.patient_id);
      const me = patients.find((p) => p.id === pid);
      if (!me) return '';
      if (me.parent_patient_id) {
        const parent = patients.find((p) => p.id === Number(me.parent_patient_id));
        return parent ? `${parent.first_name} ${parent.surname}` : `${me.first_name} ${me.surname}`;
      }
      return `${me.first_name} ${me.surname}`;
    })();

  const sidebarAccountHolderEmpNo =
    accountHolder?.employee_number ??
    selectedPatient?.parent_employee_number ??
    (() => {
      if (!data.patient_id) return '';
      const pid = Number(data.patient_id);
      const me = patients.find((p) => p.id === pid);
      if (!me) return '';
      if (me.parent_patient_id) {
        const parent = patients.find((p) => p.id === Number(me.parent_patient_id));
        return parent?.employee_number ?? '';
      }
      return me.employee_number ?? '';
    })();

  const sidebarAccountHolderCompanyName =
    accountHolder?.company_name ??
    (() => {
      if (!data.patient_id) return undefined;
      const pid = Number(data.patient_id);
      const me = patients.find((p) => p.id === pid);
      if (!me) return undefined;
      const companyId = me.parent_patient_id
        ? patients.find((p) => p.id === Number(me.parent_patient_id))?.company_id
        : me.company_id;
      if (!companyId) return undefined;
      return companies.find((c) => c.id === Number(companyId))?.name;
    })();

  const fund = accountHolderFund ?? { on_fund: false };

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Create Triage Record" />
      <div className="w-full px-4 py-6 sm:px-6">
        <div className="grid gap-6 sm:grid-cols-12">
          <div className="col-span-12 space-y-6 sm:col-span-8">
            <div className="mb-6 flex justify-end">
              <Button
                type="button"
                variant="outline"
                onClick={handleReset}
                className="flex items-center gap-2"
                style={{ cursor: 'pointer' }}
              >
                <RefreshCw className="h-4 w-4" />
                Reset Form
              </Button>
            </div>

            <form onSubmit={handleSubmit} className="space-y-8">
              {/* Patient & Clinic */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                  <User className="h-5 w-5" />
                  Patient & Clinic Information
                </h2>

                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div>
                    <Label htmlFor="patient_id">Patient</Label>
                    <select
                      id="patient_id"
                      value={data.patient_id}
                      onChange={(e) => setData('patient_id', e.target.value)}
                      disabled={Boolean(initialPatientId)}
                      className={`w-full rounded-md border px-3 py-2 text-sm shadow-sm ${
                        initialPatientId
                          ? 'cursor-not-allowed border-gray-300 bg-gray-200 text-gray-500 dark:border-gray-700 dark:bg-gray-700 dark:text-gray-400'
                          : 'border-gray-300 bg-white text-gray-900 focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500'
                      }`}
                    >
                      <option value="">Select Patient</option>
                      {patients.map((patient: any) => (
                        <option key={patient.id} value={patient.id}>
                          {patient.first_name} {patient.surname}
                        </option>
                      ))}
                    </select>
                    {errors.patient_id && <p className="mt-1 text-sm text-red-600">{errors.patient_id}</p>}
                  </div>

                  <div>
                    <Label htmlFor="consultor_id">Consulting RGN / Doctor</Label>
                    <select
                      id="consultor_id"
                      value={data.consultor_id}
                      onChange={(e) => setData('consultor_id', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Consultor</option>
                      {nurses.map((nurse: any) => (
                        <option key={nurse.id} value={nurse.id}>
                          {nurse.name}
                        </option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>

              {/* Triage Level */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                  <AlertCircle className="h-5 w-5" />
                  Triage Level
                </h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-4">
                  {triageLevels.map((level: string) => (
                    <div key={level} className="flex items-center">
                      <input
                        type="radio"
                        id={`triage_level_${level}`}
                        name="triage_level"
                        value={level}
                        checked={data.triage_level === level}
                        onChange={(e) => setData('triage_level', e.target.value)}
                        className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                      />
                      <Label htmlFor={`triage_level_${level}`} className="ml-2">
                        {level}
                      </Label>
                    </div>
                  ))}
                </div>

                <div className="mt-4 rounded-md border border-blue-200 bg-blue-50 p-3 text-sm text-blue-800 dark:border-blue-900/40 dark:bg-blue-950/40 dark:text-blue-200">
                  <p className="flex items-start gap-2">
                    <Info className="mt-0.5 h-4 w-4 shrink-0" />
                    <span>
                      <strong>Why “Routine” by default?</strong> New triage records default to <em>Routine</em> to
                      reflect a standard, non-urgent visit. If the patient presents urgent or emergent symptoms,
                      please select the appropriate higher triage level.
                    </span>
                  </p>
                </div>

                {errors.triage_level && <p className="mt-2 text-sm text-red-600">{errors.triage_level}</p>}
              </div>

              {/* Billing */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                  <CreditCard className="h-5 w-5" />
                  Billing Information
                </h2>

                <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                  <div>
                    <Label htmlFor="bill_type">Bill Type</Label>
                    <select
                      id="bill_type"
                      value={data.bill_type}
                      onChange={(e) => setData('bill_type', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Bill Type</option>
                      {billTypes.map((type: string) => (
                        <option key={type} value={type}>
                          {type}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <Label htmlFor="bill_class">Bill Class</Label>
                    <select
                      id="bill_class"
                      value={data.bill_class}
                      onChange={(e) => setData('bill_class', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Bill Class</option>
                      {billClasses.map((cls: string) => (
                        <option key={cls} value={cls}>
                          {cls}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <Label htmlFor="payment_method">Payment Method</Label>
                    <select
                      id="payment_method"
                      value={data.payment_method}
                      onChange={(e) => setData('payment_method', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Payment Method</option>
                      {paymentMethods.map((method: string) => (
                        <option key={method} value={method}>
                          {method}
                        </option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>

              <div className="flex justify-end gap-4">
                <Button type="button" variant="outline" onClick={() => window.history.back()} style={{ cursor: 'pointer' }}>
                  Cancel
                </Button>
                <Button type="submit" disabled={processing} style={{ cursor: 'pointer' }}>
                  {processing ? 'Saving...' : 'Save Triage Record'}
                </Button>
              </div>
            </form>
          </div>

          {/* Right Sidebar */}
          <div className="col-span-12 space-y-6 sm:col-span-4">
            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
              <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                <ClipboardList className="h-5 w-5" />
                Patient Information
              </h2>

              {data.patient_id ? (
                (() => {
                  const patient = patients.find((p) => p.id === Number(data.patient_id));
                  const isDependent = !!patient?.parent_patient_id;
                  const parent = isDependent ? patients.find((p) => p.id === Number(patient!.parent_patient_id)) : null;

                  const companyId = parent?.company_id ?? patient?.company_id;
                  const company = companyId ? companies.find((c) => c.id === Number(companyId)) : undefined;

                  // --- DISPLAY: Account Holder Employee Number ---
                  const accountHolderEmpNo =
                    sidebarAccountHolderEmpNo || '(not provided)';

                  return (
                    <div className="space-y-2 text-sm">
                      <p>
                        <strong>Name:</strong> {patient?.first_name} {patient?.surname}
                      </p>

                      <div className="flex items-center space-x-2">
                        <strong>{isDependent ? 'Parent Company:' : 'Company:'}</strong>
                        <span className="inline-block rounded-full bg-blue-100 px-3 py-1 text-xs font-medium text-blue-800">
                          {company?.name ?? 'N/A'}
                        </span>
                      </div>

                      <p>
                        <strong>National ID Number:</strong> {patient?.id_number ?? '—'}
                      </p>

                      {/* NEW: Account Holder Employee Number (always the account holder’s) */}
                      <p>
                        <strong>Account Holder Employee #:</strong> {accountHolderEmpNo || '—'}
                      </p>

                      {isDependent && parent && (
                        <div className="space-y-1">
                          <p>
                            <strong>Parent:</strong> {parent.first_name} {parent.surname}
                          </p>
                          <p>
                            <strong>Parent National ID:</strong> {parent.id_number ?? '—'}
                          </p>
                        </div>
                      )}
                    </div>
                  );
                })()
              ) : (
                <p className="text-muted-foreground">No patient selected</p>
              )}
            </div>

            {/* NEW: Medical Fund status for Account Holder */}
            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
              <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                <FileText className="h-5 w-5" />
                Medical Fund (Account Holder)
              </h2>

              {accountHolder ? (
                fund.on_fund ? (
                  <div className="rounded-md border border-emerald-200 bg-emerald-50 p-3 text-sm text-emerald-800 dark:border-emerald-900/40 dark:bg-emerald-900/20 dark:text-emerald-200">
                    <div className="font-medium">{accountHolder.full_name}</div>
                    <div className="mt-1 grid grid-cols-1 gap-1 sm:grid-cols-2">
                      <div>
                        <span className="text-muted-foreground">Employee #:</span>{' '}
                        {fund.employee_number || '—'}
                      </div>
                      <div>
                        <span className="text-muted-foreground">Company:</span>{' '}
                        {fund.company_name || accountHolder.company_name || '—'}
                      </div>
                    </div>
                    <div className="mt-1 text-xs">
                      On <strong>{fund.tag || 'Providence Fund'}</strong>
                    </div>
                  </div>
                ) : (
                  <div className="rounded-md border border-rose-200 bg-rose-50 p-3 text-sm text-rose-800 dark:border-rose-900/40 dark:bg-rose-900/20 dark:text-rose-200">
                    <div className="font-medium">{accountHolder.full_name}</div>
                    <div className="mt-1">Not on medical fund.</div>
                  </div>
                )
              ) : (
                <div className="text-sm text-muted-foreground">Select a patient to see fund status.</div>
              )}
            </div>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
