import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import { Head, useForm, usePage } from '@inertiajs/react';
import { AlertCircle, Building, ClipboardList, CreditCard, FileText, RefreshCw, User } from 'lucide-react';

interface TriageFormData {
    [key: string]: any;
    patient_id: string;
    clinic_id: string;
    nurse_id: string;
    consultor_id: string;
    consultation_id: string;
    triage_level: string;
    bill_type: string;
    bill_class: string;
    payment_point: string;
    payment_method: string;
    chief_complaint: string;
    notes: string;
}

interface PageProps extends InertiaPageProps {
    triage: any;
    clinics: any[];
    nurses: any[];
    companies: any[];
    patients: any[];
    triageLevels: string[];
    billTypes: string[];
    billClasses: string[];
    paymentMethods: string[];
    auth: {
        user: {
            id: string;
            clinic_id: string | null;
            role: string;
        };
    };
}

export default function TriageEdit() {
    const { triage, clinics, nurses, companies, patients, triageLevels, billTypes, billClasses, paymentMethods, auth } = usePage<PageProps>().props;

    const { data, setData, put, processing, errors } = useForm<TriageFormData>({
        patient_id: triage.patient_id,
        clinic_id: triage.clinic_id,
        nurse_id: triage.nurse_id,
        consultor_id: triage.consultor_id,
        consultation_id: triage.consultation_id,
        triage_level: triage.triage_level,
        bill_type: triage.bill_type,
        bill_class: triage.bill_class,
        payment_point: triage.payment_point,
        payment_method: triage.payment_method,
        chief_complaint: triage.chief_complaint,
        notes: triage.notes,
    });

    const handleReset = () => {
        setData({
            patient_id: triage.patient_id,
            clinic_id: triage.clinic_id,
            nurse_id: triage.nurse_id,
            consultor_id: triage.consultor_id,
            consultation_id: triage.consultation_id,
            triage_level: triage.triage_level,
            bill_type: triage.bill_type,
            bill_class: triage.bill_class,
            payment_point: triage.payment_point,
            payment_method: triage.payment_method,
            chief_complaint: triage.chief_complaint,
            notes: triage.notes,
        });
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        put(route('triages.update', triage.id));
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Patients', href: route('patients.index') },
        { title: 'Triage', href: route('triages.index') },
        { title: 'Edit', href: '#' },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Edit Triage Record" />
            <div className="w-full px-4 py-6 sm:px-6">
                <div className="grid gap-6 sm:grid-cols-12">
                    <div className="col-span-12 space-y-6 sm:col-span-8">
                        <div className="mb-6 flex justify-end">
                            <Button
                                type="button"
                                variant="outline"
                                onClick={handleReset}
                                className="flex items-center gap-2"
                                style={{
                                    cursor: 'pointer',
                                }}
                            >
                                <RefreshCw className="h-4 w-4" />
                                Reset Form
                            </Button>
                        </div>
                        <form onSubmit={handleSubmit} className="space-y-8">
                            {/* Patient and Clinic Information */}
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                                    <User className="h-5 w-5" />
                                    Patient & Clinic Information
                                </h2>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                                    <div>
                                        <Label htmlFor="patient_id">Patient</Label>
                                        <select
                                            id="patient_id"
                                            value={data.patient_id}
                                            onChange={(e) => setData('patient_id', e.target.value)}
                                            disabled
                                            className={`w-full rounded-md border px-3 py-2 text-sm shadow-sm ${
                                                data.patient_id
                                                    ? 'cursor-not-allowed border-gray-300 bg-gray-200 text-gray-500 dark:border-gray-700 dark:bg-gray-700 dark:text-gray-400'
                                                    : 'border-gray-300 bg-white text-gray-900 focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500'
                                            }`}
                                        >
                                            <option value="">Select Patient</option>
                                            {patients.map((patient: any) => (
                                                <option key={patient.id} value={patient.id}>
                                                    {patient.first_name} {patient.surname}
                                                </option>
                                            ))}
                                        </select>
                                        {errors.patient_id && <p className="mt-1 text-sm text-red-600">{errors.patient_id}</p>}
                                    </div>
                                    <div>
                                        <Label htmlFor="consultor_id">Consulting RGN / Doctor</Label>
                                        <select
                                            id="consultor_id"
                                            value={data.consultor_id}
                                            onChange={(e) => setData('consultor_id', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                                        >
                                            <option value="">Select Consultor</option>
                                            {nurses.map((nurse: any) => (
                                                <option key={nurse.id} value={nurse.id}>
                                                    {nurse.name}
                                                </option>
                                            ))}
                                        </select>
                                    </div>
                                </div>
                            </div>

                            {/* Triage Level */}
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                                    <AlertCircle className="h-5 w-5" />
                                    Triage Level
                                </h2>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-4">
                                    {triageLevels.map((level: string) => (
                                        <div key={level} className="flex items-center">
                                            <input
                                                type="radio"
                                                id={`triage_level_${level}`}
                                                name="triage_level"
                                                value={level}
                                                checked={data.triage_level === level}
                                                onChange={(e) => setData('triage_level', e.target.value)}
                                                className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                                            />
                                            <Label htmlFor={`triage_level_${level}`} className="ml-2">
                                                {level}
                                            </Label>
                                        </div>
                                    ))}
                                </div>
                                {errors.triage_level && <p className="mt-2 text-sm text-red-600">{errors.triage_level}</p>}
                            </div>

                            {/* Billing Information */}
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                                    <CreditCard className="h-5 w-5" />
                                    Billing Information
                                </h2>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                                    <div>
                                        <Label htmlFor="bill_type">Bill Type</Label>
                                        <select
                                            id="bill_type"
                                            value={data.bill_type}
                                            onChange={(e) => setData('bill_type', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                                        >
                                            <option value="">Select Bill Type</option>
                                            {billTypes.map((type: string) => (
                                                <option key={type} value={type}>
                                                    {type}
                                                </option>
                                            ))}
                                        </select>
                                    </div>
                                    <div>
                                        <Label htmlFor="bill_class">Bill Class</Label>
                                        <select
                                            id="bill_class"
                                            value={data.bill_class}
                                            onChange={(e) => setData('bill_class', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                                        >
                                            <option value="">Select Bill Class</option>
                                            {billClasses.map((cls: string) => (
                                                <option key={cls} value={cls}>
                                                    {cls}
                                                </option>
                                            ))}
                                        </select>
                                    </div>
                                    <div>
                                        <Label htmlFor="payment_method">Payment Method</Label>
                                        <select
                                            id="payment_method"
                                            value={data.payment_method}
                                            onChange={(e) => setData('payment_method', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                                        >
                                            <option value="">Select Payment Method</option>
                                            {paymentMethods.map((method: string) => (
                                                <option key={method} value={method}>
                                                    {method}
                                                </option>
                                            ))}
                                        </select>
                                    </div>
                                </div>
                            </div>

                            {/* Clinical Information */}
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                                    <Building className="h-5 w-5" />
                                    Clinical Information
                                </h2>
                                <div className="space-y-4">
                                    <div>
                                        <Label htmlFor="chief_complaint">Chief Complaint</Label>
                                        <textarea
                                            id="chief_complaint"
                                            value={data.chief_complaint}
                                            onChange={(e) => setData('chief_complaint', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                                            rows={3}
                                        />
                                        {errors.chief_complaint && <p className="mt-1 text-sm text-red-600">{errors.chief_complaint}</p>}
                                    </div>
                                    <div>
                                        <Label htmlFor="notes">Notes</Label>
                                        <textarea
                                            id="notes"
                                            value={data.notes}
                                            onChange={(e) => setData('notes', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                                            rows={3}
                                        />
                                    </div>
                                </div>
                            </div>

                            <div className="flex justify-end gap-4">
                                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={processing}>
                                    {processing ? 'Saving...' : 'Save Triage Record'}
                                </Button>
                            </div>
                        </form>
                    </div>

                    {/* Right Sidebar - Could be used for patient history or other info */}
                    <div className="col-span-12 space-y-6 sm:col-span-4">
                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                                <ClipboardList className="h-5 w-5" />
                                Patient Information
                            </h2>
                            {data.patient_id ? (
                                (() => {
                                    const patient = patients.find((p) => p.id === parseInt(data.patient_id));
                                    const isDependent = !!patient?.parent_patient_id;
                                    const parent = isDependent ? patients.find((p) => p.id === parseInt(patient.parent_patient_id!)) : null;

                                    const companyId = parent?.company_id ?? patient?.company_id;
                                    const company = companies.find((c) => c.id === parseInt(companyId));

                                    return (
                                        <div className="space-y-2 text-sm">
                                            <p>
                                                <strong>Name:</strong> {patient?.first_name} {patient?.surname}
                                            </p>

                                            <div className="flex items-center space-x-2">
                                                <strong>{isDependent ? 'Parent Company:' : 'Company:'}</strong>
                                                <span className="inline-block rounded-full bg-blue-100 px-3 py-1 text-xs font-medium text-blue-800">
                                                    {company?.name ?? 'N/A'}
                                                </span>
                                            </div>

                                            <p>
                                                <strong>National ID Number:</strong> {patient?.id_number}
                                            </p>

                                            {isDependent && parent && (
                                                <div className="space-y-1">
                                                    <p>
                                                        <strong>Parent:</strong> {parent.first_name} {parent.surname}
                                                    </p>
                                                    <p>
                                                        <strong>Parent National ID:</strong> {parent.id_number}
                                                    </p>
                                                </div>
                                            )}
                                        </div>
                                    );
                                })()
                            ) : (
                                <p className="text-muted-foreground">No patient selected</p>
                            )}
                        </div>

                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                                <FileText className="h-5 w-5" />
                                Recent Triage Notes
                            </h2>
                            <div className="space-y-2 text-sm">
                                <p className="text-muted-foreground">No previous triage records found</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
