import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Plus, Search, Trash, Download } from 'lucide-react';
import moment from 'moment';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import * as XLSX from 'xlsx';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Triage', href: '/triage' }];

interface PagePropsWithTriage extends PageProps {
    triages: {
        data: any;
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    clinics: Array<{ id: number; name: string }>;
    consultors: Array<{ id: number; name: string }>;
    filters: {
        search?: string;
        clinic_id?: number;
        consultor_id?: number;
        triage_level?: string;
        bill_type?: string;
        bill_class?: string;
        payment_method?: string;
        start_date?: string;
        end_date?: string;
    };
}

type AuthUser = {
    id: number;
    name: string;
    role?: string;
    roles?: string[];
    can?: Record<string, boolean>;
};

export default function TriageIndex() {
    const { triages, clinics, consultors, filters } = usePage<PagePropsWithTriage>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');
    const [startDate, setStartDate] = useState(filters.start_date || '');
    const [endDate, setEndDate] = useState(filters.end_date || '');
    const [clinicId, setClinicId] = useState(filters.clinic_id || '');
    const [consultorId, setConsultorId] = useState(filters.consultor_id || '');
    const [triageLevel, setTriageLevel] = useState(filters.triage_level || '');
    const [billType, setBillType] = useState(filters.bill_type || '');
    const [billClass, setBillClass] = useState(filters.bill_class || '');
    const [paymentMethod, setPaymentMethod] = useState(filters.payment_method || '');


    const { auth } = usePage<{ auth: { user: AuthUser } }>().props;

    // visible if role is exactly "superadmin" (with a couple of safe fallbacks)
    const canModify =
        auth?.user?.role === 'superadmin' ||
        auth?.user?.roles?.includes?.('superadmin') ||
        auth?.user?.role === 'receptionist' ||                 // ✅ allow receptionist
        auth?.user?.roles?.includes?.('receptionist') ||       // ✅ allow receptionist (multi-role)
        auth?.user?.can?.delete_consultations === true;


    const handleFilter = () => {
        router.get(
            `${API}/triages`,
            {
                search: searchQuery,
                start_date: startDate,
                end_date: endDate,
                clinic_id: clinicId,
                consultor_id: consultorId,
                triage_level: triageLevel,
                bill_type: billType,
                bill_class: billClass,
                payment_method: paymentMethod,
            },
            { preserveState: true },
        );
    };

    const handleResetFilters = () => {
        setSearchQuery('');
        setStartDate('');
        setEndDate('');
        setClinicId('');
        setConsultorId('');
        setTriageLevel('');
        setBillType('');
        setBillClass('');
        setPaymentMethod('');
        router.get(`${API}/triages`, {}, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This triage record will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('triage.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The triage record has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    const exportToExcel = () => {
        const data = triages.data.map((triage: any) => ({
            Date: moment(triage.created_at).format('MMMM Do YYYY'),
            'Patient Name': triage.patient?.first_name || '',
            'Last Name': triage.patient?.surname || '',
            Clinic: triage.clinic?.name || '',
            'Assigned Consultor/Doctor': triage.consultor?.name || '',
            'Triage Level': triage.triage_level || '',
            'Bill Type': triage.bill_type || '',
            'Bill Class': triage.bill_class || '',
            'Payment Method': triage.payment_method || '',
        }));

        // Create worksheet
        const worksheet = XLSX.utils.json_to_sheet(data);

        // Get the keys from first data object (all column headers)
        const keys = data.length > 0 ? Object.keys(data[0]) : [];

        // Calculate max length for each column dynamically
        const cols = keys.map((key) => {
            const maxLength = Math.max(
                key.length,
                ...data.map((row: { [x: string]: { toString: () => { (): any; new(): any; length: any; }; }; }) => (row[key] ? row[key].toString().length : 0))
            );
            return { wch: maxLength + 2 };
        });

        // Assign column widths to worksheet
        worksheet['!cols'] = cols;

        // Create workbook
        const workbook = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(workbook, worksheet, 'Triages');

        // Generate file and download
        XLSX.writeFile(workbook, 'triages_export.xlsx', {
            compression: true,
        });
    };


    const handlePageChange = (selected: { selected: number }) => {
        router.get(
            `${API}/triage`,
            {
                page: selected.selected + 1,
                search: searchQuery,
                start_date: startDate,
                end_date: endDate,
                clinic_id: clinicId,
                consultor_id: consultorId,
                triage_level: triageLevel,
                bill_type: billType,
                bill_class: billClass,
                payment_method: paymentMethod,
            },
            { preserveState: true },
        );
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Triage" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Triage Records</h1>
                    <div className="flex items-center gap-3">
                        <button
                            onClick={exportToExcel}
                            className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-4 py-2 text-sm font-medium text-white hover:bg-green-700"
                            style={{
                                cursor: 'pointer',
                            }}
                        >
                            <Download className="h-4 w-4" />
                            Export Excel
                        </button>

                        <Link
                            href={route('triages.create')}
                            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700"
                        >
                            <Plus className="h-4 w-4" />
                            Add Triage
                        </Link>
                    </div>

                </div>

                {/* Accordion for Triage Explanation */}
                <div className="rounded-md border border-gray-200 bg-white dark:border-gray-700 dark:bg-gray-900">
                    <details className="group p-4">
                        <summary className="flex cursor-pointer list-none items-center justify-between text-sm font-medium text-gray-800 dark:text-gray-100">
                            <span>What is Triage?</span>
                            <svg
                                className="ml-2 h-5 w-5 transition-transform group-open:rotate-180"
                                fill="none"
                                stroke="currentColor"
                                strokeWidth="2"
                                viewBox="0 0 24 24"
                            >
                                <path strokeLinecap="round" strokeLinejoin="round" d="M19 9l-7 7-7-7" />
                            </svg>
                        </summary>
                        <div className="mt-4 text-sm text-gray-700 dark:text-gray-300">
                            <p className="mb-2">
                                <strong>Triage</strong> is a critical process in clinical settings used to determine the priority of patients’
                                treatments based on the severity of their condition. It helps ensure that those with life-threatening issues receive
                                immediate medical attention while others are queued appropriately.
                            </p>
                            <p className="mb-2">
                                During triage, a healthcare professional—typically a nurse—records vital signs, chief complaints, and clinical
                                observations. Based on this, the patient is assigned a <em>triage level</em>, such as{' '}
                                <strong>Emergency, Urgent, Routine,</strong> or <strong>Non-Urgent</strong>.
                            </p>
                            <p>
                                This structured assessment facilitates efficient patient flow, optimizes use of resources, and ensures that high-risk
                                individuals receive timely interventions, which is especially vital in emergency departments, outpatient clinics, and
                                during mass casualty incidents.
                            </p>
                        </div>
                    </details>
                </div>

                {/* Filter Section */}
                <div className="bg-card mb-6 rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
                    <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
                        {/* Search */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Search</label>
                            <div className="relative">
                                <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                                <input
                                    type="text"
                                    value={searchQuery}
                                    onChange={(e) => setSearchQuery(e.target.value)}
                                    placeholder="Search..."
                                    className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                                />
                            </div>
                        </div>

                        {/* Date Range */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Date Range</label>
                            <div className="grid grid-cols-2 gap-2">
                                <input
                                    type="date"
                                    value={startDate}
                                    onChange={(e) => setStartDate(e.target.value)}
                                    className="border-border bg-background rounded-lg border p-2 text-sm"
                                />
                                <input
                                    type="date"
                                    value={endDate}
                                    onChange={(e) => setEndDate(e.target.value)}
                                    className="border-border bg-background rounded-lg border p-2 text-sm"
                                />
                            </div>
                        </div>

                        {/* Clinic Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Clinic</label>
                            <select
                                value={clinicId}
                                onChange={(e) => setClinicId(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Clinics</option>
                                {clinics.map((clinic) => (
                                    <option key={clinic.id} value={clinic.id}>
                                        {clinic.name}
                                    </option>
                                ))}
                            </select>
                        </div>

                        {/* Consultor Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Consultor</label>
                            <select
                                value={consultorId}
                                onChange={(e) => setConsultorId(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Consultors</option>
                                {consultors.map((consultor) => (
                                    <option key={consultor.id} value={consultor.id}>
                                        {consultor.name}
                                    </option>
                                ))}
                            </select>
                        </div>

                        {/* Triage Level Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Triage Level</label>
                            <select
                                value={triageLevel}
                                onChange={(e) => setTriageLevel(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Levels</option>
                                <option value="Emergency">Emergency</option>
                                <option value="Urgent">Urgent</option>
                                <option value="Routine">Routine</option>
                                <option value="Non-Urgent">Non-Urgent</option>
                            </select>
                        </div>

                        {/* Bill Type Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Bill Type</label>
                            <select
                                value={billType}
                                onChange={(e) => setBillType(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Types</option>
                                <option value="Consultation Adult">Consultation Adult</option>
                                <option value="Consultation Child">Consultation Child</option>
                                <option value="Follow-up">Follow-up</option>
                                <option value="Emergency">Emergency</option>
                                <option value="Vaccination">Vaccination</option>
                                <option value="Lab Test">Lab Test</option>
                                <option value="Radiology">Radiology</option>
                                <option value="Review Adult">Review Adult</option>
                                <option value="Review Child">Review Child</option>
                                <option value="Antigen Test">Antigen Test</option>
                                <option value="PCR Test">PCR Test</option>
                                <option value="Rapid Test">Rapid Test</option>
                                <option value="Vaccine">Vaccine</option>
                                <option value="Lead">Lead</option>
                                <option value="Treatment">Treatment</option>
                                <option value="Other">Other</option>
                            </select>
                        </div>

                        {/* Bill Class Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Bill Class</label>
                            <select
                                value={billClass}
                                onChange={(e) => setBillClass(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Classes</option>
                                <option value="Self">Self</option>
                                <option value="Family">Family</option>
                                <option value="Parent">Parent</option>
                                <option value="Extended Family">Extended Family</option>
                                <option value="Other">Other</option>
                            </select>
                        </div>

                        {/* Payment Method Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Payment Method</label>
                            <select
                                value={paymentMethod}
                                onChange={(e) => setPaymentMethod(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Methods</option>
                                <option value="Payroll">Payroll</option>
                                <option value="Pensioner">Pensioner</option>
                                <option value="Attachee">Attachee</option>
                                <option value="Customer">Customer</option>
                                <option value="Cash">Cash</option>
                                <option value="Insurance">Insurance</option>
                                <option value="Minerva">Minerva</option>
                                <option value="PSMAS (Cash/5)">PSMAS (Cash/5)</option>
                                <option value="FLIMAS">FLIMAS</option>
                                <option value="Cimas">Cimas</option>
                                <option value="EMF">EMF</option>
                                <option value="Alliance">Alliance</option>
                                <option value="Generation">Generation</option>
                                <option value="FML">FML</option>
                                <option value="Bonvie">Bonvie</option>
                                <option value="Fidelity">Fidelity</option>
                                <option value="First Mutual">First Mutual</option>
                                <option value="Mars">Mars</option>
                                <option value="Cellmed">Cellmed</option>
                                <option value="Railmed">Railmed</option>
                                <option value="NMAS">NMAS</option>
                                <option value="Mammoth">Mammoth</option>
                                <option value="Emerald">Emerald</option>
                                <option value="Varichem">Varichem</option>
                            </select>
                        </div>
                    </div>

                    <div className="mt-4 flex justify-end gap-2">
                        <button
                            onClick={handleResetFilters}
                            className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
                        >
                            Reset Filters
                        </button>
                        <button onClick={handleFilter} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                            Apply Filters
                        </button>
                    </div>
                </div>

                {/* Table Section */}
                <div className="overflow-x-auto rounded-xl border">
                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {[
                                    'Date',
                                    'Patient Name',
                                    'Last Name',
                                    'Clinic',
                                    'Assigned Consultor/Doctor',
                                    'Triage Level',
                                    'Bill Type',
                                    'Bill Class',
                                    'Payment Method',
                                    'Actions',
                                ].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {triages.data.map((triage: any) => (
                                <tr key={triage.id} className="hover:bg-muted/50 transition">
                                    <td className="px-4 py-3">{moment(triage.created_at).format('MMMM Do YYYY')}</td>
                                    <td className="px-4 py-3">{triage.patient?.first_name}</td>
                                    <td className="px-4 py-3">{triage.patient?.surname}</td>
                                    <td className="px-4 py-3">{triage.clinic?.name}</td>
                                    <td className="px-4 py-3">{triage.consultor?.name}</td>
                                    <td className="px-4 py-3">
                                        <span
                                            className={`inline-flex items-center rounded-full px-3 py-1 text-xs font-semibold ${triage.triage_level === 'Emergency'
                                                ? 'bg-red-100 text-red-800'
                                                : triage.triage_level === 'Urgent'
                                                    ? 'bg-orange-100 text-orange-800'
                                                    : triage.triage_level === 'Routine'
                                                        ? 'bg-yellow-100 text-yellow-800'
                                                        : triage.triage_level === 'Non-Urgent'
                                                            ? 'bg-green-100 text-green-800'
                                                            : 'bg-gray-100 text-gray-800'
                                                }`}
                                        >
                                            {triage.triage_level}
                                        </span>
                                    </td>

                                    <td className="px-4 py-3">{triage.bill_type}</td>
                                    <td className="px-4 py-3">{triage.bill_class}</td>
                                    <td className="px-4 py-3">{triage.payment_method}</td>
                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <Link
                                            href={route('triages.show', triage.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                                            title="View"
                                        >
                                            <Eye className="h-4 w-4" />
                                        </Link>

                                        {
                                            canModify && (
                                                <>

                                                    <Link
                                                        href={route('triages.edit', triage.id)}
                                                        className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                                        title="Edit"
                                                    >
                                                        <Pencil className="h-4 w-4" />
                                                    </Link>

                                                    <button
                                                        onClick={() => handleDelete(triage.id)}
                                                        className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                                        title="Delete"
                                                    >
                                                        <Trash className="h-4 w-4" />
                                                    </button>
                                                </>
                                            )
                                        }

                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                {/* Pagination Section */}
                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={triages.last_page}
                        forcePage={triages.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}
