import { Button } from '@/components/ui/button';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, usePage } from '@inertiajs/react';
import { QRCodeSVG } from 'qrcode.react';
import { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';

const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';

interface Triage {
    id: number;
    patient_id: number;
    clinic_id: number;
    nurse_id: number;
    consultor_id: number;
    consultation_id: number | null;
    triage_level: string | null;
    bill_type: string | null;
    bill_class: string | null;
    payment_point: number | null;
    payment_method: string | null;
    chief_complaint: string | null;
    notes: string | null;
    created_at: string;
    updated_at: string;
    patient: {
        id: number;
        first_name: string;
        surname: string;
        id_number: string;
        company_id: number | null;
    };
    clinic: {
        id: number;
        name: string;
    };
    consultor: {
        id: number;
        name: string;
    };
    paymentCompany: {
        id: number;
        name: string;
    };
}

export default function TriageShow() {
    const { triage } = usePage<{ triage: Triage }>().props;

    const contentRef = useRef<HTMLDivElement>(null);
    const reactToPrintFn = useReactToPrint({ contentRef });

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Triage', href: route('triages.index') },
        { title: `${triage.patient.first_name} ${triage.patient.surname}`, href: route('triages.show', triage.id) },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`Triage: ${triage.patient.first_name} ${triage.patient.surname}`} />

            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                {/* Header: Image + Name + Buttons */}
                <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                    <div className="flex items-center gap-4">
                        <div className="bg-muted text-muted-foreground flex h-24 w-24 items-center justify-center rounded-full border">
                            <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
                        </div>
                        <div>
                            <h1 className="text-foreground text-2xl font-bold">
                                {triage.patient.first_name} {triage.patient.surname}
                            </h1>
                        </div>
                    </div>

                    {/* Action Buttons */}
                    <div className="flex flex-wrap gap-2">
                        <Button
                            onClick={reactToPrintFn}
                            variant="outline"
                            style={{
                                cursor: 'pointer',
                            }}
                        >
                            Print Triage Record
                        </Button>
                    </div>
                </div>

                <div className="grid grid-cols-12 gap-8">
                    <div className="col-span-8 space-y-6">
                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-4 text-xl font-semibold">Triage Information</h2>
                            <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                                <div>
                                    <div className="text-muted-foreground text-sm">Triage Level</div>
                                    <div className="text-foreground font-medium">{triage.triage_level || '-'}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">Bill Type</div>
                                    <div className="text-foreground font-medium">{triage.bill_type || '-'}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">Bill Class</div>
                                    <div className="text-foreground font-medium">{triage.bill_class || '-'}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">Payment Method</div>
                                    <div className="text-foreground font-medium">{triage.payment_method || '-'}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">Chief Complaint</div>
                                    <div className="text-foreground font-medium">{triage.chief_complaint || '-'}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">Notes</div>
                                    <div className="text-foreground font-medium">{triage.notes || '-'}</div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div className="col-span-4">
                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-4 text-xl font-semibold">Patient Information</h2>
                            <div className="grid grid-cols-1 gap-4">
                                <div>
                                    <div className="text-muted-foreground text-sm">Patient Information</div>
                                    <div className="text-foreground font-medium">#{triage.patient.id} {triage.patient.first_name} {triage.patient.surname}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">ID Number</div>
                                    <div className="text-foreground font-medium">{triage.patient.id_number}</div>
                                </div>
                                <div>
                                    <div className="text-muted-foreground text-sm">Company</div>
                                    <div className="text-foreground font-medium">{triage.patient.company_id ? 'Company Name' : '-'}</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                    <h2 className="mb-4 text-xl font-semibold">QR Code</h2>
                    <div className="flex justify-center">
                        <QRCodeSVG value={`Triage ID: ${triage.id}`} size={200} />
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
