import { Head, useForm } from '@inertiajs/react';
import { useMemo, useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import Swal from 'sweetalert2';

type Clinic = {
  id: number;
  name: string;
  city?: string | null;
};

type PageProps = {
  user: {
    id: number | string;
    name: string;
    email: string;
    clinic_id: number | null;
    role?: string | null;
  };
  clinics: Clinic[];
  selectedClinicIds: number[];
  primaryClinicId: number | null;
};

export default function ClinicAccessPage({
  user,
  clinics,
  selectedClinicIds,
  primaryClinicId,
}: PageProps) {
  const [query, setQuery] = useState<string>('');
  const [localSelected, setLocalSelected] = useState<Set<number>>(
    new Set(selectedClinicIds)
  );

  // useForm for loading state & visit helpers
  const { processing, put, transform } = useForm<{ clinic_ids: number[] }>({
    clinic_ids: Array.from(localSelected),
  });

  const filtered = useMemo(() => {
    const q = query.trim().toLowerCase();
    if (!q) return clinics;
    return clinics.filter(
      (c) =>
        c.name.toLowerCase().includes(q) ||
        (c.city ? c.city.toLowerCase().includes(q) : false)
    );
  }, [clinics, query]);

  const toggle = (id: number) => {
    // Prevent unchecking primary clinic
    if (primaryClinicId && id === primaryClinicId) return;

    const next = new Set(localSelected);
    if (next.has(id)) next.delete(id);
    else next.add(id);
    setLocalSelected(next);
  };

  const selectAll = () => {
    const next = new Set<number>(clinics.map((c) => c.id));
    if (primaryClinicId) next.add(primaryClinicId); // ensure primary included
    setLocalSelected(next);
  };

  const clearAll = () => {
    const next = new Set<number>();
    if (primaryClinicId) next.add(primaryClinicId); // keep primary forced
    setLocalSelected(next);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    // Inject the current selection into the payload at submit time
    transform((current) => ({
      ...current,
      clinic_ids: Array.from(localSelected),
    }));

    put(route('users.clinic-access.update', user.id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({
          icon: 'success',
          title: 'Saved',
          text: 'Clinic access updated successfully.',
          timer: 1200,
          showConfirmButton: false,
        });
      },
      onError: (errs) => {
        const msg =
          errs && typeof errs === 'object'
            ? Object.values(errs).join('\n')
            : 'Failed to save.';
        Swal.fire({ icon: 'error', title: 'Error', text: msg });
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Users', href: route('users.index') },
        { title: user.name, href: route('users.show', user.id) },
        { title: 'Clinic Access', href: '#' },
      ]}
    >
      <Head title={`Clinic Access • ${user.name}`} />

      {/* Full-width container */}
      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 rounded-lg border p-5">
          <div className="flex flex-col gap-1">
            <h1 className="text-xl font-semibold">Clinic Access for {user.name}</h1>
            <p className="text-sm text-muted-foreground">{user.email}</p>
            {primaryClinicId && (
              <p className="text-xs text-blue-600">
                Primary clinic is always included and cannot be removed.
              </p>
            )}
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          {/* Search + bulk actions full-width */}
          <div className="flex flex-col gap-3 sm:flex-row sm:items-end sm:justify-between">
            <div className="sm:max-w-md w-full">
              <Label htmlFor="search">Search clinics</Label>
              <Input
                id="search"
                value={query}
                onChange={(e) => setQuery(e.target.value)}
                placeholder="Type clinic name or city..."
              />
            </div>

            <div className="flex gap-2">
              <Button
                type="button"
                variant="secondary"
                onClick={selectAll}
                style={{ cursor: 'pointer' }}
              >
                Select all
              </Button>
              <Button
                type="button"
                variant="outline"
                onClick={clearAll}
                style={{ cursor: 'pointer' }}
              >
                Clear
              </Button>
            </div>
          </div>

          {/* List spans full width */}
          <div className="rounded-lg border">
            <div className="max-h-[70vh] overflow-auto divide-y">
              {filtered.map((c) => {
                const checked =
                  localSelected.has(c.id) ||
                  (primaryClinicId ? c.id === primaryClinicId : false);
                const disabled = primaryClinicId ? c.id === primaryClinicId : false;

                return (
                  <label
                    key={c.id}
                    className={`flex cursor-pointer items-center justify-between p-3 ${
                      disabled ? 'opacity-70' : ''
                    }`}
                  >
                    <div className="flex items-center gap-3">
                      <input
                        type="checkbox"
                        className="h-5 w-5"
                        checked={checked}
                        onChange={() => toggle(c.id)}
                        disabled={disabled}
                      />
                      <div>
                        <div className="font-medium">{c.name}</div>
                        {c.city ? (
                          <div className="text-xs text-muted-foreground">{c.city}</div>
                        ) : null}
                      </div>
                    </div>
                    {disabled && (
                      <span className="rounded-full bg-blue-600 px-2 py-0.5 text-xs font-medium text-white">
                        Primary
                      </span>
                    )}
                  </label>
                );
              })}

              {filtered.length === 0 && (
                <div className="p-6 text-center text-sm text-muted-foreground">
                  No clinics match your search.
                </div>
              )}
            </div>
          </div>

          <div className="flex justify-end gap-3">
            <Button type="button" variant="outline" onClick={() => window.history.back()}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing} style={{ cursor: 'pointer' }}>
              {processing ? 'Saving…' : 'Save Changes'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
