import { Head, useForm } from '@inertiajs/react';
import { useMemo, useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import Swal from 'sweetalert2';

type Company = { id: number; name: string };

type PageProps = {
  user: { id: number | string; name: string; email: string; role?: string | null };
  companies: Company[];
  selectedCompanyIds: number[];
};

export default function CompanyAccessPage({ user, companies, selectedCompanyIds }: PageProps) {
  const [query, setQuery] = useState('');
  const [localSelected, setLocalSelected] = useState<Set<number>>(new Set(selectedCompanyIds));

  const { processing, put, transform } = useForm<{ company_ids: number[] }>({
    company_ids: Array.from(localSelected),
  });

  const filtered = useMemo(() => {
    const q = query.trim().toLowerCase();
    if (!q) return companies;
    return companies.filter((c) => c.name.toLowerCase().includes(q));
  }, [companies, query]);

  // 🔹 Live list of selected companies (reflects current checkboxes)
  const selectedCompanies = useMemo(
    () => companies.filter((c) => localSelected.has(c.id)),
    [companies, localSelected]
  );

  const toggle = (id: number) => {
    const next = new Set(localSelected);
    if (next.has(id)) next.delete(id);
    else next.add(id);
    setLocalSelected(next);
  };

  const selectAll = () => setLocalSelected(new Set<number>(companies.map((c) => c.id)));
  const clearAll = () => setLocalSelected(new Set<number>());

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    transform((current) => ({ ...current, company_ids: Array.from(localSelected) }));

    put(route('users.company-access.update', user.id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({
          icon: 'success',
          title: 'Saved',
          text: 'Company access updated.',
          timer: 1200,
          showConfirmButton: false,
        });
      },
      onError: (errs) => {
        const msg = errs && typeof errs === 'object' ? Object.values(errs).join('\n') : 'Failed to save.';
        Swal.fire({ icon: 'error', title: 'Error', text: msg });
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Users', href: route('users.index') },
        { title: user.name, href: route('users.show', user.id) },
        { title: 'Company Access', href: '#' },
      ]}
    >
      <Head title={`Company Access • ${user.name}`} />

      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 rounded-lg border p-5">
          <div className="flex flex-col gap-1">
            <h1 className="text-xl font-semibold">Company Access for {user.name}</h1>
            <p className="text-sm text-muted-foreground">{user.email}</p>
          </div>

          {/* 🔹 Current access summary */}
          <div className="mt-4">
            <div className="mb-2 text-sm font-medium">
              Current access ({selectedCompanies.length}):
            </div>

            {selectedCompanies.length > 0 ? (
              <div className="flex flex-wrap gap-2">
                {selectedCompanies.map((c) => (
                  <span
                    key={c.id}
                    className="inline-flex items-center rounded-full border px-3 py-1 text-xs font-medium"
                    title={c.name}
                  >
                    {c.name}
                  </span>
                ))}
              </div>
            ) : (
              <div className="text-xs text-muted-foreground">No companies selected.</div>
            )}
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-end sm:justify-between">
            <div className="sm:max-w-md w-full">
              <Label htmlFor="search">Search companies</Label>
              <Input
                id="search"
                value={query}
                onChange={(e) => setQuery(e.target.value)}
                placeholder="Type company name..."
              />
            </div>

            <div className="flex gap-2">
              <Button type="button" variant="secondary" onClick={selectAll} style={{ cursor: 'pointer' }}>
                Select all
              </Button>
              <Button type="button" variant="outline" onClick={clearAll} style={{ cursor: 'pointer' }}>
                Clear
              </Button>
            </div>
          </div>

          <div className="rounded-lg border">
            <div className="max-h-[70vh] overflow-auto divide-y">
              {filtered.map((c) => {
                const checked = localSelected.has(c.id);
                return (
                  <label key={c.id} className="flex cursor-pointer items-center justify-between p-3">
                    <div className="flex items-center gap-3">
                      <input
                        type="checkbox"
                        className="h-5 w-5"
                        checked={checked}
                        onChange={() => toggle(c.id)}
                      />
                      <div className="font-medium">{c.name}</div>
                    </div>
                  </label>
                );
              })}

              {filtered.length === 0 && (
                <div className="p-6 text-center text-sm text-muted-foreground">No companies match your search.</div>
              )}
            </div>
          </div>

          <div className="flex justify-end gap-3">
            <Button type="button" variant="outline" onClick={() => window.history.back()}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing} style={{ cursor: 'pointer' }}>
              {processing ? 'Saving…' : 'Save Changes'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
