import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, router, useForm } from '@inertiajs/react';
import { FormEvent, useCallback, useMemo, useState } from 'react';
import * as Yup from 'yup';
import { useDropzone } from 'react-dropzone';
import { ImageIcon, X, UploadCloud } from 'lucide-react';
import Swal from 'sweetalert2';

interface EditUserProps {
  user: {
    id: number;
    name: string;
    email: string;
    clinic_id: number | null;
    role: string;
    qualifications: string | null;
    signature: string | null;
  };
  clinics: { id: number; name: string }[];
  roles: string[];
}

interface UserFormData {
  [key: string]: any;
  name: string;
  email: string;
  clinic_id: number | null;
  role: string;
  qualifications: string;
  signature: File | string | null;
}

export default function EditUser({ user, clinics, roles }: EditUserProps) {
  // ----- Main user form -----
  const { data, setData, processing, errors } = useForm<UserFormData>({
    name: user.name,
    email: user.email,
    clinic_id: user.clinic_id,
    role: user.role,
    qualifications: user.qualifications || '',
    signature: user.signature || null,
  });

  const schema = Yup.object().shape({
    name: Yup.string().required('Name is required'),
    email: Yup.string().email('Invalid email').required('Email is required'),
    clinic_id: Yup.number().nullable(),
    role: Yup.string().required('Role is required'),
    qualifications: Yup.string().max(255, 'Qualifications must not exceed 255 characters'),
    signature: Yup.mixed().nullable(),
  });

  // ----- Signature Dropzone state -----
  const [droppedFile, setDroppedFile] = useState<File | null>(null);

  const onDrop = useCallback((accepted: File[]) => {
    if (!accepted || accepted.length === 0) return;
    const file = accepted[0];
    setDroppedFile(file);
    setData('signature', file);
  }, [setData]);

  const { getRootProps, getInputProps, isDragActive, open } = useDropzone({
    onDrop,
    multiple: false,
    accept: { 'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp'] },
    noClick: true,
    maxSize: 2 * 1024 * 1024,
  });

  const previewUrl = useMemo(() => {
    if (droppedFile) return URL.createObjectURL(droppedFile);
    if (typeof data.signature === 'string' && data.signature) {
      return `/storage/${data.signature}`;
    }
    return '';
  }, [droppedFile, data.signature]);

  const clearSignature = () => {
    setDroppedFile(null);
    setData('signature', null);
  };

  // ----- Submit main user form -----
  const handleSubmit = async (e: FormEvent) => {
    e.preventDefault();
    try {
      await schema.validate(data, { abortEarly: false });

      const formData = new FormData();
      formData.append('name', data.name);
      formData.append('email', data.email);
      formData.append('clinic_id', data.clinic_id ? String(data.clinic_id) : '');
      formData.append('role', data.role);
      formData.append('qualifications', data.qualifications);
      formData.append('_method', 'put');

      if (data.signature instanceof File) {
        formData.append('signature', data.signature);
      } else if (data.signature === null) {
        formData.append('signature', '');
      }

      router.post(route('users.update', user.id), formData, {
        preserveScroll: true,
        forceFormData: true,
        onSuccess: () => {
          Swal.fire({
            icon: 'success',
            title: 'Updated',
            text: 'User updated successfully.',
            confirmButtonColor: '#16a34a',
          });
        },
        onError: (errs) => {
          const firstKey = Object.keys(errs)[0];
          Swal.fire({
            icon: 'error',
            title: 'Update failed',
            text: (firstKey && (errs as any)[firstKey]) || 'Please review the form and try again.',
            confirmButtonColor: '#ef4444',
          });
        },
      });
    } catch (validationError: any) {
      const list: string[] = validationError?.inner?.map((e: any) => e.message) ?? [];
      Swal.fire({
        icon: 'error',
        title: 'Validation error',
        html: list.length
          ? `<ul style="text-align:left; margin:0 12px;">${list.map((m) => `<li>${m}</li>`).join('')}</ul>`
          : 'Please review the form.',
        confirmButtonColor: '#ef4444',
      });
    }
  };

  // ----- Change Password (by email) -----
  const [pw1, setPw1] = useState('');
  const [pw2, setPw2] = useState('');
  const [changing, setChanging] = useState(false);

  const changePassword = (e: FormEvent) => {
    e.preventDefault();
    if (!pw1 || !pw2) {
      Swal.fire({ icon: 'warning', title: 'Missing password', text: 'Please fill in both password fields.' });
      return;
    }
    if (pw1 !== pw2) {
      Swal.fire({ icon: 'error', title: 'Password mismatch', text: 'Passwords do not match.' });
      return;
    }

    setChanging(true);
    router.post(
      route('users.changePasswordByEmail'),
      { email: data.email, password: pw1, password_confirmation: pw2 },
      {
        preserveScroll: true,
        onSuccess: async () => {
          setPw1('');
          setPw2('');
          await Swal.fire({
            icon: 'success',
            title: 'Password changed',
            text: 'Password updated successfully.',
            confirmButtonColor: '#16a34a',
          });
          // ⬇️ redirect back to Users index after success
          router.visit(route('users.index'));
        },
        onError: (errs) => {
          const firstKey = Object.keys(errs)[0];
          Swal.fire({
            icon: 'error',
            title: 'Failed to change password',
            text: (firstKey && (errs as any)[firstKey]) || 'Please try again.',
            confirmButtonColor: '#ef4444',
          });
        },
        onFinish: () => setChanging(false),
      }
    );
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Users', href: route('users.index') },
    { title: 'Edit User', href: route('users.edit', user.id) },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Edit User" />

      <div className="w-full space-y-8 px-4 py-10 sm:px-6">
        <h1 className="text-foreground text-2xl font-bold">Edit User</h1>

        <form onSubmit={handleSubmit} className="space-y-8">
          <div className="grid grid-cols-1 gap-6 lg:grid-cols-3">
            {/* Left: Basic details */}
            <Card className="lg:col-span-2">
              <CardHeader>
                <CardTitle>Basic Details</CardTitle>
              </CardHeader>
              <CardContent className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                <div className="grid gap-2">
                  <Label htmlFor="name">
                    Name<span className="text-red-500">*</span>
                  </Label>
                  <Input
                    id="name"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                    className="w-full"
                  />
                  <InputError message={errors.name} />
                </div>

                <div className="grid gap-2">
                  <Label htmlFor="email">
                    Email<span className="text-red-500">*</span>
                  </Label>
                  <Input
                    id="email"
                    type="email"
                    value={data.email}
                    onChange={(e) => setData('email', e.target.value)}
                    className="w-full"
                  />
                  <InputError message={errors.email} />
                </div>

                <div className="grid gap-2">
                  <Label htmlFor="clinic_id">Clinic</Label>
                  <select
                    id="clinic_id"
                    name="clinic_id"
                    value={data.clinic_id ?? ''}
                    onChange={(e) =>
                      setData('clinic_id', e.target.value === '' ? null : Number(e.target.value))
                    }
                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                  >
                    <option value="">-- Select Clinic --</option>
                    {clinics.map((clinic) => (
                      <option key={clinic.id} value={clinic.id}>
                        {clinic.name}
                      </option>
                    ))}
                  </select>
                  <InputError message={errors.clinic_id} />
                </div>

                <div className="grid gap-2">
                  <Label htmlFor="role">
                    Role<span className="text-red-500">*</span>
                  </Label>
                  <select
                    id="role"
                    name="role"
                    value={data.role}
                    onChange={(e) => setData('role', e.target.value)}
                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                  >
                    <option value="">-- Select Role --</option>
                    {roles.map((role) => (
                      <option key={role} value={role}>
                        {role}
                      </option>
                    ))}
                  </select>
                  <InputError message={errors.role} />
                </div>

                <div className="grid gap-2 sm:col-span-2">
                  <Label htmlFor="qualifications">Qualifications</Label>
                  <Input
                    id="qualifications"
                    value={data.qualifications}
                    onChange={(e) => setData('qualifications', e.target.value)}
                    className="w-full"
                  />
                  <InputError message={errors.qualifications} />
                </div>
              </CardContent>
            </Card>

            {/* Right: Signature Dropzone */}
            <Card>
              <CardHeader>
                <CardTitle>Signature</CardTitle>
              </CardHeader>
              <CardContent>
                <div
                  {...getRootProps()}
                  className={[
                    'rounded-lg border border-dashed p-4',
                    isDragActive ? 'bg-emerald-50 border-emerald-300' : 'bg-transparent',
                  ].join(' ')}
                >
                  <input {...getInputProps()} />
                  {previewUrl ? (
                    <div className="relative">
                      <img
                        src={previewUrl}
                        alt="Signature preview"
                        className="h-40 w-full rounded-md object-contain bg-white"
                      />
                      <div className="mt-3 flex items-center gap-2">
                        <Button type="button" variant="outline" size="sm" onClick={open}>
                          Replace
                        </Button>
                        <Button type="button" variant="ghost" size="sm" onClick={clearSignature}>
                          <X className="mr-2 h-4 w-4" />
                          Remove
                        </Button>
                      </div>
                    </div>
                  ) : (
                    <div className="flex flex-col items-center justify-center py-10 text-center text-sm text-muted-foreground">
                      <div className="mb-3 rounded-full bg-muted p-3">
                        {isDragActive ? (
                          <UploadCloud className="h-6 w-6" />
                        ) : (
                          <ImageIcon className="h-6 w-6" />
                        )}
                      </div>
                      <p className="mb-2">Drag & drop signature image here</p>
                      <p className="text-xs">PNG, JPG, GIF — up to 2MB</p>
                      <div className="mt-4">
                        <Button type="button" variant="outline" onClick={open}>
                          Choose file
                        </Button>
                      </div>
                    </div>
                  )}
                </div>
                <InputError message={errors.signature} />
              </CardContent>
            </Card>
          </div>

          <div className="flex justify-end gap-3">
            <Button type="button" variant="outline" onClick={() => window.history.back()}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing}>
              {processing ? 'Updating.........' : 'Update User'}
            </Button>
          </div>
        </form>

        {/* Change Password (by email) */}
        <Separator className="my-6" />
        <Card>
          <CardHeader>
            <CardTitle>Change Password (by email)</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={changePassword} className="grid max-w-lg grid-cols-1 gap-4 sm:grid-cols-2">
              <div className="sm:col-span-2">
                <Label>Email</Label>
                <Input value={data.email} disabled className="bg-muted/50" />
                <p className="mt-1 text-xs text-muted-foreground">
                  Password will be updated for this email.
                </p>
              </div>

              <div className="grid gap-2">
                <Label htmlFor="pw1">New password</Label>
                <Input
                  id="pw1"
                  type="password"
                  value={pw1}
                  onChange={(e) => setPw1(e.target.value)}
                />
              </div>

              <div className="grid gap-2">
                <Label htmlFor="pw2">Confirm password</Label>
                <Input
                  id="pw2"
                  type="password"
                  value={pw2}
                  onChange={(e) => setPw2(e.target.value)}
                />
              </div>

              <div className="sm:col-span-2 flex justify-end">
                <Button type="submit" disabled={changing}>
                  {changing ? 'Changing…' : 'Change Password'}
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
