// resources/js/Pages/Users/Index.tsx
import AppLayout from '@/layouts/app-layout';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import {
  Pencil,
  Search,
  UserCheck,
  LogOut,
  Trash2,
  Columns3,
  Pill,
  Building2,
  FolderCog,
  Puzzle,
} from 'lucide-react';
import ReactPaginate from 'react-paginate';
import { Modal, Button as AntdButton, Form } from 'antd';
import Swal from 'sweetalert2';
import axios from 'axios';
import React, { useEffect, useRef, useState } from 'react';

// shadcn/ui
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import { Input as ShadcnInput } from '@/components/ui/input';
import { Button as ShadcnButton } from '@/components/ui/button';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  Select as ShadcnSelect,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuCheckboxItem,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
  TooltipProvider,
} from '@/components/ui/tooltip';

interface User {
  id: number;
  name: string;
  email: string;
  signature: string | null;
  signature_url: string | null;
  role: string;
  clinic_id: number | null;
  clinic?: { id: number; name: string };
}

interface Props extends PageProps {
  auth: {
    user: {
      id: number;
      name: string;
      email: string;
      role: string;
      signature: string | null;
    };
  };
  users: {
    data: User[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: { search?: string; role?: string | null };
  roles: string[];
  impersonation?: { active: boolean; impersonator_id?: number };
}

const ALL_SENTINEL = '__all__';
const LS_KEY_COLS = 'users.index.visibleColumns';

type ColumnKey =
  | 'name'
  | 'email'
  | 'clinic'
  | 'role'
  | 'signature'
  | 'actions'
  | 'user_stock';

type ColumnState = Record<ColumnKey, boolean>;

const DEFAULT_COLS: ColumnState = {
  name: true,
  email: false,
  clinic: true,
  role: true,
  signature: true,
  actions: true,
  user_stock: true,
};

// Debounce hook
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return debounced;
}

export default function UsersIndex() {
  const { users, filters, roles, auth, impersonation } = usePage<Props>().props;

  // search + role filter (null => no filter)
  const [searchQuery, setSearchQuery] = useState(filters.search || '');
  const [roleFilter, setRoleFilter] = useState<string | null>(filters.role ?? null);

  // Debounced search value
  const debouncedSearch = useDebouncedValue(searchQuery, 450);
  const firstRun = useRef(true);

  // change password modal
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [form] = Form.useForm();
  const [showPassword, setShowPassword] = useState(false);

  // delete state
  const [deletingUserId, setDeletingUserId] = useState<number | null>(null);

  const roleParam = roleFilter ?? '';
  const isImpersonating = Boolean((impersonation as any)?.active);

  // Admin can impersonate
  const currentUserRole = auth.user.role;
  const canCurrentUserImpersonate = currentUserRole === 'superadmin';

  // ---- DYNAMIC COLUMNS ----
  const [visibleCols, setVisibleCols] = useState<ColumnState>(() => {
    try {
      const raw = localStorage.getItem(LS_KEY_COLS);
      if (raw) {
        const parsed = JSON.parse(raw) as ColumnState;
        return { ...DEFAULT_COLS, ...parsed };
      }
    } catch {}
    return DEFAULT_COLS;
  });

  const allOn = Object.values(visibleCols).every(Boolean);

  const setCol = (key: ColumnKey, value: boolean) => {
    setVisibleCols((prev) => {
      const next = { ...prev, [key]: value };
      localStorage.setItem(LS_KEY_COLS, JSON.stringify(next));
      return next;
    });
  };

  const toggleAllCols = (value: boolean) => {
    const next = Object.keys(DEFAULT_COLS).reduce((acc, k) => {
      acc[k as ColumnKey] = value;
      return acc;
    }, {} as ColumnState);
    setVisibleCols(next);
    localStorage.setItem(LS_KEY_COLS, JSON.stringify(next));
  };

  const resetCols = () => {
    setVisibleCols(DEFAULT_COLS);
    localStorage.setItem(LS_KEY_COLS, JSON.stringify(DEFAULT_COLS));
  };

  const isVisible = (key: ColumnKey) => !!visibleCols[key];

  // -------------------------

  const handleSearch = () => {
    router.get(
      route('users.index'),
      { search: searchQuery, role: roleParam },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  // Auto-search when typing stops (debounced)
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    router.get(
      route('users.index'),
      { search: debouncedSearch, role: roleParam, page: 1 },
      { preserveState: true, replace: true, preserveScroll: true }
    );
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  const handleReset = () => {
    setSearchQuery('');
    setRoleFilter(null);
    router.get(route('users.index'), {}, { preserveState: true, replace: true, preserveScroll: true });
  };

  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      route('users.index'),
      { page: selected.selected + 1, search: searchQuery, role: roleParam },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  // Modal
  const showModal = (user: User) => {
    setSelectedUser(user);
    setIsModalOpen(true);
    form.setFieldsValue({ email: user.email, password: '', confirmPassword: '' });
  };

  const handleOk = () => {
    form.validateFields().then(async (values) => {
      try {
        const csrfToken = (document.querySelector('meta[name="csrf-token"]') as HTMLMetaElement)?.content || '';
        const response = await axios.post(
          route('users.changePasswordByEmail'),
          {
            email: values.email,
            password: values.password,
            password_confirmation: values.confirmPassword,
          },
          { headers: { 'X-CSRF-TOKEN': csrfToken } }
        );

        if (response.status === 200) {
          await Swal.fire({ icon: 'success', title: 'Success', text: 'Password changed successfully.' });
          setIsModalOpen(false);
          form.resetFields();
        } else {
          throw new Error('Failed to change password.');
        }
      } catch (error: any) {
        let errorMsg = 'An unexpected error occurred.';
        if (error?.response?.data?.errors) {
          errorMsg = Object.values(error.response.data.errors).flat().join('\n');
        } else if (error?.message) {
          errorMsg = error.message;
        }
        Swal.fire({ icon: 'error', title: 'Error', text: errorMsg });
      }
    });
  };

  const handleCancel = () => setIsModalOpen(false);
  const togglePasswordVisibility = () => setShowPassword((s) => !s);

  const roleBadge = (role: string) => {
    const classes =
      role === 'superadmin'
        ? 'bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-100'
        : role === 'doctor'
          ? 'bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-blue-100'
          : role === 'nurse'
            ? 'bg-emerald-100 text-emerald-800 dark:bg-emerald-800 dark:text-emerald-100'
            : 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-100';
    return <span className={`inline-block rounded-full px-3 py-1 text-xs font-semibold ${classes}`}>{role}</span>;
  };

  // Impersonation actions
  const startImpersonation = (userId: number) => {
    router.visit(route('impersonate', userId)); // GET
  };

  const stopImpersonation = () => {
    router.visit(route('impersonate.leave')); // GET
  };

  // Delete user (with Swal confirmation)
  const handleDeleteUser = async (user: User) => {
    // front-end guards
    if (user.id === auth.user.id) {
      await Swal.fire({ icon: 'warning', title: 'Not allowed', text: "You can't delete your own account." });
      return;
    }
    if (user.role === 'superadmin' && auth.user.role !== 'superadmin') {
      await Swal.fire({ icon: 'warning', title: 'Not allowed', text: 'Only a superadmin can delete another superadmin.' });
      return;
    }

    const { isConfirmed } = await Swal.fire({
      title: `Delete ${user.name}?`,
      icon: 'warning',
      html: `
        <div style="text-align:left">
          <p>This will permanently remove:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>User account</b></li>
            <li><b>Associated records</b> (consultations, notes, logs, etc.)</li>
          </ul>
          <p style="margin-top:.5rem;"><b>This action cannot be undone.</b></p>
        </div>
      `,
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      focusCancel: true,
    });

    if (!isConfirmed) return;

    setDeletingUserId(user.id);
    Swal.fire({ title: 'Deleting...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });

    router.delete(route('users.destroy', user.id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire('Deleted', 'User removed successfully.', 'success');
      },
      onError: (errors) => {
        const msg = errors && Object.values(errors).flat().join('\n');
        Swal.fire('Error', msg || 'Failed to delete user.', 'error');
      },
      onFinish: () => setDeletingUserId(null),
    });
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Users', href: route('users.index') }]}>
      <Head title="Users" />

      <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
          <h1 className="text-2xl font-bold">Users</h1>

          <div className="flex items-center gap-2">
            {/* Optional: Stop button if in impersonation mode */}
            {isImpersonating && (
              <ShadcnButton variant="destructive" onClick={stopImpersonation} className="inline-flex items-center gap-2">
                <LogOut className="h-4 w-4" />
                Stop Impersonating
              </ShadcnButton>
            )}

            {/* Columns selector */}
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <ShadcnButton variant="outline" className="inline-flex items-center gap-2">
                  <Columns3 className="h-4 w-4" />
                  Columns
                </ShadcnButton>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end" className="w-56">
                <DropdownMenuLabel>Toggle columns</DropdownMenuLabel>
                <DropdownMenuSeparator />
                {/* Quick actions */}
                <DropdownMenuItem onClick={() => toggleAllCols(true)} disabled={allOn}>
                  Select all
                </DropdownMenuItem>
                <DropdownMenuItem onClick={resetCols}>Reset</DropdownMenuItem>
                <DropdownMenuSeparator />

                {/* Checkboxes */}
                <DropdownMenuCheckboxItem
                  checked={isVisible('name')}
                  onCheckedChange={(v) => setCol('name', Boolean(v))}
                >
                  Name
                </DropdownMenuCheckboxItem>
                <DropdownMenuCheckboxItem
                  checked={isVisible('email')}
                  onCheckedChange={(v) => setCol('email', Boolean(v))}
                >
                  Email
                </DropdownMenuCheckboxItem>
                <DropdownMenuCheckboxItem
                  checked={isVisible('clinic')}
                  onCheckedChange={(v) => setCol('clinic', Boolean(v))}
                >
                  Assigned Clinic
                </DropdownMenuCheckboxItem>
                <DropdownMenuCheckboxItem
                  checked={isVisible('role')}
                  onCheckedChange={(v) => setCol('role', Boolean(v))}
                >
                  Role
                </DropdownMenuCheckboxItem>
                <DropdownMenuCheckboxItem
                  checked={isVisible('signature')}
                  onCheckedChange={(v) => setCol('signature', Boolean(v))}
                >
                  Signature
                </DropdownMenuCheckboxItem>
                <DropdownMenuCheckboxItem
                  checked={isVisible('actions')}
                  onCheckedChange={(v) => setCol('actions', Boolean(v))}
                >
                  Actions
                </DropdownMenuCheckboxItem>
                <DropdownMenuCheckboxItem
                  checked={isVisible('user_stock')}
                  onCheckedChange={(v) => setCol('user_stock', Boolean(v))}
                >
                  User Stock
                </DropdownMenuCheckboxItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>

        {/* Filters */}
        <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center">
          {/* Search */}
          <div className="relative w-full sm:w-80">
            <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
            <input
              type="text"
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)} // debounced
              onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
              placeholder="Search by name or email…"
              className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
            />
          </div>

          {/* Role filter */}
          <div className="w-full sm:w-64">
            <ShadcnSelect value={roleFilter ?? undefined} onValueChange={(val) => setRoleFilter(val === ALL_SENTINEL ? null : val)}>
              <SelectTrigger className="w-full">
                <SelectValue placeholder="All roles" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value={ALL_SENTINEL}>All roles</SelectItem>
                {roles.map((r) => (
                  <SelectItem key={r} value={r}>
                    {r.charAt(0).toUpperCase() + r.slice(1)}
                  </SelectItem>
                ))}
              </SelectContent>
            </ShadcnSelect>
          </div>

          {/* Actions */}
          <div className="flex gap-2">
            <ShadcnButton onClick={handleSearch}>Search</ShadcnButton>
            <ShadcnButton variant="outline" onClick={handleReset}>
              Reset
            </ShadcnButton>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-full">
            <TableHeader>
              <TableRow>
                {isVisible('name') && <TableHead className="uppercase">Name</TableHead>}
                {isVisible('email') && <TableHead className="uppercase">Email</TableHead>}
                {isVisible('clinic') && <TableHead className="uppercase">Assigned Clinic</TableHead>}
                {isVisible('role') && <TableHead className="uppercase">Role</TableHead>}
                {isVisible('signature') && <TableHead className="uppercase">Signature</TableHead>}
                {isVisible('actions') && <TableHead className="uppercase">Actions</TableHead>}
                {isVisible('user_stock') && <TableHead className="uppercase">User Stock</TableHead>}
              </TableRow>
            </TableHeader>
            <TableBody>
              {users.data.map((user) => {
                const isSelf = user.id === auth.user.id;
                const targetIsAdmin = user.role === 'superadmin';
                const showImpersonate = canCurrentUserImpersonate && !targetIsAdmin && !isSelf;

                const canDelete =
                  user.id !== auth.user.id && (auth.user.role === 'superadmin' || user.role !== 'superadmin');

                return (
                  <TableRow key={user.id} className="hover:bg-muted/50 transition">
                    {isVisible('name') && <TableCell className="py-3">{user.name}</TableCell>}
                    {isVisible('email') && <TableCell className="py-3">{user.email}</TableCell>}
                    {isVisible('clinic') && <TableCell className="py-3">{user.clinic?.name ?? '-'}</TableCell>}
                    {isVisible('role') && <TableCell className="py-3">{roleBadge(user.role)}</TableCell>}
                    {isVisible('signature') && (
                      <TableCell className="py-3">
                        {user.signature_url ? (
                          <img
                            src={user.signature_url}
                            onError={(e) => {
                              const target = e.currentTarget as HTMLImageElement;
                              const fallback = `http://providencehealth.online/public/storage/${user.signature}`;
                              if (user.signature && target.src !== fallback) target.src = fallback;
                            }}
                            alt="Signature"
                            style={{ height: '3.2rem' }}
                          />
                        ) : (
                          <span className="text-muted-foreground text-xs">No signature</span>
                        )}
                      </TableCell>
                    )}

                    {isVisible('actions') && (
                      <TableCell className="py-3">
                        <div className="flex flex-wrap items-center gap-2">
                          <a
                            href={route('users.edit', user.id)}
                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                            title="Edit"
                          >
                            <Pencil className="h-4 w-4" />
                          </a>

                          <ShadcnButton size="sm" variant="outline" onClick={() => showModal(user)} title="Change Password">
                            Change Password
                          </ShadcnButton>

                          {/* Impersonate */}
                          {showImpersonate && (
                            <ShadcnButton
                              size="sm"
                              onClick={() => startImpersonation(user.id)}
                              className="inline-flex items-center gap-2"
                              title={`Impersonate ${user.name}`}
                              style={{ cursor: 'pointer' }}
                            >
                              <UserCheck className="h-4 w-4" />
                              Impersonate
                            </ShadcnButton>
                          )}

                          {/* Delete user */}
                          <ShadcnButton
                            size="sm"
                            variant="destructive"
                            onClick={() => handleDeleteUser(user)}
                            disabled={!canDelete || deletingUserId === user.id}
                            title={canDelete ? 'Delete user' : 'Not allowed'}
                            className="inline-flex items-center gap-2"
                            style={{ cursor: !canDelete || deletingUserId === user.id ? 'not-allowed' : 'pointer' }}
                          >
                            <Trash2 className="h-4 w-4" />
                            {deletingUserId === user.id ? 'Deleting…' : 'Delete'}
                          </ShadcnButton>

                          {/* Optional: Stop impersonation per-row */}
                          {isImpersonating && (
                            <ShadcnButton
                              size="sm"
                              variant="destructive"
                              onClick={stopImpersonation}
                              className="inline-flex items-center gap-2"
                              title="Stop Impersonation"
                            >
                              <LogOut className="h-4 w-4" />
                              Stop
                            </ShadcnButton>
                          )}
                        </div>
                      </TableCell>
                    )}

                    {isVisible('user_stock') && (
                      <TableCell className="py-3">
                        <TooltipProvider delayDuration={150}>
                          <div className="flex items-center gap-2">
                            {/* View Stock Cabinet */}
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <a
                                  href={route('nurse-medication-stocks.index', user.id)}
                                  className="cursor-pointer"
                                  aria-label={`View ${user.name}'s Medication Stock`}
                                >
                                  <ShadcnButton size="sm" variant="secondary" className="p-2">
                                    <Pill className="h-4 w-4" />
                                  </ShadcnButton>
                                </a>
                              </TooltipTrigger>
                              <TooltipContent side="top">
                                <span>View Stock Cabinet</span>
                              </TooltipContent>
                            </Tooltip>

                            {/* Clinic Access */}
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <a
                                  href={route('users.clinic-access.edit', user.id)}
                                  className="cursor-pointer"
                                  aria-label={`Manage ${user.name}'s Clinic Access`}
                                >
                                  <ShadcnButton size="sm" variant="outline" className="p-2">
                                    <Building2 className="h-4 w-4" />
                                  </ShadcnButton>
                                </a>
                              </TooltipTrigger>
                              <TooltipContent side="top">
                                <span>Clinic Access</span>
                              </TooltipContent>
                            </Tooltip>

                            {/* User File Access */}
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <a
                                  href={route('users.company-access.edit', user.id)}
                                  className="cursor-pointer"
                                  aria-label={`Manage ${user.name}'s File Access`}
                                >
                                  <ShadcnButton size="sm" variant="secondary" className="p-2">
                                    <FolderCog className="h-4 w-4" />
                                  </ShadcnButton>
                                </a>
                              </TooltipTrigger>
                              <TooltipContent side="top">
                                <span>User File Access</span>
                              </TooltipContent>
                            </Tooltip>

                            {/* Module Access */}
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <a
                                  href={route('users.module-access.edit', user.id)}
                                  className="cursor-pointer"
                                  aria-label={`Manage ${user.name}'s Module Access`}
                                >
                                  <ShadcnButton size="sm" variant="outline" className="p-2">
                                    <Puzzle className="h-4 w-4" />
                                  </ShadcnButton>
                                </a>
                              </TooltipTrigger>
                              <TooltipContent side="top">
                                <span>Module Access</span>
                              </TooltipContent>
                            </Tooltip>
                          </div>
                        </TooltipProvider>
                      </TableCell>
                    )}
                  </TableRow>
                );
              })}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={users.last_page}
            forcePage={users.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
            nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>

      {/* Change Password Modal (AntD) */}
      <Modal
        title={selectedUser ? `Change Password — ${selectedUser.name}` : 'Change Password'}
        open={isModalOpen}
        onOk={handleOk}
        onCancel={handleCancel}
        footer={[
          <AntdButton key="back" onClick={handleCancel}>
            Cancel
          </AntdButton>,
          <AntdButton key="submit" type="primary" onClick={handleOk}>
            Submit
          </AntdButton>,
        ]}
      >
        <Card>
          <CardHeader>
            <CardTitle>Change Password</CardTitle>
          </CardHeader>
          <CardContent>
            <Form form={form} name="change_password_form" onFinish={handleOk} layout="vertical">
              <Form.Item name="email" label="Email">
                <ShadcnInput disabled className="text-gray-900 dark:text-gray-100" />
              </Form.Item>

              <Form.Item name="password" label="New Password" rules={[{ required: true, message: 'Please enter your new password!' }]}>
                <div className="relative">
                  <ShadcnInput type={showPassword ? 'text' : 'password'} className="text-gray-900 dark:text-gray-100" placeholder="New Password" />
                  <button
                    type="button"
                    onClick={togglePasswordVisibility}
                    className="absolute inset-y-0 right-0 flex items-center pr-3 text-xs underline"
                  >
                    {showPassword ? 'Hide' : 'Show'}
                  </button>
                </div>
              </Form.Item>

              <Form.Item
                name="confirmPassword"
                label="Confirm Password"
                dependencies={['password']}
                rules={[
                  { required: true, message: 'Please confirm your password!' },
                  ({ getFieldValue }) => ({
                    validator(_, value) {
                      if (!value || getFieldValue('password') === value) return Promise.resolve();
                      return Promise.reject(new Error('The two passwords that you entered do not match!'));
                    },
                  }),
                ]}
              >
                <div className="relative">
                  <ShadcnInput type={showPassword ? 'text' : 'password'} className="text-gray-900 dark:text-gray-100" placeholder="Confirm Password" />
                  <button
                    type="button"
                    onClick={togglePasswordVisibility}
                    className="absolute inset-y-0 right-0 flex items-center pr-3 text-xs underline"
                  >
                    {showPassword ? 'Hide' : 'Show'}
                  </button>
                </div>
              </Form.Item>
            </Form>
          </CardContent>
        </Card>
      </Modal>
    </AppLayout>
  );
}
