// resources/js/Pages/Users/ModuleAccessEdit.tsx
import * as React from 'react';
import { Head, Link, useForm } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { Input } from '@/components/ui/input';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import {
  CheckCircle2,
  ShieldAlert,
  User as UserIcon,
  ArrowLeft,
  Save,
  Filter,
  CheckSquare,
  Square,
} from 'lucide-react';
import { cn } from '@/lib/utils';
import AppLayout from '@/layouts/app-layout';
import Swal from 'sweetalert2';

type Module = { id: number; slug: string; name: string };

type PageProps = {
  user: { id: number; name: string; email: string; role?: string | null };
  modules: Module[];
  enabled_map: Record<number, boolean>;
  readonly?: boolean; // superadmin → true
};

export default function ModuleAccessEdit(props: PageProps) {
  const { user, modules, enabled_map, readonly = false } = props;

  const initiallyEnabledIds = React.useMemo(
    () =>
      Object.entries(enabled_map)
        .filter(([, enabled]) => !!enabled)
        .map(([id]) => Number(id)),
    [enabled_map]
  );

  // Single source of truth: Inertia form data
  const { data, setData, put, processing, reset, errors } = useForm<{ module_ids: number[] }>({
    module_ids: initiallyEnabledIds,
  });

  const selected = data.module_ids; // convenience alias
  const [query, setQuery] = React.useState('');

  const onSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (readonly) return;

    put(route('users.module-access.update', user.id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({
          icon: 'success',
          title: 'Saved',
          text: 'Module access updated.',
          confirmButtonColor: '#2563eb',
        });
      },
      onError: (formErrors) => {
        // If there is a known validation error, show the first one; otherwise show generic
        const firstError =
          (formErrors && Object.values(formErrors)[0]) ||
          'Something went wrong while saving. Please try again.';
        Swal.fire({
          icon: 'error',
          title: 'Error',
          text: String(firstError),
          confirmButtonColor: '#ef4444',
        });
      },
    });
  };

  const toggle = (id: number, next: boolean) => {
    setData(
      'module_ids',
      next ? Array.from(new Set([...selected, id])) : selected.filter((x) => x !== id)
    );
  };

  const toggleAll = (next: boolean) => {
    setData('module_ids', next ? modules.map((m) => m.id) : []);
  };

  const filtered = React.useMemo(() => {
    const q = query.trim().toLowerCase();
    if (!q) return modules;
    return modules.filter((m) => m.name.toLowerCase().includes(q) || m.slug.toLowerCase().includes(q));
  }, [modules, query]);

  const allChecked = selected.length === modules.length && modules.length > 0;
  const noneChecked = selected.length === 0;

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Users', href: route('users.index') },
        { title: user.name, href: route('users.show', user.id) },
        { title: 'Module Access', href: '#' },
      ]}
    >
      <Head title={`Module Access · ${user.name}`} />

      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 flex items-center justify-between">
          <div className="flex items-center gap-3">
            <Link
              href={route('users.index')}
              className="inline-flex items-center gap-2 text-sm text-muted-foreground hover:text-foreground"
            >
              <ArrowLeft className="h-4 w-4" />
              Back to Users
            </Link>
          </div>
        </div>

        <Card className="shadow-sm">
          <CardHeader>
            <CardTitle className="flex flex-col gap-2">
              <span className="text-xl">Module Access</span>
              <div className="flex flex-wrap items-center gap-3 text-sm text-muted-foreground">
                <span className="inline-flex items-center gap-2">
                  <UserIcon className="h-4 w-4" />
                  <span className="font-medium text-foreground">{user.name}</span>
                </span>
                <Separator orientation="vertical" className="mx-1 h-4" />
                <span>{user.email}</span>
                {user.role && (
                  <>
                    <Separator orientation="vertical" className="mx-1 h-4" />
                    <Badge variant="secondary" className="uppercase">
                      {user.role}
                    </Badge>
                  </>
                )}
              </div>
            </CardTitle>
          </CardHeader>

          <Separator />

          <CardContent className="space-y-6 pt-6">
            {readonly && (
              <Alert variant="default" className="border-amber-300 bg-amber-50 text-amber-900">
                <ShieldAlert className="h-4 w-4" />
                <AlertTitle>Superadmin</AlertTitle>
                <AlertDescription>
                  Superadmin has access to all modules by default. Editing is disabled.
                </AlertDescription>
              </Alert>
            )}

            {!!errors?.module_ids && (
              <Alert variant="destructive">
                <AlertTitle>Validation Error</AlertTitle>
                <AlertDescription>{errors.module_ids}</AlertDescription>
              </Alert>
            )}

            <div className="flex flex-col gap-3 md:flex-row md:items-center md:justify-between">
              <div className="flex items-center gap-2">
                <Filter className="h-4 w-4 text-muted-foreground" />
                <Input
                  value={query}
                  onChange={(e) => setQuery(e.target.value)}
                  placeholder="Search modules…"
                  className="w-[260px]"
                  disabled={readonly}
                />
              </div>
              <div className="flex items-center gap-2">
                <Button
                  type="button"
                  variant="outline"
                  size="sm"
                  onClick={() => toggleAll(true)}
                  disabled={readonly || allChecked}
                  className="inline-flex items-center gap-2"
                  style={{ cursor: readonly || allChecked ? 'not-allowed' : 'pointer' }}
                >
                  <CheckSquare className="h-4 w-4" />
                  Select all
                </Button>
                <Button
                  type="button"
                  variant="outline"
                  size="sm"
                  onClick={() => toggleAll(false)}
                  disabled={readonly || noneChecked}
                  className="inline-flex items-center gap-2"
                  style={{ cursor: readonly || noneChecked ? 'not-allowed' : 'pointer' }}
                >
                  <Square className="h-4 w-4" />
                  Clear all
                </Button>
              </div>
            </div>

            <div className="rounded-md border">
              <div className="flex items-center justify-between border-b px-4 py-3 text-sm">
                <div className="text-muted-foreground">
                  {selected.length} of {modules.length} enabled
                </div>
                {allChecked && (
                  <div className="inline-flex items-center gap-1 text-emerald-700">
                    <CheckCircle2 className="h-4 w-4" />
                    <span className="text-xs">All modules selected</span>
                  </div>
                )}
              </div>

              <ScrollArea className="h-[360px]">
                <div className="divide-y">
                  {filtered.map((m) => {
                    const checked = selected.includes(m.id);
                    return (
                      <label
                        key={m.id}
                        className={cn(
                          'flex cursor-pointer items-center justify-between gap-4 px-4 py-3 transition-colors hover:bg-muted/50',
                          readonly && 'cursor-not-allowed opacity-60'
                        )}
                      >
                        <div className="min-w-0">
                          <div className="truncate font-medium">{m.name}</div>
                          <div className="text-xs text-muted-foreground">{m.slug}</div>
                        </div>

                        <div className="flex items-center gap-3">
                          <Badge variant={checked ? 'default' : 'secondary'}>
                            {checked ? 'Enabled' : 'Disabled'}
                          </Badge>
                          <Checkbox
                            checked={checked}
                            onCheckedChange={(v) => toggle(m.id, Boolean(v))}
                            disabled={readonly}
                          />
                        </div>
                      </label>
                    );
                  })}

                  {filtered.length === 0 && (
                    <div className="px-4 py-10 text-center text-sm text-muted-foreground">
                      No modules match “{query}”.
                    </div>
                  )}
                </div>
              </ScrollArea>
            </div>
          </CardContent>

          <CardFooter className="flex items-center justify-between">
            <Button asChild variant="ghost">
              <Link href={route('users.index')}>
                <ArrowLeft className="mr-2 h-4 w-4" />
                Back
              </Link>
            </Button>

            <form onSubmit={onSubmit} className="inline-flex items-center gap-2">
              <Button
                type="submit"
                disabled={readonly || processing}
                className="inline-flex items-center gap-2"
                style={{ cursor: readonly ? 'not-allowed' : 'pointer' }}
              >
                <Save className="h-4 w-4" />
                {processing ? 'Saving…' : 'Save changes'}
              </Button>
              {!readonly && (
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => {
                    setQuery('');
                    setData('module_ids', initiallyEnabledIds);
                    reset();
                  }}
                >
                  Reset
                </Button>
              )}
            </form>
          </CardFooter>
        </Card>
      </div>
    </AppLayout>
  );
}
