import { Head, useForm } from '@inertiajs/react';
import { Eye, EyeOff, LoaderCircle, Settings } from 'lucide-react';
import { FormEventHandler, useState, useEffect } from 'react';

import InputError from '@/components/input-error';
import TextLink from '@/components/text-link';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AuthLayout from '@/layouts/auth-layout';
import { getAPI, setAPI } from '@/config'; // Import your config functions

type LoginForm = {
    email: string;
    password: string;
    remember: boolean;
};

interface LoginProps {
    status?: string;
    canResetPassword: boolean;
}

export default function Login({ status, canResetPassword }: LoginProps) {
    const { data, setData, post, processing, errors, reset } = useForm<Required<LoginForm>>({
        email: '',
        password: '',
        remember: false,
    });

    const [showPassword, setShowPassword] = useState(false);
    const [showApiSettings, setShowApiSettings] = useState(false);
    const [apiUrl, setApiUrl] = useState(getAPI());
    const [apiError, setApiError] = useState('');

    useEffect(() => {
        // Update the API URL when it changes
        setApiUrl(getAPI());
    }, []);

    const submit: FormEventHandler = (e) => {
        e.preventDefault();
        post(route('login'), {
            onFinish: () => reset('password'),
        });
    };

    const handleApiUrlChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        setApiUrl(e.target.value);
        setApiError('');
    };

    const saveApiUrl = () => {
        if (!apiUrl.trim()) {
            setApiError('API URL is required');
            return;
        }

        try {
            // Basic validation - check if it's a valid URL or path
            if (apiUrl.startsWith('http')) {
                new URL(apiUrl); // This will throw if invalid
            }

            setAPI(apiUrl);
            setShowApiSettings(false);
            // Reload the page to apply the new API URL
            window.location.reload();
        } catch (error) {
            setApiError('Please enter a valid URL (e.g., http://example.com/api or /public)');
        }
    };

    return (
        <AuthLayout title="Log in to your account" description="Enter your email and password below to log in">
            <Head title="Log in" />

            {/* API Settings Toggle */}
            <div className="mb-4 flex justify-end">
                <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={() => setShowApiSettings(!showApiSettings)}
                    className="flex items-center gap-2"
                    style={{
                        cursor: 'pointer',
                    }}
                >
                    <Settings className="h-4 w-4" />
                    API Settings
                </Button>
            </div>

            {/* API URL Input */}
            {showApiSettings && (
                <div className="mb-6 rounded-lg border p-4">
                    <h3 className="mb-3 text-lg font-semibold">API Configuration</h3>
                    <div className="grid gap-2">
                        <Label htmlFor="apiUrl">API Base URL</Label>
                        <Input
                            id="apiUrl"
                            type="text"
                            value={apiUrl}
                            onChange={handleApiUrlChange}
                            placeholder="Enter API URL (e.g., http://192.168.10.245:8080/health/public/ or /public)"
                            className="font-mono text-sm"
                        />
                        {apiError && <p className="text-sm text-red-600">{apiError}</p>}
                        <p className="text-muted-foreground text-xs">
                            Enter the full URL (http://...) or a relative path (/public)
                        </p>
                    </div>
                    <div className="mt-3 flex gap-2">
                        <Button onClick={saveApiUrl} size="sm" style={{
                            cursor: 'pointer',
                        }}>
                            Save
                        </Button>
                        <Button
                            onClick={() => setShowApiSettings(false)}
                            variant="outline"
                            size="sm"
                            style={{
                                cursor: 'pointer',
                            }}
                        >
                            Cancel
                        </Button>
                    </div>
                </div>
            )}

            <form className="flex flex-col gap-6" onSubmit={submit}>
                <div className="grid gap-6">
                    {/* Email */}
                    <div className="grid gap-2">
                        <Label htmlFor="email">Email address</Label>
                        <Input
                            id="email"
                            type="email"
                            required
                            autoFocus
                            tabIndex={1}
                            autoComplete="email"
                            value={data.email}
                            onChange={(e) => setData('email', e.target.value)}
                            placeholder="name@providencehealth.co.zw"
                        />
                        <InputError message={errors.email} />
                    </div>

                    {/* Password */}
                    <div className="grid gap-2">
                        <div className="flex items-center">
                            <Label htmlFor="password">Password</Label>
                            {canResetPassword && (
                                <TextLink href={route('password.request')} className="ml-auto text-sm" tabIndex={5}>
                                    Forgot password?
                                </TextLink>
                            )}
                        </div>
                        <div className="relative">
                            <Input
                                id="password"
                                type={showPassword ? 'text' : 'password'}
                                required
                                tabIndex={2}
                                autoComplete="current-password"
                                value={data.password}
                                onChange={(e) => setData('password', e.target.value)}
                                placeholder="Password"
                                className="pr-10"
                            />
                            <button
                                type="button"
                                onClick={() => setShowPassword(!showPassword)}
                                className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700"
                            >
                                {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                            </button>
                        </div>
                        <InputError message={errors.password} />
                    </div>

                    {/* Remember me */}
                    <div className="flex items-center space-x-3">
                        <Checkbox
                            id="remember"
                            name="remember"
                            checked={data.remember}
                            onClick={() => setData('remember', !data.remember)}
                            tabIndex={3}
                        />
                        <Label htmlFor="remember">Remember me</Label>
                    </div>

                    {/* Submit button */}
                    <Button type="submit" className="mt-4 w-full" tabIndex={4} disabled={processing}>
                        {processing && <LoaderCircle className="h-4 w-4 animate-spin" />}
                        Log in
                    </Button>
                </div>

                {/* Signup link */}
                <div className="text-muted-foreground text-center text-sm">
                    Don't have an account?{' '}
                    <TextLink href={route('register')} tabIndex={5}>
                        Sign up
                    </TextLink>
                </div>
            </form>

            {/* Status message */}
            {status && <div className="mb-4 text-center text-sm font-medium text-green-600">{status}</div>}

            {/* Current API URL indicator */}
            <div className="mt-6 text-center">
                <p className="text-muted-foreground text-xs">
                    Current API: <code className="rounded bg-gray-100 px-1 py-0.5">{getAPI()}</code>
                </p>
            </div>
        </AuthLayout>
    );
}