import { AgeRangeChart } from '@/components/ui/age-range-chart';
import { ConsultationChart } from '@/components/ui/consultation-chart';
import { EmployeeDependentsChart } from '@/components/ui/employee-dependents-chart';
import { GenderDistributionChart } from '@/components/ui/gender-distribution-chart';
import InjuryOnDutyChart from '@/components/ui/injury-on-duty-chart';
import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';
import { ReferralChart } from '@/components/ui/referral-chart';
import { SickLeaveChart } from '@/components/ui/sick-leave-chart';
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { type BreadcrumbItem, type SharedData } from '@/types';
import { Head, usePage } from '@inertiajs/react';
import axios from 'axios';
import { useEffect, useState } from 'react';
import SearchPage from './Patients/SearchPage';
import SearchComponent from './Patients/components/SearchComponent';
import ConsultorTriages from '@/components/ConsultorTriages';
import {  SectionCards } from '@/components/PatientStats';

interface Company {
    id: number;
    name: string;
}

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Dashboard',
        href: `${API}/dashboard`,
    },
];

export default function Dashboard() {
    const page = usePage<SharedData>();
    const { auth, companies = [] } = page.props; // Provide a default value for companies

    const [stats, setStats] = useState<{ employees: number; dependents: number; total: number } | null>(null);

    const [year, setYear] = useState<number>(new Date().getFullYear());
    const [companyId, setCompanyId] = useState<number | null>(null);

    useEffect(() => {
        axios.get(`${API}/patients/stats`).then((res) => {
            setStats(res.data);
        });
    }, [year, companyId]);

    const renderRoleSpecificContent = () => {
        const { role } = auth.user;

        switch (role) {
            case 'superadmin':
                return (
                    <>
                        <div className="grid grid-cols-6 gap-3">
                            <SectionCards stats={stats} loading={!stats} />
                            <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                                <ConsultationChart />
                            </div>

                            <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                                <div className="grid grid-cols-1 gap-4 md:grid-cols-3">
                                    {/* Gender Distribution */}
                                    <GenderDistributionChart />
                                    <EmployeeDependentsChart />
                                    <AgeRangeChart />
                                </div>
                            </div>
                            <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                                <div className="grid grid-cols-1 gap-6 md:grid-cols-6">
                                    {/* Sick Leave Issued Per Month */}
                                    <div className="md:col-span-3">
                                        <SickLeaveChart />
                                    </div>
                                    <div className="md:col-span-3">
                                        <ReferralChart />
                                    </div>
                                </div>
                            </div>

                            <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                                <div className="grid grid-cols-1 gap-6 md:grid-cols-6">
                                    <InjuryOnDutyChart />
                                </div>
                            </div>


                        </div>
                        <ConsultorTriages />
                    </>

                );

            case 'doctor':
                return (
                    <div className="grid grid-cols-6 gap-3">
                        <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-2 text-lg font-semibold text-neutral-700 dark:text-neutral-100">Open Consultation Files</h2>
                            {/* Doctor-specific consultations chart or list */}
                            <ConsultorTriages />
                        </div>
                    </div>
                );

            case 'nurse':
                return (
                    <div className="grid grid-cols-6 gap-3">
                        <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-2 text-lg font-semibold text-neutral-700 dark:text-neutral-100">Your Tasks</h2>
                            {/* Nurse-specific tasks or list */}
                            <ConsultorTriages />
                        </div>

                    </div>
                );

            case 'pharmacist':
                return (
                    <div className="grid grid-cols-6 gap-3">
                        <div className="col-span-6 rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-2 text-lg font-semibold text-neutral-700 dark:text-neutral-100">Medication Stocks</h2>
                            {/* Pharmacist-specific medication stocks */}
                            <PlaceholderPattern />
                        </div>
                    </div>
                );

            case 'receptionist':
                return (
                    <SearchComponent />
                );

            case 'user':
                return (
                    <div className="flex h-full items-center justify-center rounded-xl border">
                        <div className="text-center">
                            <h2 className="text-xl font-medium">Welcome to your dashboard</h2>
                            <p className="text-neutral-600 dark:text-neutral-400">
                                Your role doesn't have specific dashboard metrics configured yet.
                            </p>
                        </div>
                    </div>
                );

            default:
                return (
                    <div className="flex h-full items-center justify-center rounded-xl border">
                        <div className="text-center">
                            <h2 className="text-xl font-medium">Welcome to your dashboard</h2>
                            <p className="text-neutral-600 dark:text-neutral-400">
                                Your role doesn't have specific dashboard metrics configured yet.
                            </p>
                        </div>
                    </div>
                );
        }
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Dashboard" />
            <div className="flex h-full flex-1 flex-col gap-4 rounded-xl p-4">{renderRoleSpecificContent()}</div>
        </AppLayout>
    );
}
